import os

from pxr import Usd, UsdGeom


def get_stage(filename, asset_name, create=False):
    if os.path.exists(filename):
        print('The file do exists, opening')
        stage = Usd.Stage.Open(filename)

    else:
        abs_path = os.path.abspath(filename)
        dirname = os.path.dirname(abs_path)
        if not os.path.exists(dirname):
            os.makedirs(dirname)

        print('The file do not exists, creating')
        stage = Usd.Stage.CreateNew(filename)
        asset_prim = stage.OverridePrim('/%s' % asset_name)
        stage.SetDefaultPrim(asset_prim)
        if create:
            stage.GetRootLayer().Save()

    return stage


def add_file_variant(asset_name, layer_type, variant_name, path, variant_value, select_variant='', name=''):
    print('updating file: %s %s %s ' % (asset_name, layer_type, variant_name))
    if name:
        filename = './%s/%s/%s_%s_%s.usda' % (asset_name, layer_type, layer_type, name, variant_name)
    else:
        filename = './%s/%s/%s_%s.usda' % (asset_name, layer_type, layer_type, variant_name)
    variant_set_name = '%s_%s' % (layer_type.title(), variant_name)

    stage = get_stage(filename, asset_name)

    asset_prim = stage.OverridePrim('/%s' % asset_name)
    stage.SetDefaultPrim(asset_prim)



    variant_sets = asset_prim.GetVariantSets()
    v_sets_names = variant_sets.GetNames()

    if variant_set_name in v_sets_names:
        variant_set = variant_sets.GetVariantSet(variant_set_name)

    else:
        variant_set = variant_sets.AddVariantSet(variant_set_name)

    variant_name_list = variant_set.GetVariantNames()
    variant_name = variant_name.title()

    print(variant_name_list)
    if variant_name in variant_name_list:
        print('variant already in the layer')
        return

    variant_set.AddVariant(variant_value)

    variant_set.SetVariantSelection(variant_value)

    with variant_set.GetVariantEditContext():
        payloads = asset_prim.GetPayloads()
        payloads.AddPayload(path)

    if select_variant:
        print('set default variant to previous')
        variant_set.SetVariantSelection(select_variant)

    # print(stage.GetRootLayer().ExportToString())
    stage.GetRootLayer().Save()

    return filename


def build_asset_assembly(asset_name, asset_layers=None, force_update=True):
    if not asset_layers:
        asset_layers = ['procedural', 'cloth', 'cfx', 'shading', 'textures', 'uvs', 'geometry']

    assembly_path = './%s/asset_assembly.usda' % asset_name
    if not force_update:
        full_path = os.path.abspath(assembly_path)
        if os.path.exists(full_path):
            print('Asset assembly for %s already exists' % asset_name)
            return

    stage = get_stage(assembly_path, asset_name)

    asset_prim = stage.GetPrimAtPath('/%s' % asset_name)
    if not asset_prim.IsValid():
        print('redefine prime asset')
        asset_prim = stage.DefinePrim('/%s' % asset_name, 'Xform')

    render_prim = stage.OverridePrim('/%s/render' % asset_name)

    render_prim.SetTypeName("Scope")
    render_purpose = render_prim.GetAttribute('purpose')
    render_purpose.Set('render')


    proxy_prim = stage.OverridePrim('/%s/proxy' % asset_name)
    proxy_prim.SetTypeName("Scope")
    proxy_purpose = proxy_prim.GetAttribute('purpose')
    proxy_purpose.Set('proxy')

    stage.SetDefaultPrim(asset_prim)

    payloads = asset_prim.GetPayloads()
    payloads.ClearPayloads()

    for layer in asset_layers:
        path = './%s/%s_variant.usda' % (layer, layer)
        relative_path = './%s/%s/%s_variant.usda' % (asset_name, layer, layer)
        print('create layer: %s ' % relative_path)
        get_stage(relative_path, asset_name, create=True)

        payloads.AddPayload(path)

    # print(stage.GetRootLayer().ExportToString())
    stage.GetRootLayer().Save()


def add_file_to_asset(asset_name, layer_type, variant_name, path, version, update_assembly=True):
    build_asset_assembly(asset_name, force_update=update_assembly)

    version_file = add_file_variant(asset_name, layer_type, 'version', path, version, name=variant_name)

    relative_path = './%s' % version_file.split('/')[-1]

    variant_file = add_file_variant(asset_name, layer_type, 'variant', relative_path, variant_name,
                                    select_variant='Master')

if __name__ == '__main__':

    root_path = 'V:/SGD/publish/usd/assets'
    if not os.path.exists(root_path):
        os.makedirs(root_path)
    os.chdir(root_path)

    variant = 'Master'
    version = '002'

    path = '../../../WhiteHome01.usda'
    layer_type = 'geometry'
    asset = 'Room'

    # add_file_to_asset(asset, layer_type, variant, path, version)

    layer_type = 'shading'
    path = '../../../WhiteHome01_metal.usda'
    variant = 'Metal'

    add_file_to_asset(asset, layer_type, variant, path, version)

    path = '../../../WhiteHome01_laminate.usda'
    variant = 'Laminate'
    version = '002'

    # add_file_to_asset(asset, layer_type, variant, path, version)
