import maya.cmds as cmds


class UsdRig():
    def __init__(self, bifrost_node, usd_scene=None):
        self.bifrost_node_name = bifrost_node
        self.usd_scene = usd_scene
        self.characters = {}

    def add_layer(self, stage, layer_path):

        new_layer = cmds.vnnCompound(self.bifrost_node, '/', addNode='BifrostGraph,USD::Layer,open_usd_layer')[0]
        cmds.vnnConnect(self.bifrost_node, '/%s.layer' % new_layer, '/%s.sublayers.layer' % stage)
        cmds.vnnNode(self.bifrost_node, '/%s' % new_layer, setPortDefaultValues=('file', layer_path))

    def create_stage(self):
        self.bifrost_node = cmds.createNode('bifrostGraphShape', name='%sShape' % self.bifrost_node_name)

        parent_node = cmds.listRelatives(self.bifrost_node, parent=True)
        cmds.rename(parent_node[0], self.bifrost_node_name)
        stage = cmds.vnnCompound(self.bifrost_node, '/', addNode='BifrostGraph,USD::Stage,create_usd_stage')[0]

        self.add_to_stage = cmds.vnnCompound(self.bifrost_node, '/', addNode='BifrostGraph,USD::Stage,add_to_stage')[0]
        cmds.vnnConnect(self.bifrost_node, '/%s.stage' % stage, '/%s.stage' % self.add_to_stage)

        if self.usd_scene:
            cmds.vnnNode(self.bifrost_node, '/%s' % stage, createInputPort=('sublayers.layer', 'auto'))

            self.add_layer(stage, self.usd_scene)

        cmds.vnnCompound(self.bifrost_node, "/", removeNode='input')

        return self.add_to_stage

    def create_character_container(self, character_name):
        cmds.vnnCompound(self.bifrost_node, "/", create=character_name)
        cmds.vnnCompound(self.bifrost_node, '/%s' % character_name, createOutputPort=("character_prim", "auto"))

        cmds.vnnNode(self.bifrost_node, '/%s' % self.add_to_stage,
                     createInputPort=("prim_definitions.%s_prim" % character_name, "auto"))

        cmds.vnnConnect(self.bifrost_node, '/%s.character_prim' % character_name,
                        '/%s.prim_definitions.%s_prim' % (self.add_to_stage, character_name))

        input_node = cmds.vnnCompound(self.bifrost_node, '/', addIONode=True)[0]
        input_node = cmds.vnnCompound(self.bifrost_node, '/', renameNode=(input_node, '%s_input' % character_name))

        return '/%s' % character_name

    def add_children(self, container_path, maya_parent, prim_parent):
        for child in cmds.listRelatives(maya_parent, c=True, f=True):
            child_short = child.split('|')[-1]
            if cmds.nodeType(child) == 'transform':
                transform_prim = \
                cmds.vnnCompound(self.bifrost_node, container_path, addNode='BifrostGraph,USD::Prim,define_usd_prim')[0]
                cmds.vnnNode(self.bifrost_node, "%s/%s" % (container_path, transform_prim),
                             setPortDefaultValues=('path', '/%s' % child_short))

                cmds.vnnNode(self.bifrost_node, '%s/%s' % (container_path, prim_parent),
                             createInputPort=('children.%s_prim' % child_short, 'auto'))

                cmds.vnnConnect(self.bifrost_node, '%s/%s.prim_definition' % (container_path, transform_prim),
                                '%s/%s.children.%s_prim' % (container_path, prim_parent, child_short))

                self.add_children(container_path, child, transform_prim)
                cmds.vnnCompound(self.bifrost_node, container_path, renameNode=(transform_prim, child_short))

            elif cmds.nodeType(child) == 'mesh':
                if child.endswith('Orig'):
                    continue

                mesh_prim = cmds.vnnCompound(self.bifrost_node, container_path,
                                             addNode='BifrostGraph,USD::Prim,define_usd_mesh')[0]
                cmds.vnnNode(self.bifrost_node, "%s/%s" % (container_path, mesh_prim),
                             setPortDefaultValues=('path', '/%s' % child_short))
                cmds.vnnNode(self.bifrost_node, '%s/%s' % (container_path, prim_parent),
                             createInputPort=('children.%s_prim' % child_short, 'auto'))

                cmds.vnnNode(self.bifrost_node, '%s/%s' % (container_path, mesh_prim),
                             setPortDefaultValues=('use_frame', '1'))

                cmds.vnnConnect(self.bifrost_node, '%s/%s.mesh_definition' % (container_path, mesh_prim),
                                '%s/%s.children.%s_prim' % (container_path, prim_parent, child_short))

                options = 'pathinfo={path=%s;setOperation=+;active=true}' % child

                cmds.vnnNode(self.bifrost_node, "%s/input" % container_path, createOutputPort=(child_short, 'Object'),
                             portOptions=options)

                cmds.vnnConnect(self.bifrost_node, '%s.%s' % (container_path, child_short),
                                '%s/%s.mesh' % (container_path, mesh_prim))
                cmds.vnnNode(self.bifrost_node, "%s_input" % container_path, createOutputPort=(child_short, 'Object'),
                             portOptions=options)
                cmds.vnnConnect(self.bifrost_node, '.%s' % child_short, '%s.%s' % (container_path, child_short))

                cmds.vnnCompound(self.bifrost_node, container_path, renameNode=(mesh_prim, child_short))

    def add_rig(self, usd_character_root, geometry_root, character_name):
        container_path = self.create_character_container(character_name)

        character_prim = \
        cmds.vnnCompound(self.bifrost_node, container_path, addNode='BifrostGraph,USD::Prim,define_usd_prim')[0]
        cmds.vnnNode(self.bifrost_node, "%s/%s" % (container_path, character_prim),
                     setPortDefaultValues=('path', usd_character_root))
        cmds.vnnConnect(self.bifrost_node, '%s/%s.prim_definition' % (container_path, character_prim),
                        '%s/output.character_prim' % container_path)

        self.add_children(container_path, geometry_root, character_prim)

        cmds.vnnCompound(self.bifrost_node, container_path, renameNode=(character_prim, 'Geo'))


def add_mesh_to_bifrost(bifrost_node, node, stage_node):
    node = cmds.ls(node, l=True)[0]
    basename = node.split('|')[-1]
    path = node.replace('|', '/')

    input_node = cmds.vnnCompound(bifrost_node, '/', addIONode=True)[0]
    input_node = cmds.vnnCompound(bifrost_node, '/', renameNode=(input_node, basename))
    options = 'pathinfo={path=%s;setOperation=+;active=true}' % path
    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % basename,
                 createOutputPort=('mesh_%s' % basename, 'Object'), portOptions=options)
    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % basename, createOutputPort=('mesh', 'Object'),
                 portOptions=options)

    transform_prim = cmds.vnnCompound(bifrost_node, '/', addNode='BifrostGraph,USD::Prim,define_usd_prim')

    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % transform_prim[0],
                 setPortDefaultValues=('path', '/Asset/Character/%s' % basename))

    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % stage_node,
                 createInputPort=('prim_definitions.prim_definition', 'auto'))

    cmds.vnnConnect(bifrost_node, '/%s.prim_definition' % transform_prim[0],
                    '/%s.prim_definitions.prim_definition' % stage_node)

    mesh_prim = cmds.vnnCompound(bifrost_node, '/', addNode='BifrostGraph,USD::Prim,define_usd_mesh')
    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % mesh_prim[0],
                 setPortDefaultValues=('path', '%sShape' % basename))

    cmds.vnnNode("|bifrostGraph1|bifrostGraphShape1", "/%s" % transform_prim[0],
                 createInputPort=('children.mesh_definition', 'auto'))

    cmds.vnnConnect(bifrost_node, '/%s.mesh_definition' % mesh_prim[0],
                    '/%s.children.mesh_definition' % transform_prim[0])

    print(input_node)
    cmds.vnnConnect(bifrost_node, '/%s.mesh_%s' % (basename, basename), '/%s.mesh' % mesh_prim[0])


def create_stage(bifrost_node):
    print('create stage')
    stage = cmds.vnnCompound(bifrost_node, '/', addNode='BifrostGraph,USD::Stage,create_usd_stage')
    add_to_stage = cmds.vnnCompound(bifrost_node, '/', addNode='BifrostGraph,USD::Stage,add_to_stage')
    print(stage, add_to_stage)

    cmds.vnnConnect(bifrost_node, '/%s.stage' % stage[0], '/%s.stage' % add_to_stage[0])

    return add_to_stage[0]


if __name__ == '__main__':
    node = 'pCubeShape1'
    bifrost_node = 'bifrostGraphShape1'
    stage = create_stage(bifrost_node)
    root = 'gr_combined'
    nodes = cmds.listRelatives(root, ad=True, f=True)
    for node in nodes:
        if cmds.nodeType(node) != 'mesh':
            continue
        print(node)
        add_mesh_to_bifrost(bifrost_node, node, stage)