import os
import importlib

import usd.lib.usd_manager as usd_manager

import launcher.lib.set_environment as set_environment


def create_entity_assembly(manager_instance, layers_config=None):
    print('=== create_entity_assembly ===')
    print('Entity: %s' % manager_instance.entity_name)
    print('Entity type: %s' % manager_instance.entity_type)

    if layers_config is None:
        layers_config = manager_instance.get_layers_config()

    manager_instance.clear_sublayers()

    for layer_config in layers_config:
        for layer_name, layer_data in layer_config.items():
            print('Layer: %s' % layer_name)
            if layer_name == 'breakdown':
                print('Generating breakdown')
                import usd.lib.build_breakdown as build_breakdown
                importlib.reload(build_breakdown)
                builder = build_breakdown.BreakdownBuilder(manager_instance.entity_name,
                                                           project=manager_instance.project)
                breakdown_path = builder.build_breakdown()
                manager_instance.add_sublayer(breakdown_path)
                continue

            new_layer_file = usd_manager.UsdManager(manager_instance.project)
            new_layer_file.set_entity(manager_instance.entity_name,
                                      manager_instance.entity_type,
                                      asset_type=manager_instance.asset_type)

            layer_path = get_layer_filename(new_layer_file, layer_name)

            if os.path.exists(layer_path):
                os.remove(layer_path)

            new_layer_file.open(layer_path)

            if isinstance(layer_data, list):
                create_entity_assembly(new_layer_file, layers_config=layer_data)
            new_layer_file.save_stage()
            manager_instance.add_sublayer(layer_path)

    print('adding prims', manager_instance.entity_type)
    if manager_instance.entity_type == 'Asset':
        asset_prim = manager_instance.stage.DefinePrim('/asset', 'Xform')
        manager_instance.default_prim = '/asset'
        if manager_instance.asset_type != 'Sets' or self.asset_type != 'Collections':
            render_prim = manager_instance.stage.DefinePrim('/asset/render', 'Xform')
            proxy_prim = manager_instance.stage.DefinePrim('/asset/proxy', 'Xform')

            manager_instance.set_prim_purpose('/asset/render', 'render')
            manager_instance.set_prim_purpose('/asset/proxy', 'proxy')

    else:
        world_prim = manager_instance.stage.DefinePrim('/World', 'Xform')
        manager_instance.default_prim = '/World'

    manager_instance.save_stage()


def add_asset_version(manager_instance, step, variant, version, version_path, add_empty_layer=False):

    print('=== add_asset_version ===')
    print('Entity: %s' % manager_instance.entity_name)
    print('Entity type: %s' % manager_instance.entity_type)
    print('Pipeline step: %s' % step)
    print('Variant: %s' % variant)
    print('Version: %s' % version)

    manager_instance.check_assembly_exists()

    layers, _ = manager_instance.check_layer_in_assembly(step)

    if not layers:
        print('Can\'t find layers definition for the pipeline step %s' % step)
        return

    layer_name = layers[-1]

    variant_filename = get_layer_filename(manager_instance, layer_name)
    version_filename = get_layer_filename(manager_instance, layer_name, variant=variant)

    version_manager = usd_manager.UsdManager(manager_instance.project)
    version_manager.set_entity(manager_instance.entity_name,
                               manager_instance.entity_type,
                               asset_type=manager_instance.asset_type)

    version_manager.open(version_filename)

    if isinstance(version, int):
        version_str = '%03d' % version
    else:
        version_str = version

    versions = version_manager.add_variant('%s_version' % layer_name, version_str, version_path)
    numeric_versions = [int(ver) for ver in versions if ver.isdigit()]

    if numeric_versions and version >= max(numeric_versions):
        version_manager.add_variant('%s_version' % layer_name, 'latest', version_path)

        if 'approved' not in versions:
            version_manager.add_variant('%s_version' % layer_name, 'recommended', version_path)
            version_manager.set_selected_variant(version_manager.default_prim, '%s_version' % layer_name, 'recommended')
        else:

            version_manager.set_selected_variant(version_manager.default_prim, '%s_version' % layer_name, 'approved')

    print('Saving version layer: %s' % version_manager.filename)
    version_manager.save_stage()

    variant_manager = usd_manager.UsdManager(manager_instance.project)
    variant_manager.set_entity(manager_instance.entity_name,
                               manager_instance.entity_type,
                               asset_type=manager_instance.asset_type)

    variant_manager.open(variant_filename)
    if add_empty_layer:
        variant_manager.add_variant('%s_variant' % layer_name, 'None', None)

    variant_manager.add_variant('%s_variant' % layer_name, variant, version_filename)
    variant_manager.set_selected_variant(variant_manager.default_prim, '%s_variant' % layer_name, 'Master')

    print('Saving variant layer: %s' % variant_manager.filename)
    variant_manager.save_stage()

    manager_instance.check_layer_in_assembly(step)


def get_assembly_root(manager_instance):
    root = manager_instance.project_config['paths']['usd_files']
    if manager_instance.entity_type == 'Asset':
        asset_type = manager_instance.asset_type.replace(' ', '_')
        print('root', root)
        print('asset_type', asset_type)
        print('root', manager_instance.entity_name)
        assembly_folder = '%s/assets/%s/%s' % (root, asset_type, manager_instance.entity_name)
    else:
        bits = manager_instance.entity_name.split('_')
        season = bits[0]
        episode = '_'.join(bits[:2])
        sequence = '_'.join(bits[:3])
        assembly_folder = '%s/shots/%s/%s/%s/%s' % (root, season, episode, sequence,  manager_instance.entity_name)

    return assembly_folder


def get_assembly_filename(manager_instance):

    if manager_instance.entity_type == 'Asset':
        filename = '%s/asset_assembly.usda' % manager_instance.assembly_folder
    else:
        filename = '%s/shot_assembly.usda' % manager_instance.assembly_folder

    return filename


def get_layer_filename(manager_instance, step, variant=None):
    print(step, manager_instance.entity_name, variant)
    print('assembly folder', manager_instance.assembly_folder)
    if variant is None:
        layer_name = '%s/%s/%s_%s.usda' % (manager_instance.assembly_folder, step, manager_instance.entity_name, step)
    else:
        layer_name = '%s/%s/%s_%s_%s.usda' % (manager_instance.assembly_folder, step, manager_instance.entity_name, step, variant)

    return layer_name


def exists_root_layer(manager_instance):
    path = get_assembly_filename(manager_instance)
    return os.path.exists(path)


def approve_layer_version(manager_instance, step, variant, version):
    print('=== approve_layer_version ===')
    print('Entity: %s' % manager_instance.entity_name)
    print('Entity type: %s' % manager_instance.entity_type)
    print('Variant: %s' % variant)
    print('Pipeline step: %s' % step)

    layers, _ = manager_instance.check_layer_in_assembly(step)

    if not layers:
        print('cant find layers')
        return
    layer_name = layers[-1]

    version_filename = get_layer_filename(manager_instance, step, variant=variant)

    version_manager = usd_manager.UsdManager(manager_instance.project)
    version_manager.open(version_filename)
    variant_set_name = '%s_version' % layer_name
    variant_sets = version_manager.default_prim.GetVariantSets()
    version_set = variant_sets.AddVariantSet(variant_set_name)
    version_set.SetVariantSelection('%03d' % version)

    with version_set.GetVariantEditContext():
        prim_metadata  = version_manager.default_prim.GetMetadata("payload")
        if not prim_metadata:
            return
        version_path = prim_metadata.ApplyOperations([])[0].assetPath

    version_manager.add_variant('%s_version' % layer_name, 'approved', version_path)
    version_manager.add_variant('%s_version' % layer_name, 'recommended', version_path)

    version_set.SetVariantSelection('recommended')
    version_manager.save_stage()


def get_variant_versions(manager_instance, layer_name, variant):
    if not manager_instance.default_prim.IsValid():
        print('not valid prim')
        return {}

    all_variant_sets = manager_instance.default_prim.GetVariantSets()
    variant_set_name = '%s_variant' % layer_name
    version_set_name = '%s_version' % layer_name

    variant_set = all_variant_sets.GetVariantSet(variant_set_name)
    variant_set.SetVariantSelection(variant)

    with variant_set.GetVariantEditContext():
        version_set = all_variant_sets.GetVariantSet(version_set_name)
        return version_set.GetVariantNames()

