
from pxr import Sdf, Usd, UsdShade

LIST_ATTRS = ['addedItems', 'appendedItems', 'deletedItems', 'explicitItems',
              'orderedItems', 'prependedItems']


def repath_properties(layer, old_path, new_path):
    """Re-path property relationship targets and attribute connections.
    This will replace any relationship or connections from old path
    to new path by replacing start of any path that matches the new path.
    Args:
        layer (Sdf.Layer): Layer to move prim spec path.
        old_path (Union[Sdf.Path, str]): Source path to move from.
        new_path (Union[Sdf.Path, str]): Destination path to move to.
    Returns:
        bool: Whether any re-pathing occurred for the given paths.
    """

    old_path_str = str(old_path)
    peformed_repath = False

    def replace_in_list(spec_list):
        """Replace paths in SdfTargetProxy or SdfConnectionsProxy"""
        for attr in LIST_ATTRS:
            entries = getattr(spec_list, attr)
            for i, entry in enumerate(entries):
                entry_str = str(entry)
                if entry == old_path or entry_str.startswith(
                        old_path_str + "/"):
                    # Repath
                    entries[i] = Sdf.Path(
                        str(new_path) + entry_str[len(old_path_str):])
                    peformed_repath = True

    def repath(path):
        spec = layer.GetObjectAtPath(path)
        if isinstance(spec, Sdf.RelationshipSpec):
            replace_in_list(spec.targetPathList)
        if isinstance(spec, Sdf.AttributeSpec):
            replace_in_list(spec.connectionPathList)

    # Repath any relationship pointing to this src prim path
    layer.Traverse("/", repath)

    return peformed_repath


def reparent_prim(layer, src_prim_path, dest_prim_path):
    """Move a PrimSpec and repath connections.
    Note that the parent path of the destination must
    exist, otherwise the namespace edit to that path
    will fail.
    Args:
        layer (Sdf.Layer): Layer to move prim spec path.
        src_prim_path (Union[Sdf.Path, str]): Source path to move from.
        dest_prim_path (Union[Sdf.Path, str]): Destination path to move to.
    Returns:
        bool: Whether the move was successful
    """

    src_prim_path = Sdf.Path(src_prim_path)
    dest_prim_path = Sdf.Path(dest_prim_path)
    dest_parent = dest_prim_path.GetParentPath()
    dest_name = dest_prim_path.name
    layer.GetPrimAtPath(dest_prim_path)

    with Sdf.ChangeBlock():
        reparent_edit = Sdf.NamespaceEdit.ReparentAndRename(
            src_prim_path,
            dest_parent,
            dest_name,
            -1
        )

        edit = Sdf.BatchNamespaceEdit()
        print(edit)
        edit.Add(reparent_edit)
        print('Applied', layer.Apply(edit))
        print('src_prim_path', src_prim_path)
        print('GetPrimAtPath', layer.GetPrimAtPath(src_prim_path))
        if not layer.Apply(edit) and layer.GetPrimAtPath(src_prim_path):
            print("Failed prim spec move")
            return False

        repath_properties(layer, src_prim_path, dest_prim_path)

    return True


# Example usage in Maya to move /mtl to /asset/mtl for all
# maya usd proxy shapes in the scene in their current edit target layer

def apply_material(stage, mesh_path, material_path):
    mesh_prim = stage.GetPrimAtPath(mesh_path)
    material_prim = stage.GetPrimAtPath(material_path)

    mesh_prim.ApplyAPI(UsdShade.MaterialBindingAPI)
    material = UsdShade.Material(material_prim)

    UsdShade.MaterialBindingAPI(mesh_prim).Bind(material)


if __name__ == '__main__':
    source_path = 'D:/Documentos/maya/projects/default/scenes/shader_tests.usda'
    stage = Usd.Stage.Open(source_path)
    layer = stage.GetRootLayer()

    # Reparent and repath connections and relationships

    # move_prim_spec(layer, "/mtl", "/pCube1/mtl")
    apply_material(stage, '/pCube1/pCubeShape1', '/pCube1/mtl/aiStandardSurface1')

    print(layer.ExportToString())
    # Force viewport reset because sometimes viewport doesn't recognize
    # the shader moved
