import os
import shutil
import importlib
import glob


from pprint import pprint, pformat

import maya.cmds as cmds

import shotgrid_lib.database as database
import shotgrid_lib.lib.publish_helpers as publish_helpers

import launcher.lib.set_environment as set_environment

#importlib.reload(shotgrid_helpers)
#importlib.reload(database)
importlib.reload(publish_helpers)


texture_node_types = {'file': 'fileTextureName', 'aiImage': 'filename'}

channel_codes = ['MSK', 'DFC', 'MTL', 'NRM', 'RLR', 'HGT', 'IBL', 'EMC']


def get_udim(path):
    udim = None
    basename = os.path.basename(path)
    basename, ext = basename.rsplit('.', 1)
    check_udim = basename.split('_')[-1]
    if check_udim.isdigit():
        udim = int(check_udim)
    return udim


def get_udims(path, udim):
    import glob

    pattern = path.replace(udim, '*')

    all_files = glob.glob(pattern)

    all_udims = []
    for udim_file in all_files:
        udim = get_udim(udim_file)
        if not udim:
            continue
        check_file = pattern.replace('*', '%04d' % udim)
        udim_file = udim_file.replace('\\', '/')
        check_file = check_file.replace('\\', '/')
        if check_file != udim_file:
            continue

        all_udims.append(udim)

    return all_udims, pattern

class TexturePublisher():
    def __init__(self, project):
        self.project = project
        self.database = database.DataBase()
        self.database.fill(project=self.project)
        self.database.query_sg_database('Asset', as_precache=True)
        self.database.query_sg_database('CustomEntity10', as_precache=True)
        self.texture_publish_type = self.database.entity_view('CustomEntity10')['texture']

        self.project_environment = set_environment.ProjectEnvironment(project=self.project)
        self.project_tools_data = self.project_environment.get_project_tools(project)

        project_data = self.project_environment.config_solver.get_config('project')

        self.server_path = project_data['paths']['publish_server']
        self.sandbox_path = project_data['paths']['assets']['sandbox']
        self.project_tools_data['maketx']['name'] = 'maketx'
        self.make_tx_launcher_data = self.project_environment.get_launch_data(self.project_tools_data['maketx'])
        self.make_tx_path = self.make_tx_launcher_data[0]

        self.texture_data = {}

    def get_mask(self, path):
        for channel_check in channel_codes:
            if path.find(channel_check) > -1:
                return channel_check
        return 'DFC'

    def get_texture_data(self, asset_name):
        print('== get texture data ==')
        self.asset_name = asset_name
        self.asset_view = self.database.entity_view('Asset')[self.asset_name]

        all_textures = {}

        for node_type, texture_attribute in texture_node_types.items():
            texture_nodes = cmds.ls(type=node_type)
            for node in texture_nodes:
                texture_path = cmds.getAttr('%s.%s' % (node, texture_attribute))
                all_textures[node] = {'source_path': texture_path}

        sorted_textures = {}

        asset_published_elements = self.asset_view.sg_published_elements
        if asset_published_elements.empty:
            texture_view = database.emptyView()
        else:
            texture_view = asset_published_elements.find_with_filters(sg_publish_type=self.texture_publish_type)

        for node, path_data in all_textures.items():

            path = path_data['source_path']
            published_file_data = None
            for published in texture_view:
                if path.startswith(published.sg_published_folder) and path.find('[') == -1:
                    published_file_data = published
                    print('already published')
                    break

            if published_file_data and not published_file_data.empty:
                path_bits = path.split('/')
                for channel_check in channel_codes:
                    if path.find(channel_check) > -1:
                        channel_mask = channel_check

                name = published_file_data.sg_subset
                sorted_textures[name] = sorted_textures.get(name, {})

                metadata = published_file_data.sg_metadata
                if 'sg_metadata' in metadata:
                    metadata = metadata['sg_metadata']

                udims = metadata['channels'][channel_mask]['udims']

                pattern = published_file_data.sg_files[channel_mask]
                sorted_textures[name][channel_mask] = {'udims': udims,
                                                       'relative_path': pattern,
                                                       'published_folder': published_file_data.sg_published_folder,
                                                       'path': '%s/%s' % (published_file_data.sg_published_folder, pattern),
                                                       'node': node,
                                                       'published_data': published_file_data}

            else:
                basename = os.path.basename(path)
                basename, ext = basename.rsplit('.', 1)
                bits = basename.split('_')
                udim = None
                if bits[-1].isdigit() or bits[-1] == '<UDIM>':
                    udim = bits[-1]
                
                channel = self.get_mask(path)
                name = bits[0]
                sorted_textures[name] = sorted_textures.get(name, {})

                if udim:
                    udims, pattern = get_udims(path, udim)
                else:
                    udims = [1001]
                    pattern = path
                pattern = pattern.replace('*', '<UDIM>')
                sorted_textures[name][channel] = {'udims': udims,
                                                  'path': pattern,
                                                  'node': node,
                                                  'published_data': False}
        self.texture_data = sorted_textures
        return sorted_textures


    def to_local(self):
        import library.sandbox as sandbox
        asset_name = os.environ.get('PIPE_ASSET_NAME')
        if not asset_name:
            print('can\'t get local path, set the context widget to the correct asset')
            return
        asset_type = os.environ['PIPE_ASSET_TYPE']
        sandbox_builder = sandbox.Sandbox( project=self.project)
        sandbox_builder.set_context(asset_name=asset_name, asset_type=asset_type)
        sandbox_path = sandbox_builder.generate_path()

        local_path = '%s/maya/sourceimages' % sandbox_path

        for subset_name, subset_data in self.texture_data.items():
            for mask_name, mask_data in subset_data.items():
                if not mask_data['published_data']:
                    continue

                full_source_path = '%s/%s' %(mask_data['published_folder'], mask_data['relative_path'])
                full_target_path = '%s/%s' % (local_path, mask_data['relative_path'])
                full_target_folder = os.path.dirname(full_target_path)
                full_source_pattern = full_source_path.replace('<UDIM>', '*')

                all_texture_file_path = glob.glob(full_source_pattern)
                if not os.path.exists(full_target_folder):
                    os.makedirs(full_target_folder)

                for texture_file_path in all_texture_file_path:
                    basename = os.path.basename(texture_file_path)
                    full_target_path = '%s/%s' %(full_target_folder, basename)

                    shutil.copy2(texture_file_path, full_target_path)
                    node_type = cmds.nodeType(mask_data['node'])
                    path_attribute = texture_node_types.get(node_type, '')
                    cmds.setAttr('%s.%s' % (mask_data['node'], path_attribute), full_target_path, type='string')

    def make_tx(self, path):
        import subprocess
        print('convert to arnold')
        files_patterm = path.replace('<UDIM>', '*')
        all_files = glob.glob(files_patterm)
        if len(all_files) == 1:
            path_pattern = all_files[0]
        else:
            path_pattern = path.replace('<UDIM>', '<udim>')
        output_path = '%s.tx' % path_pattern.rsplit('.', 1)[0]

        print('make tx exe path ', self.make_tx_path)
        print('make tx path_pattern', path_pattern)
        print('make tx output', output_path)

        process = subprocess.run([self.make_tx_path,
                                  '-v',
                                  '-o',
                                  output_path,
                                  path_pattern],
                                 # shell=False,
                                 # env=env,
                                 capture_output=True,
                                 text=True)
        print(process.stderr)
        print(process.stdout)
        return

    def copy_udims(self, source_pattern, target_pattern, udims):
        print('Copy udims')
        print('*' * 50)
        print(udims)
        publish_folder = os.path.dirname(target_pattern)

        if not os.path.exists(publish_folder):
            os.makedirs(publish_folder)

        for udim in udims:
            source = source_pattern.replace('<UDIM>', '%04d' % udim)
            target = target_pattern.replace('<UDIM>', '%04d' % udim)

            print('copy from : %s' % source)
            print('copy to : %s' % target)
            shutil.copy2(source, target)
            self.make_tx(target)


    def set_attribute(self, node_name, attribute_name, value):
        if not cmds.attributeQuery(attribute_name, n=node_name, exists=True):
            cmds.addAttr(node_name, ln=attribute_name, dt='string')

        print('set attribute %s of %s to %s' % (attribute_name, node_name, value))

        cmds.setAttr('%s.%s' % (node_name, attribute_name), value, type='string')



    def publish(self, task_code, texture_data=None, comment=None):
        if texture_data:
            self.texture_data = texture_data

        self.task_code = task_code

        textures_published = []

        for texture_name, data in self.texture_data.items():
            all_mask_data = {}

            published = True
            publish_data = None
            for mask_code, mask_data in data.items():
                if not mask_data['published_data']:
                    published = False
                    break
                else:
                    publish_data = mask_data['published_data']

            if published:
                textures_published.append(publish_data)
                continue

            for mask_code, mask_data in data.items():
                all_mask_data[mask_code] = {'node': mask_data['node'],
                                            'udims': mask_data['udims'],
                                            'source_path': mask_data['path']}

            extra_data = {'sg_metadata': {'channels': all_mask_data}}

            parameters = {}
            if not comment:
                comment = 'Auto publish textures'

            parameters['task_code'] = self.task_code
            parameters['entity_name'] = self.asset_name
            parameters['comment'] = comment
            parameters['subset'] = texture_name
            parameters['metadata'] = extra_data
            parameters['step'] = 'Shading'


            publish_data = publish_helpers.create_new_publish(self.database, 'texture', parameters, server=self.server_path)

            new_publish_folder = publish_data.sg_published_folder
            files = {}
            for mask_code, mask_data in data.items():
                published_mask_folder = '%s/%s' % (new_publish_folder, mask_code)

                pattern = mask_data['path']
                extension = pattern.split('.')[-1]

                relative_path = '%s/%s_<UDIM>.%s' % (mask_code, texture_name, extension)
                files[mask_code] = relative_path
                target_pattern = '%s/%s_<UDIM>.%s' % (published_mask_folder, texture_name, extension)

                tx_path = '%s/%s_<UDIM>.tx' % (published_mask_folder, texture_name)

                self.copy_udims(pattern, target_pattern, mask_data['udims'])

                node_name = mask_data['node']
                node_type = cmds.nodeType(mask_data['node'])

                attribute = texture_node_types.get(node_type, 'fileTextureName')

                cmds.setAttr('%s.%s' % (mask_data['node'], attribute), target_pattern, type='string')

                if node_type == 'aiImage':
                    cmds.setAttr('%s.autoTx' % mask_data['node'],0)
                elif node_type == 'file':
                    cmds.setAttr('%s.aiAutoTx' % mask_data['node'], 0)

                self.set_attribute(node_name, 'path', pattern)
                self.set_attribute(node_name, 'assetName', publish_data.sg_asset_name)
                self.set_attribute(node_name, 'assetType', publish_data.sg_asset.sg_asset_type_name)
                self.set_attribute(node_name, 'fileTag', mask_code)
                self.set_attribute(node_name, 'pipelineStep', 'Texture')
                self.set_attribute(node_name, 'variant', '')
                self.set_attribute(node_name, 'subset', texture_name)
                self.set_attribute(node_name, 'version', str(publish_data.sg_version_number))
                self.set_attribute(node_name, 'shotgrid_id', str(publish_data.id))



            self.database.sg.update('CustomEntity09', int(publish_data.id), {'sg_files': pformat(files)})

            textures_published.append({'id': int(publish_data.id), 'type': 'CustomEntity09'})

        shader_extra_data = {'sg_down_dependencies': textures_published}
        #cmds.file(save=True, f=True)
        return shader_extra_data

