import os
import importlib
import logging

from pprint import pprint

from PySide import QtWidgets, QtCore

import packages_io

import library.ui.labeled as labeled
import library.ui.context_widget as context_widget
import library.ui.dialog_widgets as dialog_widgets

import library.core.config_manager as config_manager

import shotgrid_lib.lib.progress_query_widget as progress_query_widget

import texture_uploader.lib.texture_uploader_core as uploader_core

importlib.reload(packages_io)
importlib.reload(uploader_core)
importlib.reload(context_widget)

logger = logging.getLogger(__name__)

class TextureUploaderWidget(QtWidgets.QWidget):

    precached_entities = ['Step', 'HumanUser', 'Project', 'CustomEntity10', 'CustomEntity11']
    main_module = 'texture_publisher'
    def __init__(self, parent=None, project=None, task=None):
        super(TextureUploaderWidget, self).__init__( parent=parent)
        self.project = project
        self.task = task
        self.config_manager = config_manager.ConfigSolver(project=self.project)
        self.project_config = self.config_manager.project_config
        self.builder_config = self.config_manager.get_config('builder', module='builder')
        self.context_name = self.get_context()
        self.uploader = uploader_core.TexturePublisher(project=self.project)

        self.pipeline_step = task.step_name
        self.database = task._database


        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(3)
        self.main_layout.setContentsMargins(10, 10, 10, 5)

        self.add_context_widget(self.main_layout)
        self.add_task_widget(self.main_layout)
        self.add_comment_widget(self.main_layout)
        self.add_texture_widget(self.main_layout)
        self.main_layout.addStretch(1)
        self.upload_button = QtWidgets.QPushButton('Upload')
        self.upload_button.pressed.connect(self.launch_upload)
        self.main_layout.addWidget(self.upload_button)
        self.setLayout(self.main_layout)

        self.init_context_widget()

    def launch_upload(self):
        task_code = str(self.task_widget.getValue())
        new_dict = {}

        for index in range(self.texture_widget.topLevelItemCount()):
            item = self.texture_widget.topLevelItem(index)
            status = item.checkState(3)
            set = item.text(0)
            print(set, status)
            if status == QtCore.Qt.CheckState.Checked:
                new_dict[set] = self.texture_data[set]
        comment = self.comment_widget.getValue()

        self.uploader.publish(task_code, texture_data=new_dict, comment=comment)

        dlg = dialog_widgets.CustomDialog(self, 'Done!!', 'The upload finished correctly')
        dlg.exec_()

    def get_context(self):
        context_type = os.environ.get('PIPE_CONTEXT')
        if not context_type:
            return ''

        if context_type == 'Asset':
            asset_type = os.environ.get('PIPE_ASSET_TYPE')
            asset_name = os.environ.get('PIPE_ASSET_NAME')
            if asset_type and asset_name:
                return '%s.%s' % (asset_type, asset_name)

        return ''

    def init_context_widget(self):

        print('init_context_widget')
        self.tasks_view = self.database['Task']

        all_context = {}

        for task in self.tasks_view:
            entity_name = task.entity.code
            if task.entity.type == 'Asset':
                asset_type = task.entity.sg_asset_type
                all_context[asset_type] = all_context.get(asset_type, [])
                if entity_name not in all_context[asset_type]:
                    all_context[asset_type].append(entity_name)

        self.context_widget.setValues(all_context)
        self.context_widget.setEnabled(True)
        self.context_widget.setValue(self.context_name)
        QtWidgets.QApplication.restoreOverrideCursor()


    def add_context_widget(self, layout):
        context_labels = self.builder_config['context_labels']['asset']
        self.context_widget = labeled.CascadeComboWidget('', 'Asset',
                                                         combo_labels=context_labels)
        layout.addWidget(self.context_widget)
        self.context_widget.modified.connect(self.update_textures)
        self.context_widget.modified.connect(self.update_task)

        self.context_widget.setEnabled(False)

        logger.info('set context to : %s' % self.context_name)
        self.context_widget.setValue(self.context_name)

    def add_task_widget(self, layout):
        self.task_widget = labeled.LabeledCombo('', 'Task')
        layout.addWidget(self.task_widget)

    def add_comment_widget(self, layout):
        self.comment_widget = labeled.LabeledText('', 'Comment', editable=True)
        layout.addWidget(self.comment_widget)
    def update_textures(self):
        self.texture_widget.clear()
        print('update textures')
        context = self.context_widget.getValue()
        print(context)
        if not context.get('name', ''):
            return
        self.texture_data = self.uploader.get_texture_data(context['name'])

        print('-' * 100)
        for texture_set, set_data in self.texture_data.items():
            root_item = QtWidgets.QTreeWidgetItem()
            root_item.setText(0, texture_set)
            root_item.setText(1, ','.join(list(set_data.keys())))
            self.texture_widget.addTopLevelItem(root_item)
            published = True
            for channel_name, channel_data in set_data.items():
                channel_item = QtWidgets.QTreeWidgetItem()
                channel_item.setText(1, channel_name)
                str_udims = [str(udim) for udim in channel_data['udims']]
                channel_item.setText(2, ','.join(str_udims))
                if channel_data.get('published_data'):
                    channel_item.setText(3, 'Published')
                else:
                    channel_item.setText(3, 'Local')
                    published = False
                channel_item.setText(5, channel_data['path'])
                root_item.addChild(channel_item)
            if published:
                root_item.setText(3, 'Published')
            else:
                root_item.setText(3, 'Local')
                root_item.setCheckState(3, QtCore.Qt.CheckState.Checked)
    def add_texture_widget(self, parent_layout):

        self.texture_widget = QtWidgets.QTreeWidget()
        labels = ['Set','Channel', 'UDIMS', 'Published', '', 'Path']
        self.texture_widget.setColumnCount(6)
        self.texture_widget.setHeaderLabels(labels)
        parent_layout.addWidget(self.texture_widget)



    def update_task(self):
        value = self.context_widget.getValue()
        if not value.get('name'):
            self.task_widget.setValues([])
            return []

        entity_name = value['name']
        context = self.database.entity_view('Asset')[entity_name]

        task_list = self.database.entity_view('Task')
        pipeline_step_view = self.database.entity_view('Step').find_with_filters(code=self.pipeline_step,
                                                                                 single_item=True)
        current_task = task_list.find_with_filters(entity=context, step=pipeline_step_view)

        task_list = current_task.content
        if not isinstance(task_list, list):
            task_list = [task_list]

        self.task_widget.setValues(task_list)

        return current_task.content





class TextureUploaderWindow(packages_io.MainWindow ):
    """
    Main publisher for all the pipeline steps

    publish_type: The type of the publish to be created
    project: The code of the Shotgrid project to work on it

    """
    def __init__(self, task=None, project=None):
        self.title = 'Texture uploader'

        super(TextureUploaderWindow, self ).__init__(self.title)
        self.project = project
        self.task=task
        self.setContentsMargins(6, 6, 6, 6)
        self.main_widget = TextureUploaderWidget(project=self.project, parent=self, task=task)
        self.apply_style(self)
        self.setMinimumSize(900, 600)
        self.setCentralWidget(self.main_widget)


def open_texture_uploader(task=None, project=None):
    window = TextureUploaderWindow(task=task, project=project)
    window.show()