import os
import sys
import getpass
import logging
import importlib
import importlib.resources as imp_resources
import yaml
from yaml.loader import SafeLoader

from PySide import QtCore, QtGui, QtWidgets


import packages_io
import shotgrid_lib.database as database

import library.ui.tool_table as tool_table
import library.ui.labeled as labeled

import shotgrid_lib.lib.progress_query_widget as progress_query_widget
import launcher.lib.set_environment as set_environment

importlib.reload(tool_table)

logger = logging.getLogger(__name__)
# logger.setLevel(logging.INFO)

bin_path = os.path.dirname(os.path.abspath(__file__))
root_path = os.path.dirname(bin_path)

config_file = os.path.join(root_path, 'config', 'layouts.yaml')

layouts_data = {}
with open(config_file) as f:
    all_layouts_data = yaml.load(f, Loader=SafeLoader)


def getMayaWindow():
    import maya.OpenMayaUI as mui

    ptr = mui.MQtUtil.mainWindow()
    return QtCore.sip.wrapinstance(QtCore.long(ptr), QtCore.QObject)

class DataCard(QtWidgets.QFrame):
    clicked = QtCore.Signal(int)

    def __init__(self, view, parent=None):
        super(DataCard, self).__init__(parent=parent)
        self.view = view
        if hasattr(parent, 'container'):
            self.container = parent.container
        else:
            self.container = parent

        self.setObjectName('Light')
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.Fixed)
        self.layout_config = self.get_layout_config()
        height = self.layout_config.get('height', 105)
        self.setFixedHeight(height)

        self.layout = self.build_layout(self.layout_config)
        self.layout.setSpacing(3)
        self.layout.setContentsMargins(5, 5, 5, 5)
        self.setLayout(self.layout)

    def get_layout_config(self):
        layouts_data = all_layouts_data.get(self.view.type, {})
        return layouts_data

    def get_value(self, instance, variable):

        bits = variable.split('.', 1)
        if len(bits) > 1:
            new_instance = getattr(instance, bits[0])
            return self.get_value(new_instance, bits[-1])
        else:
            return getattr(instance, bits[0])

    def init_widget(self, widget, init_function_name, view, value_name, value):
        module_name, function_name = init_function_name.rsplit('.', 1)
        module = importlib.import_module(module_name)
        function = getattr(module, function_name)

        values = function(view, value_name)
        widget.setValues(values)
        widget.setValue(value)

    def build_layout(self, data):
        if data['arrange'] == 'vertical':
            layout = QtWidgets.QVBoxLayout()
        else:
            layout = QtWidgets.QHBoxLayout()

        for line in data['items']:
            if 'arrange' in line.keys():
                h_layout = self.build_layout(line)
                layout.addLayout(h_layout)
            else:
                if line['value'] == 'self':
                    value = self.view
                else:
                    value = self.get_value(self.view, line['value'])

                if isinstance(value, database.emptyView):
                    value = ''

                module_name =  line.get('module')
                if module_name:
                    widget_module = importlib.import_module(module_name)
                    widget = getattr(widget_module, line['widget'])
                else:
                    widget = getattr(labeled, line['widget'])

                label = line.get('label')
                if line.get('args'):
                    for key, arg_value in line['args'].items():
                        if arg_value == 'self':
                            line['args'][key] = self
                        if arg_value == 'view':
                            line['args'][key] = self.view
                    new_widget = widget(value, label, parent=self, **line['args'])
                else:
                    new_widget = widget(value, label, parent=self)

                layout.addWidget(new_widget)
                var_name = label.lower().replace(' ', '_')
                setattr(self, var_name, new_widget)
                width = line.get('width')
                if width:
                    new_widget.set_width(**width)

        if data.get('connect'):
            for connection in data['connect']:
                function = getattr(self, connection['function'])
                emitter = getattr(self, connection['emitter'])
                signal = getattr(emitter, connection['signal'])
                signal.connect(function)

        return layout

    def on_select(self):
        self.clicked.emit(self.view.id)



class CardContainer(QtWidgets.QFrame):
    on_click = QtCore.Signal(int, bool)
    selected = QtCore.Signal(str)
    def __init__(self, view, attribute, parent=None, widget_class=DataCard, manager=None):
        super(CardContainer, self).__init__(parent=parent)

        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
        self.setObjectName('Light')
        self.card = parent
        self.manager = manager
        self.view = view
        self.attribute = attribute
        self.widget_class = widget_class
        self.images_cache = {}

        self.selected_widget_position = None
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        self.visible = False
        self.widgets = {}

        self.main_widget = QtWidgets.QWidget()
        self.main_widget.setObjectName('Dark')
        self.main_widget.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.scroll_layout = QtWidgets.QVBoxLayout()
        self.scroll_layout.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignTop)

        self.scroll_layout.setSpacing(5)
        self.scroll_layout.setContentsMargins(0, 4, 5, 0)

        self.main_widget.setLayout(self.scroll_layout)

        self.scroll_area = QtWidgets.QScrollArea(self)
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setWidget(self.main_widget)
        self.scroll_area.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.main_layout.addWidget(self.scroll_area)
        self.setLayout(self.main_layout)

    def set_view(self, view):
        self.view = view
        self.project = self.view._database.project

        self.update()

    def clean(self):
        while ((child := self.scroll_layout.takeAt(0)) != None):
            try:
                child.widget().deleteLater()
            except:
                pass
        self.widgets = {}

    def update(self):
        if self.widgets:
            self.clean()

        if self.attribute is None:
            loop = self.view
        else:
            loop = getattr(self.view, self.attribute)

        if not loop.empty:
            loop = loop.get_thumbnails()

        for index, item in enumerate(sorted(loop)):
            self.widgets[item.id] = self.widget_class(item, parent=self)

            self.scroll_layout.insertWidget(0, self.widgets[item.id])
            self.widgets[item.id].clicked.connect(self.clicked)
            self.widgets[item.id].setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding,
                                                QtWidgets.QSizePolicy.MinimumExpanding)
        self.scroll_layout.addStretch(1)
        QtWidgets.QApplication.processEvents()

    def clicked(self, selected_id):
        current_item = self.widgets[selected_id]
        context = current_item.get_context_data()
        task_code = current_item.task_view.content
        os.environ['PIPE_TASK'] = task_code

        entity_type = current_item.task_view.entity.type
        if entity_type == 'Asset':
            os.environ['PIPE_CONTEXT'] = 'Asset'
            os.environ['PIPE_ASSET_NAME'] = current_item.task_view.entity.code

            os.environ['PIPE_ASSET_TYPE'] = current_item.task_view.entity.sg_asset_type
        elif entity_type == 'Shot':
            os.environ['PIPE_CONTEXT'] = 'Shot'
            os.environ['PIPE_SHOT_NAME'] = current_item.task_view.entity.code

            os.environ['PIPE_SEQUENCE_NAME'] = current_item.task_view.entity.sg_sequence_name
            os.environ['PIPE_EPISODE_NAME'] = current_item.task_view.entity.sg_sequence.episode_name

        self.selected.emit(context)

        if not current_item.is_open():
            for item_id, item in self.widgets.items():
                item.setHidden(False)
                item.close_children()

        else:

            for item_id, item in self.widgets.items():
                if item_id == selected_id:
                    item.setHidden(False)
                    item.open_children()
                else:
                    item.setHidden(True)

class TaskActions(QtWidgets.QWidget):
    def __init__(self, task_view, parent=None, container=None):
        super(TaskActions, self).__init__(parent=parent)
        self.visible = False
        self.task_view = task_view
        self.card_container = container

        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
        self.layout = QtWidgets.QVBoxLayout()
        self.layout.setSpacing(0)
        self.layout.setContentsMargins(0, 0, 0, 0)

        self.tab_widget = QtWidgets.QTabWidget(self)
        self.tab_widget.setObjectName('Normal')

        # self.tab_widget.setFixedHeight(500)
        # self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
        self.setLayout(self.layout)

    def update(self):

        tab1 = CardContainer(self.task_view, 'sg_versions', parent=self, manager=self.card_container.manager)
        tab2 = CardContainer(self.task_view, 'open_notes', parent=self, manager=self.card_container.manager)

        tab1.set_view(self.task_view)
        tab2.set_view(self.task_view)

        project_tools = self.card_container.manager.project_tools
        tools_config = self.card_container.manager.tools_config
        tools_utils = self.card_container.manager.utils_config
        tab3 = tool_table.ToolsTable(tools_config, project_tools, parent=self)
        tab_utils = tool_table.ToolsTable(tools_utils, project_tools, parent=self)
        tab1.setObjectName('Normal')
        tab2.setObjectName('Normal')
        tab3.setObjectName('Normal')
        tab_utils.setObjectName('Normal')

        tab3.load_config(self.task_view.step_name)
        tab_utils.load_config(self.task_view.step_name)

        self.tab_widget.addTab(tab3, "Tools")
        self.tab_widget.addTab(tab_utils, "Utils")
        self.tab_widget.addTab(tab2, "Notes")
        self.tab_widget.addTab(tab1, "Versions")

        if self.task_view.entity.type == 'Shot':
            tab_breakdown = CardContainer(self.task_view.entity, 'sg_breakdowns', parent=self, manager=self.card_container.manager)
            tab_breakdown.set_view(self.task_view.entity)
            self.tab_widget.addTab(tab_breakdown, "Breakdown")

    def showEvent(self, event: QtGui.QShowEvent) -> None:

        if self.visible is False:
            self.visible = True
            self.update()

        self.layout.addWidget(self.tab_widget)
        return super().showEvent(event)


class TaskCard(QtWidgets.QWidget):
    clicked = QtCore.Signal(int, bool)

    def __init__(self, task_view, parent=None, actions=True, thumbnail=True):
        super(TaskCard, self).__init__(parent=parent)
        self.task_view = task_view
        self.container = parent
        self.created_children = False
        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)

        self.main_layout = QtWidgets.QVBoxLayout()
        self.setMinimumHeight(100)
        self.main_layout.setSpacing(2)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        self.data_card = DataCard(self.task_view, parent=self)
        self.data_card.clicked.connect(self.on_select)
        self.main_layout.addWidget(self.data_card)

        self.setLayout(self.main_layout)


    def get_context_data(self):
        entity = self.task_view.entity
        return entity.code

    def is_open(self):
        if not self.created_children:
            return False
        return not self.task_actions.isHidden()

    def open_children(self):
        if not self.created_children:
            self.created_children = True
            self.task_actions = TaskActions(self.task_view, parent=self, container=self.container)
            self.main_layout.addWidget(self.task_actions)

        self.task_actions.setHidden(False)
        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

    def close_children(self):
        if self.created_children:
            self.task_actions.setHidden(True)

            self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)

    def on_select(self, task_id):
        value = self.is_open()
        if value:
            self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
            self.close_children()
        else:
            self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)
            self.open_children()

        self.clicked.emit(self.task_view.id, value)


class FilterWidget(QtWidgets.QFrame):
    filter_changed = QtCore.Signal()

    def __init__(self, sg_database, parent=None, actions=True, thumbnail=True):
        super(FilterWidget, self).__init__(parent=parent)

        self.database = sg_database

        self.setObjectName('Light')
        entities = ['All', 'Shot', 'Asset']
        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)

        self.main_layout = QtWidgets.QVBoxLayout()
        self.setMinimumHeight(120)
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(10, 5, 10, 10)

        self.users_widget = labeled.LabeledCombo('', 'Users', parent=self)
        # status_list = task_view.get_options('sg_status_list')

        self.status_widget = labeled.LabeledCombo('', 'Status',  parent=self)
        self.type_widget = labeled.LabeledCombo('All', 'Context', values=entities, parent=self)
        self.step_widget = labeled.LabeledCombo('', 'Step', parent=self)

        self.users_widget.currentIndexChanged.connect(self.emit_changed)
        self.status_widget.currentIndexChanged.connect(self.emit_changed)
        self.type_widget.currentIndexChanged.connect(self.emit_changed)
        self.step_widget.currentIndexChanged.connect(self.emit_changed)

        self.main_layout.addWidget(self.users_widget)
        self.main_layout.addWidget(self.type_widget)
        self.main_layout.addWidget(self.step_widget)
        self.main_layout.addWidget(self.status_widget)

        self.users_widget.set_width(label=80)
        self.status_widget.set_width(label=80)
        self.type_widget.set_width(label=80)
        self.step_widget.set_width(label=80)


        self.setLayout(self.main_layout)

    def set_view(self, view):

        self.blockSignals(True)
        self.task_view = view

        default_status = 'All'
        default_step = 'All'
        default_user_login = getpass.getuser()

        default_user = self.database['HumanUser'].find_with_filters(login=default_user_login, single_item=True)

        raw_user_list = self.database['HumanUser'].single_values('name')
        raw_user_list = list(sorted(raw_user_list))

        self.users_widget.setValues(raw_user_list)
        self.users_widget.setValue(default_user.name)
        if self.task_view.empty:
            self.set_status(['All'], 'All')
            self.set_steps(['All'], 'All')
            self.blockSignals(False)

            return

        status_list = self.task_view.single_values('sg_status_list')
        step_list = self.task_view.single_values('step')

        self.set_status(status_list, default_status)
        self.set_steps(step_list, default_step)

        self.blockSignals(False)


    def emit_changed(self):
        self.filter_changed.emit()

    def get_filters(self):
        output = {'task_assignees': self.users_widget.getValue(),
                  'sg_status_list': self.status_widget.getValue(),
                  'asset_type': self.type_widget.getValue(),
                  'pipeline_step': self.step_widget.getValue(),
                  }
        return output

    def set_status(self, status_list, status):
        status_list.insert(0, 'All')
        self.status_widget.setValues(status_list)
        self.status_widget.setValue(status)

    def set_steps(self, step_list, step):
        step_list.insert(0, 'All')
        self.step_widget.setValues(step_list)
        self.step_widget.setValue(step)


class FilterView( QtWidgets.QFrame):
    def __init__(self,
                 context_chooser=None,
                 project=None,
                 parent=None,
                 threaded=False):

        super(FilterView, self).__init__(parent=parent)
        self.project = project
        self.setObjectName('Dark')
        self.context_chooser = context_chooser
        self.users_queried = []
        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.progress_query = progress_query_widget.ProgressQuery(self.project, parent=self, threaded=threaded)

        self.user_login = getpass.getuser()

        self.current_user = None
        self.user_task = None

        overrides = os.environ.get('SETTING_OVERRIDES', '')

        self.project_environment = set_environment.ProjectEnvironment(project=self.project)
        self.project_tools = self.project_environment.get_project_tools(overrides=overrides)

        config_data = self.project_environment.get_config_file('step_tools', module='task_manager')
        config_utils = self.project_environment.get_config_file('step_utils', module='task_manager')
        for key, value in config_data.items():
            logger.info('%s = %s' % (key, str(value)))

        self.tools_config = config_data
        self.utils_config = config_utils

        filters = [['task_assignees.HumanUser.login', 'is', self.user_login],
                   ['sg_status_list', 'not_in', ['fin', 'cmpt', 'omt', 'na', 'stb']]]

        self.progress_query.set_entity_prefilter('Task', filters)
        self.progress_query.set_entity_prefilter('HumanUser', [])

        self.progress_query.set_precache_entities(['Step',
                                                   'Shot',
                                                   'Asset',
                                                   #'Task',
                                                   'HumanUser',
                                                   'Step',
                                                   'CustomEntity12',
                                                   'CustomEntity11',
                                                   'CustomEntity10'])
        self.database = self.progress_query.database
        self.progress_query.finished.connect(self.set_view)
        self.progress_query.precache_dependencies = {'Task': ['entity', 'sg_versions']}
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(10, 10, 10, 10)

        self.filters_widget = FilterWidget(self.database)
        self.main_layout.addWidget(self.filters_widget)
        self.main_widget = CardContainer(None, None, parent=self, widget_class=TaskCard, manager=self)
        self.main_widget.selected.connect(self.update_context_chooser)
        self.main_layout.addWidget(self.main_widget)
        self.update_widget = QtWidgets.QPushButton('Update')
        self.main_layout.addWidget(self.update_widget)
        self.update_widget.clicked.connect(self.update_filters)
        self.setLayout(self.main_layout)
        self.filters_widget.filter_changed.connect(self.update_filters)
        self.progress_query.finished.connect(self.update_filters)

    def start(self):
        logger.debug('start query database')
        self.progress_query.start()

    def set_view(self, user_login=None ):
        if user_login:
            self.user_login = user_login

        logger.debug('Set user to : %s' % self.user_login)
        selected_user_view = self.database['HumanUser'].find_with_filters(login=self.user_login,
                                                                          single_item=True)

        if selected_user_view.empty:
            user_filters = [['login', 'is', self.user_login]]
            self.database.query_sg_database('HumanUser', filters=user_filters, update=True)
            selected_user_view = self.database['HumanUser'].find_with_filters(login=self.user_login,
                                                                              single_item=True)
        self.tasks_view = self.database['Task']

        if self.tasks_view is not None and not self.tasks_view.empty:

            self.user_task = self.tasks_view.find_with_filters(task_assignees=selected_user_view)
        else:
            self.user_task = database.emptyView()

        self.main_widget.set_view(self.user_task)
        self.filters_widget.set_view(self.user_task)
        self.current_user = selected_user_view.name

    def update_context_chooser(self, context):
        keys = ['asset_name', 'shot']
        if self.context_chooser:
            self.context_chooser.setValue(context)

    def filter_table(self):
        filters = self.filters_widget.get_filters()
        asset_type = filters.get('asset_type', '')
        status = filters.get('sg_status_list', '')
        user = filters.get('task_assignees', '')
        pipeline_step = filters.get('pipeline_step', '')
        self.database.get_events()
        user_filters = [['name', 'is', user]]
        self.database.query_sg_database('HumanUser', filters=user_filters)
        selected_user_view = self.database['HumanUser'].find_with_filters(name=user, single_item=True)
        self.filters_widget.blockSignals(True)

        logger.debug('Querying database for user: %s' % user)

        task_filters = [['task_assignees', 'is', selected_user_view.as_shotgun()],
                        ['sg_status_list', 'not_in', ['fin', 'cmpt', 'omt', 'na', 'stb']]]
        logger.info(str(task_filters))
        self.database.query_sg_database('Task', filters=task_filters, as_precache=False)
        all_tasks = self.database['Task']
        self.user_task = all_tasks.find_with_filters(task_assignees=selected_user_view,
                                                     sg_status_list=['ip', 'wtg', 'rev', 'rtk', 'pcr', 'pdr'])

        logger.info('num tasks: %s' % len(self.user_task))

        self.users_queried.append(user)
        self.current_user = user

        if not self.user_task.empty:
            status_list = self.user_task.single_values('sg_status_list')
            step_list = list(set(self.user_task.step_name))
        else:
            status_list = []
            step_list = []

        self.filters_widget.set_status(status_list, status)
        self.filters_widget.set_steps(step_list, pipeline_step)

        self.filters_widget.blockSignals(False)

        view_filters = {}

        if pipeline_step != 'All':
            logger.debug('Set pipeline_step filter to : %s' % pipeline_step)

            step_view = self.database['Step'].find_with_filters(code=pipeline_step,
                                                                entity_type='Asset',
                                                                single_item=True
                                                                )
            if step_view.empty:
                step_view = self.database['Step'].find_with_filters(code=pipeline_step,
                                                                    entity_type='Shot',
                                                                    single_item=True
                                                                    )

            view_filters['step'] = step_view


        if asset_type != 'All':
            logger.debug('Set asset_type filter to : %s' % asset_type)
            view_filters['entity'] = self.database[asset_type]

        if status != 'All':
            logger.debug('Set status filter to : %s' % status)
            view_filters['sg_status_list'] = status

        filtered_tasks = self.user_task.find_with_filters(**view_filters)

        self.view = filtered_tasks
        return filtered_tasks

    def update_filters(self):
        filtered_tasks = self.filter_table()
        self.main_widget.view = filtered_tasks
        self.main_widget.update()
        QtWidgets.QApplication.processEvents()
        #filtered_tasks.precache_dependencies(fields=['entity', 'sg_versions'])

    def run(self):
        return self

class TaskManagerWindow(packages_io.MainWindow):
    windowTitle = "Task manager"
    windowObject = "TaskManagerWinObject"
    def __init__(self, project, menu_builder=None):
        super(TaskManagerWindow, self).__init__('Task manager', style_sheet='task_manager')
        self.project = project
        self.menu_builder = menu_builder
        self.setWindowTitle('Task manager')
        self.setObjectName('TaskManagerWinObject')
        self.setContentsMargins(0, 0, 0, 0)
        self.setMinimumHeight(800)
        self.setMinimumWidth(650)
        self.apply_style(self)
        self.main_widget = FilterView(project=self.project, context_chooser=menu_builder)
        self.setCentralWidget(self.main_widget)
        QtWidgets.QApplication.processEvents()

        self.main_widget.start()

    def run(self):
        return self



if __name__ == '__main__':
    app = QtWidgets.QApplication(sys.argv)
    project = os.environ['PROJECT']
    window = TaskManagerWindow(project=project)
    window.show()

    app.exec_()
