import logging
import importlib
import importlib.resources as imp_resources

from PySide import QtWidgets, QtCore
from maya.app.general.mayaMixin import MayaQWidgetDockableMixin

import task_manager.bin.task_manager_ui as task_manager_ui

import library.ui.style_sheets as style_sheets
import library.ui.style_sheets.colors as stylesheet_colors

importlib.reload(task_manager_ui)

logger = logging.getLogger(__name__)
# logger.setLevel(logging.DEBUG)

class DockableManager(MayaQWidgetDockableMixin, QtWidgets.QWidget):
    ''' MayaQWidgetDockableMixin is not a workspace control in itself, it is added
        as a child to a created workspace control. See help(MayaQWidgetDockableMixin)
        for more details. The following class is a simple widget with a layout and  a push button.
    '''

    def __init__(self,
                 project,
                 context_chooser=None,
                 parent=None,
                 window_title='',
                 threaded=True):
        super(DockableManager, self).__init__(parent=parent)
        self.project = project
        logger.info('Set task manager to %s project' % self.project)

        print('DockableManager - Set task manager to %s project' % self.project)
        self.style_sheet = 'task_manager'

        self.context_chooser = context_chooser
        # get maya window
        self.main_widget = task_manager_ui.FilterView(project=self.project,
                                                      context_chooser=self.context_chooser,
                                                      threaded=threaded)
        self.apply_style(self)
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        self.main_layout.addWidget(self.main_widget)

        self.setLayout(self.main_layout)
        # set window name
        self.setWindowTitle(window_title)
        # set window flags
        self.setWindowFlags(QtCore.Qt.Window)

    def start(self):
        self.main_widget.start()

    def apply_style(self, central_widget):

        style_sheet_resolver = imp_resources.files(style_sheets)
        with imp_resources.as_file(style_sheet_resolver) as path:
            style_sheet_path = '%s/%s.qss' % (path, self.style_sheet)

        colors = stylesheet_colors.stylesheet_color()
        with open(style_sheet_path, "r") as f:
            _style = f.read()

            for key, value in colors.items():
                _style = _style.replace('<%s>' % key, value)
            central_widget.setStyleSheet(_style)

def set_task_manager(project,
                     menu_builder=None,
                     windowTitle='Task manager',
                     windowObject="TaskManagerWinObject",
                     threaded=True):
    import maya.cmds as cmds
    print('set_task_manager - Set task manager to %s project' % project)

    task_manager_ui = DockableManager(project,
                                      context_chooser=menu_builder,
                                      window_title=windowTitle,
                                      threaded=threaded)

    task_manager_ui.setObjectName(windowObject)

    full_object = windowObject + 'WorkspaceControl'

    if cmds.workspaceControl(full_object, q=True, ex=True):
        cmds.deleteUI(full_object)

    task_manager_ui.show(dockable=True, area='right', floating=True, height=900, width=400)

    cmds.workspaceControl(full_object,
                          e=True,
                          ttc=["AttributeEditor", -1],
                          wp="preferred",
                          vis=True,
                          mw=200)

    task_manager_ui.start()

