import logging

from PySide import QtWidgets, QtCore

import shotgrid_lib.database as database


class DatabaseQuery(QtCore.QObject):
    """
    DatabaseQuery: class to manage the calls to shotgrid
    """
    finished = QtCore.Signal()
    progress = QtCore.Signal(int)
    next_load = QtCore.Signal(str)

    def __init__(self, database, precache_entities=[], entities_prefilters={}, precache_dependencies={}):
        """
        Init the database
        """

        super(DatabaseQuery, self).__init__()

        logging.info('Query project database locally :')
        self.database = database
        self.precache_entities = precache_entities
        self.entities_prefilters = entities_prefilters.copy()
        self.precache_dependencies = precache_dependencies

    def precache_entity(self, entity, filters=[]):
        self.next_load.emit(entity)
        pre_filters = self.entities_prefilters.get(entity, [])
        pre_filters += filters
        self.database.query_sg_database(entity, pre_filters, as_precache=True)

    def run(self):
        count = len(self.precache_entities) - 1
        for index, entity in enumerate(self.precache_entities):
            if count != 0:
                percent = index / count * 100
            else:
                percent = 0

            self.precache_entity(entity)
            self.progress.emit(percent)

        for precache_dependency, precache_fields in self.precache_dependencies.items():
            to_precache = self.database[precache_dependency]
            if to_precache and not to_precache.empty:
                to_precache.precache_dependencies(fields=precache_fields)
        self.progress.emit(100)
        self.finished.emit()


class ProgressQuery(QtCore.QObject):
    finished = QtCore.Signal()
    progress = QtCore.Signal(int)
    next_load = QtCore.Signal(str)

    def __init__(self, project, threaded=True, parent=None):
        self.title = 'Breakdown manager'
        super().__init__(parent=parent)

        self.project = project
        self.threaded = threaded
        self.parent_widget = parent
        self.database = database.DataBase()
        self.database.fill(self.project)
        self.precache_entities = []
        self.precache_dependencies = {}
        self.entities_prefilters = self.database._pre_filters.copy()

    def set_precache_entities(self, precache_entities):
        if isinstance(precache_entities, str):
            self.precache_entities.append(precache_entities)

        elif isinstance(precache_entities, list):
            self.precache_entities += precache_entities

    def set_entity_prefilter(self, entity, prefilters):
        self.entities_prefilters[entity] = prefilters

    def clear_entity_prefilter(self, entity):
        self.entities_prefilters[entity] = []

    def threaded_database_query(self, as_thread=True):
        QtWidgets.QApplication.setOverrideCursor(QtCore.Qt.WaitCursor)

        self.database_worker = DatabaseQuery(self.database,
                                             self.precache_entities,
                                             self.entities_prefilters,
                                             self.precache_dependencies)
        self.database_worker.next_load.connect(self.next_entity)
        self.database_worker.progress.connect(self.report_progress)
        self.database_worker.finished.connect(self.set_finished)

        self.progressDialog = QtWidgets.QProgressDialog("Connecting database ..", None, 0, 100, self.parent_widget)
        self.progressDialog.autoClose()
        self.progressDialog.show()

        if self.threaded and False:
            self.database_thread = QtCore.QThread()
            self.database_worker.finished.connect(self.database_thread.quit)
            self.database_worker.finished.connect(self.database_worker.deleteLater)

            self.database_worker.moveToThread(self.database_thread)
            self.database_thread.started.connect(self.database_worker.run)
            self.database_thread.finished.connect(self.database_thread.deleteLater)
            self.database_thread.start()

        else:
            self.database_worker.run()

    def next_entity(self, entity):
        self.progressDialog.setLabelText('Querying %s' % entity)

    def report_progress(self, percent):
        self.progressDialog.setValue(percent)

    def set_finished(self):
        self.progressDialog.setLabelText('Done')
        self.progressDialog.setValue(100)
        QtWidgets.QApplication.restoreOverrideCursor()
        self.finished.emit()

    def start(self):
        self.threaded_database_query()

