import os

from pprint import pprint

PUBLISH_ENTITY = 'CustomEntity09'
PUBLISH_TYPE_ENTITY = 'CustomEntity10'
VARIANT_ENTITY = 'CustomEntity11'
BREAKDOWN_ENTITY = 'CustomEntity11'




def get_data_from_path(database, path, project='SGD', fields=None):
    path = path.replace( '\\', '/')
    folders = path.split('/')
    hash = None
    version = None
    for folder in folders:
        last = folder.split('_')[-1]
        if not last:
            continue
        if len(last) == 16 and len(folder) > 16:
            hash = last

        if last[0] == 'v' and last[1:].isdigit():
            version = int(last[1:])

    if not version or not hash:
        return None
    database.query_sg_database(PUBLISH_ENTITY, filters=[['sg_hash', 'is', hash]])
    published = database[PUBLISH_ENTITY].find_with_filters(sg_hash=hash,
                                                           sg_version_number=version,
                                                           single_item=True,
                                                           sg_status_list='cmpt'
                                                           )
    if published:
        return published

    published = database[PUBLISH_ENTITY].find_with_filters(sg_hash=hash,
                                                           sg_version_number=version,
                                                           )


    return max(published)

def generate_hash(fields_raw, publish_data):
    import hashlib
    #fields_raw = publish_type_data['sg_hash_fields']
    fields_raw = fields_raw.replace(' ', '')
    fields = fields_raw.split(',')

    hash_items = []
    for field_key in fields:
        if field_key not in publish_data:
            return
        key_data = publish_data[field_key]
        if isinstance(key_data, dict):
            hash_items.append(str(key_data['id']))
            hash_items.append(key_data['type'])
        else:
            hash_items.append(str(key_data))

    chunk = '_'.join(hash_items)
    chunk = chunk.encode('utf-8')
    h = hashlib.sha1()
    h.update(chunk)
    this_hash = h.hexdigest()
    this_hash = this_hash[:16]

    return this_hash

def solve_pattern(pattern, publish_data):
    import shotgrid_lib.database as sg_database
    solved_pattern = pattern
    for key, value in publish_data.items():
        full_key = '[%s]' % key
        if solved_pattern.find(full_key) == -1:
            continue
        if isinstance(value, sg_database.View) and value.empty:
            continue
        elif isinstance(value, sg_database.View):
            value = value.as_shotgun()
        if isinstance(value, dict):
            if 'name' in value:
                value = value['name']
            elif 'code' in value:
                value = value['code']
            elif 'content' in value:
                value = value['content']
            else:
                continue
        elif str(value).isdigit():
            value = '%03d' % int(value)
        solved_pattern = solved_pattern.replace(full_key, value)
    return solved_pattern


def init_connection(project):
    import shotgun_api3
    import library.core.config_manager as config_manager

    config_manager = config_manager.ConfigSolver(project=project)

    connection_data = config_manager.get_config('database_connection', module='shotgrid_lib')

    sg_connection = shotgun_api3.Shotgun(connection_data['connection']['server'],
                                         script_name=connection_data['connection']['script_name'],
                                         api_key=connection_data['connection']['api_key'])
    return sg_connection

def create_asset(connection, asset_name, asset_type, project=None):
    project_data = get_project_data(connection, project)
    data = {'code': asset_name, 'sg_asset_type': asset_type, 'project': project_data}
    asset_data = connection.create('Asset', data)
    return asset_data

def create_task(connection, asset_data, task_name, step, variant_name='Master', project=None, create=True):
    project_data = get_project_data(connection, project)

    variant_data = get_variant_data(connection, variant_name, project=project)
    if not variant_data and create:
        variant_data = create_variant(connection, variant_name, project_data)
    step_data = get_step_data(connection, step)
    data = {'content': task_name, 'entity': asset_data,  'sg_variant': variant_data, 'project': project_data, 'step': step_data}
    connection.create('Task', data)

def get_step_data(connection, step, project=None):
    filters = [['code', 'is', step],
               ]

    step_data = connection.find_one('Step', filters=filters)
    return step_data


def get_publish_data_by_id(sg_connection, id, project):
    filters = [['project.Project.code', 'is', project],
               ['id', 'is', id],
               ]

    fields = ['id',
              'code',
              'sg_artist',
              'sg_version_number',
              'project',
              'sg_task',
              'sg_variant',
              'sg_context',
              'sg_asset',
              'sg_asset.Asset.sg_asset_type',
              'sg_step',
              'sg_published_folder',
              'sg_files',
              'description']
    publish_data = sg_connection.find_one(PUBLISH_ENTITY, filters=filters, fields=fields)
    return publish_data


def get_user_data(sg_connection, user_login):

    filters = [['login', 'is', user_login]]
    fields = ['id', 'name', 'login', 'type']
    file_type_data = sg_connection.find_one('HumanUser', filters=filters, fields=fields)
    return file_type_data


def get_publish_type(sg_connection, file_type, fields=[]):
    filters = [['code', 'is', file_type]]
    fields = ['id', 'code', 'sg_code_pattern', 'sg_hash_fields', 'sg_path_pattern'] + fields
    file_type_data = sg_connection.find_one(PUBLISH_TYPE_ENTITY, filters=filters, fields=fields)
    return file_type_data

def get_asset_data(sg_connection, asset_name, project, fields=[]):
    filters = [['project.Project.code', 'is', project],
               ['code', 'is', asset_name],
               ]
    base_fields = ['code', 'id', 'type'] + fields
    published = sg_connection.find_one('Asset', filters=filters, fields=base_fields)
    return published


def create_variant(connection, variant, project_data):
    data = {'project': project_data,
            'code': variant}

    out_data = connection.create(VARIANT_ENTITY, data)

    return out_data
def get_variant_data(sg_connection, variant, project=None, fields=[]):
    if project is None:
        project = os.environ['PROJECT']

    filters = [['project.Project.code', 'is', project],
               ['code', 'is', variant],
               ]
    fields = ['code', 'id', 'type'] + fields

    published = sg_connection.find_one(VARIANT_ENTITY, filters=filters, fields=fields)

    return published

def get_task_data(sg_connection, task_code, asset_name=None, shot_name=None, project=None, fields=[]):
    if project is None:
        project = os.environ['PROJECT']

    filters = [['project.Project.code', 'is', project],
               ['content', 'is', task_code]
               ]
    if asset_name:
        filters.append(['entity', 'name_is', asset_name])
    if shot_name:
        filters.append(['entity', 'name_is', shot_name])
    fields = ['content', 'id', 'type', 'sg_variant'] + fields
    published = sg_connection.find_one('Task', filters=filters, fields=fields)
    return published

def get_project_data(sg_connection, project):
    filters = [['code', 'is', project]]
    published = sg_connection.find_one('Project', filters=filters)
    return published

def get_assets(sg_connection, project, fields=[]):
    filters = [['project.Project.code', 'is', project]]
    base_fields = ['code', 'id', 'type'] + fields

    published = sg_connection.find('Asset', filters=filters, fields=base_fields)
    return published


def get_shots(sg_connection, project, fields=[]):
    filters = [['project.Project.code', 'is', project]]
    base_fields = ['code', 'id', 'type', 'sequence', 'sequence.Sequence.episode'] + fields

    published = sg_connection.find('Shot', filters=filters, fields=base_fields)
    return published

def get_breakdown(sg_connection, shot_name, project='SGD', fields=[]):
    filters = [['project.Project.code', 'is', project],
               ['sg_link', 'name_is', shot_name]
               ]

    base_fields = ['code',
                   'id',
                   'type',
                   'sg_alias',
                   'sg_shading_variant',
                   'sg_geometry_variant',
                   'sg_asset',
                   'sg_instance',
                   'sg_asset.Asset.sg_asset_type',
                   'sg_parent_asset'] + fields

    published = sg_connection.find('CustomEntity12', filters=filters, fields=base_fields)
    return published


def get_context_data(sg_connection, project='SGD', fields=[]):
    import databased.publisher.lib.helpers as publish_helpers

    context = publish_helpers.get_context()
    context_type = os.environ['PIPE_CONTEXT']
    if context_type == 'Shot':
        code = context.replace('.', '_')
        entity = 'Shot'
    else:
        code = context.split('.')[-1]
        entity = 'Asset'

    filters = [['project.Project.code', 'is', project],
               ['code', 'is', code]]

    data = sg_connection.find_one(entity, filters=filters)

    return data


if __name__ == '__main__':

    connection = init_connection()
    shot_name = 's01_ep01_sq010_sh010'
    data = get_breakdown(connection, shot_name, project='SGD', fields=[])
    pprint(data)