import sys
import importlib
import ast

from PySide import QtWidgets, QtCore, QtGui

import shotgrid_lib.database as database
import library.core.config_manager as config_manager

import library.ui.labeled as labeled

class BaseCard(QtWidgets.QFrame):
    clicked = QtCore.Signal(int)
    layouts_data = {}
    project = ''
    config_manager = None

    def __init__(self, view, parent=None, add_thumbnail=True):
        super(BaseCard, self).__init__(parent=parent)
        self.view = view
        self.database = self.view._database
        self.setAcceptDrops(True)

        self.setFixedHeight(100)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.Fixed)
        self.main_layout = QtWidgets.QHBoxLayout()
        self.setLayout(self.main_layout)
        self.add_thumbnail = add_thumbnail
        if self.add_thumbnail:
            self.add_thumbnail_widget(self.main_layout)

        self.data_layout = QtWidgets.QVBoxLayout()
        self.main_layout.addLayout(self.data_layout)

    def add_thumbnail_widget(self, layout):
        self.thumbnail_widget = labeled.ThumbnailWidget(self.view, 'Thumbnail')
        layout.addWidget(self.thumbnail_widget)

    @classmethod
    def config_card(cls):
        pass

    def add_mime_data(self, format, string, mime_item):
        app_prefix = 'application/shotgrid'
        encode_string = QtCore.QByteArray(str(string).encode())

        mime_format = '%s/%s' % (app_prefix, format)
        mime_item.setData(mime_format, encode_string)

    def mouseMoveEvent(self, e):

        if e.buttons() == QtCore.Qt.LeftButton:
            drag = QtGui.QDrag(self)
            mime = QtCore.QMimeData()
            mime.setText(self.view.code)

            for published in self.published_items:
                if published.sg_publish_type_name == 'model' and published.sg_status_list == 'cmpt':
                    files = ast.literal_eval(published.sg_files)

                    if 'abc_high' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc_high'])
                    elif 'abc' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc'])

                    full_path = r'D:\work\atlantis\mock_server\projects\sgd\publish\model\LeoMesi\ModelingLow\LeoMesi_Low_4f1bb89bcf9fc3dd\v021\abc\LeoMesi_high.abc'

                    full_path = full_path.replace('\\', '/')
                    mime.setUrls([full_path])
                    self.add_mime_data('code', published.code, mime)
                    self.add_mime_data('version_number', str(published.sg_version_number), mime)
                    self.add_mime_data('asset_name', published.sg_asset_name, mime)
                    self.add_mime_data('step', published.sg_step, mime)
                    self.add_mime_data('variant', published.sg_variant, mime)
                    self.add_mime_data('hash', published.sg_hash, mime)
                    self.add_mime_data('project', published.project, mime)
                    self.add_mime_data('sg_status_list', published.sg_status_list, mime)

            drag.setMimeData(mime)

            if self.add_thumbnail:
                thumbnail_widget = self.thumbnail_widget
                pixmap = QtGui.QPixmap(thumbnail_widget.size())
                thumbnail_widget.render(pixmap)
                drag.setPixmap(pixmap)

            print(mime.formats())
            drag.exec_(QtCore.Qt.MoveAction)

    def dropEvent(self, event):
        """ """
        print("Dropped!")


class AssetCard(BaseCard):
    clicked = QtCore.Signal(int)
    layouts_data = {}
    project = ''
    config_manager = None

    def __init__(self, view, parent=None):
        super(AssetCard, self).__init__(view, parent=parent)
        self.max_published_geometry = None
        self.setFixedHeight(100)

        self.add_asset_data(self.data_layout)
        self.add_variant_data(self.data_layout)

        self.data_layout.addStretch(1)

    def add_asset_data(self, layout):
        asset_layout = QtWidgets.QHBoxLayout()
        self.asset_name_widget = labeled.LabeledText(self.view.code, 'Asset', as_label=True)
        self.asset_name_widget.label_widget.setFixedWidth(60)
        self.asset_type_widget = labeled.LabeledText(self.view.sg_asset_type_name, 'Type', as_label=True)
        self.asset_type_widget.label_widget.setFixedWidth(60)
        asset_layout.addWidget(self.asset_name_widget)
        asset_layout.addWidget(self.asset_type_widget)
        asset_layout.addStretch(1)
        layout.addLayout(asset_layout)

    def add_variant_data(self, layout):
        variant_layout = QtWidgets.QHBoxLayout()

        steps_view = self.database.cached_table('Step')

        modeling_step = steps_view['Model']
        shading_step = steps_view['Shading']

        published_items = self.database.cached_table('CustomEntity09')

        self.published_geometry_items = published_items.find_with_filters(sg_asset=self.view, sg_step=modeling_step)
        self.published_shading_items = published_items.find_with_filters(sg_asset=self.view, sg_step=shading_step)

        self.published_items = self.published_shading_items + self.published_geometry_items

        geometry_variants = self.published_geometry_items.single_values('sg_variant')
        shading_variants = self.published_shading_items.single_values('sg_variant')

        self.geometry_variant_widget = labeled.LabeledCombo('Master', 'Geometry', values=geometry_variants)
        self.geometry_variant_widget.label_widget.setFixedWidth(60)
        self.geometry_variant_widget.currentIndexChanged.connect(self.update_publish_entities)

        self.shading_variant_widget = labeled.LabeledCombo('Master', 'Shading', values=shading_variants)

        self.shading_variant_widget.label_widget.setFixedWidth(60)
        variant_layout.addWidget(self.geometry_variant_widget)
        variant_layout.addWidget(self.shading_variant_widget)
        variant_layout.addStretch(1)
        layout.addLayout(variant_layout)
        self.update_publish_entities()

    def update_publish_entities(self):
        variant_name = str(self.geometry_variant_widget.getValue())
        variant_view = self.database.cached_table('CustomEntity11')[variant_name]
        published_variant_view = self.published_geometry_items.find_with_filters(sg_variant=variant_view,
                                                                                 sg_status_list='cmpt')

        if published_variant_view.empty:
            published_variant_view = self.published_geometry_items.find_with_filters(sg_variant=variant_view)

        if published_variant_view.empty:
            self.max_published_geometry = None
            return

        self.max_published_geometry = max(published_variant_view)
        print(self.max_published_geometry)

    def mouseMoveEvent(self, e):

        if e.buttons() == QtCore.Qt.LeftButton:
            if not self.max_published_geometry:
                return
            drag = QtGui.QDrag(self)
            mime = QtCore.QMimeData()
            mime.setText(self.view.code)

            published = self.max_published_geometry
            files = published.sg_files

            if 'abc_high' in files:
                full_path = '%s/%s' % (published.sg_published_folder, files['abc_high'])
            elif 'abc' in files:
                full_path = '%s/%s' % (published.sg_published_folder, files['abc'])

            full_path = r'D:\work\atlantis\mock_server\projects\sgd\publish\model\LeoMesi\ModelingLow\LeoMesi_Low_4f1bb89bcf9fc3dd\v021\abc\LeoMesi_high.abc'

            full_path = full_path.replace('\\', '/')
            mime.setUrls([full_path])
            self.add_mime_data('code', published.code, mime)
            self.add_mime_data('version_number', str(published.sg_version_number), mime)
            self.add_mime_data('asset_name', published.sg_asset_name, mime)
            self.add_mime_data('step', published.sg_step, mime)
            self.add_mime_data('variant', published.sg_variant, mime)
            self.add_mime_data('hash', published.sg_hash, mime)
            self.add_mime_data('project', published.project, mime)
            self.add_mime_data('sg_status_list', published.sg_status_list, mime)

            drag.setMimeData(mime)

            if self.add_thumbnail:
                thumbnail_widget = self.thumbnail_widget
                pixmap = QtGui.QPixmap(thumbnail_widget.size())
                thumbnail_widget.render(pixmap)
                drag.setPixmap(pixmap)

            print(mime.formats())
            drag.exec_(QtCore.Qt.MoveAction)


class DataCard(QtWidgets.QFrame):
    clicked = QtCore.Signal(int)
    layouts_data = {}
    project = ''
    config_manager = None

    def __init__(self, view, parent=None):
        super(DataCard, self).__init__(parent=parent)
        self.view = view
        self.setAcceptDrops(True)

        self.setFixedHeight(100)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.Fixed)

        self.layout_config = self.layouts_data.get(self.view.type, {})
        self.widget_list = []
        self.main_layout = self.build_layout(self.layout_config)
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(4, 4, 4, 4)
        self.setLayout(self.main_layout)

    def add_mime_data(self, format, string, mime_item):
        app_prefix = 'application/shotgrid'
        encode_string = QtCore.QByteArray(str(string).encode())

        mime_format = '%s/%s' % (app_prefix, format)
        mime_item.setData(mime_format, encode_string)

    def mouseMoveEvent(self, e):

        if e.buttons() == QtCore.Qt.LeftButton:
            drag = QtGui.QDrag(self)
            mime = QtCore.QMimeData()
            mime.setText(self.view.code)

            for published in self.view.sg_published_elements:
                if published.sg_publish_type_name == 'model' and published.sg_status_list == 'cmpt':
                    files = ast.literal_eval(published.sg_files)

                    if 'abc_high' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc_high'])
                    elif 'abc' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc'])

                    full_path = r'D:\work\atlantis\mock_server\projects\sgd\publish\model\LeoMesi\ModelingLow\LeoMesi_Low_4f1bb89bcf9fc3dd\v021\abc\LeoMesi_high.abc'

                    full_path = full_path.replace('\\', '/')
                    mime.setUrls([full_path])
                    self.add_mime_data('code', published.code, mime)
                    self.add_mime_data('version_number', str(published.sg_version_number), mime)
                    self.add_mime_data('asset_name', published.sg_asset_name, mime)
                    self.add_mime_data('step', published.sg_step, mime)
                    self.add_mime_data('variant', published.sg_variant, mime)
                    self.add_mime_data('hash', published.sg_hash, mime)
                    self.add_mime_data('project', published.project, mime)
                    self.add_mime_data('sg_status_list', published.sg_status_list, mime)

            drag.setMimeData(mime)

            thumbnail_widget = self
            for current_widget in self.widget_list:
                if isinstance(current_widget, labeled.ThumbnailWidget):
                    thumbnail_widget = current_widget
                    break

            pixmap = QtGui.QPixmap(thumbnail_widget.size())
            thumbnail_widget.render(pixmap)
            drag.setPixmap(pixmap)
            print(mime.formats())
            drag.exec_(QtCore.Qt.MoveAction)

    def dropEvent(self, event):
        """ """
        print("Dropped!")

    @classmethod
    def config_card(cls):
        config_manager_instance = config_manager.ConfigSolver(project='')
        config = config_manager_instance.get_config('layouts', module='breakdown_manager')

        cls.layouts_data = config

    def get_value(self, instance, variable):

        bits = variable.split('.', 1)
        if len(bits) > 1:
            new_instance = getattr(instance, bits[0])
            if isinstance(new_instance, database.View):
                output = []
                for item in new_instance:
                    output_item = self.get_value(item, bits[-1])
                    if isinstance(output_item, str) and output_item not in output:
                        output.append(output_item)
            else:
                output = self.get_value(new_instance, bits[-1])

            return output
        else:
            return getattr(instance, bits[0])

    def init_widget(self, widget, init_function_name, view, value_name, value):
        module_name, function_name = init_function_name.rsplit('.', 1)
        module = importlib.import_module(module_name)
        function = getattr(module, function_name)
        values = function(view, value_name)
        widget.setValues(values)
        widget.setValue(value)

    def query_variant(self, query_value):
        value = ''
        if query_value == 'self':
            value = self.view
        elif query_value is not None:
            value = self.get_value(self.view, query_value)

        if isinstance(value, database.emptyView):
            value = ''

        return value

    def build_layout(self, data):
        if data['arrange'] == 'vertical':
            layout = QtWidgets.QVBoxLayout()
        else:
            layout = QtWidgets.QHBoxLayout()

        for line in data['items']:
            if 'arrange' in line.keys():
                h_layout = self.build_layout(line)
                layout.addLayout(h_layout)
            else:
                label = line.get('label')

                query_value = line.get('value')
                query_values = line.get('values')
                value = self.query_variant(query_value)
                values = self.query_variant(query_values)

                if len(values) == 1:
                    value = values[0]

                widget = getattr(labeled, line['widget'])

                line['args'] = line.get('args', {})

                if values:
                    line['args']['values'] = values
                for key, arg_value in line['args'].items():
                    if arg_value == 'self':
                        line['args'][key] = self
                    if arg_value == 'view':
                        line['args'][key] = self.view

                new_widget = widget(value, label, parent=self, **line['args'])

                layout.addWidget(new_widget)
                self.widget_list.append(new_widget)
                var_name = label.lower().replace(' ', '_')
                setattr(self, var_name, new_widget)

        if data.get('connect'):
            for connection in data['connect']:
                function = getattr(self, connection['function'])
                emitter = getattr(self, connection['emitter'])
                signal = getattr(emitter, connection['signal'])
                signal.connect(function)

        return layout

    def on_select(self):
        self.clicked.emit(self.view.id)


class CardContainer(QtWidgets.QFrame):
    on_click = QtCore.Signal(int, bool)

    def __init__(self, attribute, view=None, parent=None, widget_class=DataCard):
        super(CardContainer, self).__init__(parent=parent)

        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.card = parent
        self.view = view
        self.attribute = attribute
        widget_class.config_card()
        self.widget_class = widget_class

        self.selected_widget_position = None
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        self.visible = False
        self.widgets = {}

        self.main_widget = QtWidgets.QFrame()
        self.main_widget.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.scroll_layout = QtWidgets.QVBoxLayout()
        self.scroll_layout.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignTop)

        self.scroll_layout.setSpacing(10)
        self.scroll_layout.setContentsMargins(0, 4, 0, 0)

        self.main_widget.setLayout(self.scroll_layout)

        self.scroll_area = QtWidgets.QScrollArea(self)
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setWidget(self.main_widget)
        self.scroll_area.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.main_layout.addWidget(self.scroll_area)
        self.setLayout(self.main_layout)

    def clean(self):
        while ((child := self.scroll_layout.takeAt(0)) != None):
            child.widget().deleteLater()
            self.widgets = {}

    def update_cards(self):
        print('addCards')
        if self.widgets:
            self.clean()

        if self.attribute is None:
            loop = self.view
        else:
            loop = getattr(self.view, self.attribute)

        for index, item in enumerate(loop):
            self.widgets[item.id] = self.widget_class(item, parent=self)
            self.scroll_layout.addWidget(self.widgets[item.id])
            self.widgets[item.id].clicked.connect(self.clicked)
            self.update()

    def clicked(self, selected_id):
        current_item = self.widgets[selected_id]
        if not current_item.is_open():
            for item_id, item in self.widgets.items():
                item.setHidden(False)
                item.close_children()

        else:

            for item_id, item in self.widgets.items():
                if item_id == selected_id:
                    item.setHidden(False)
                    item.open_children()
                else:
                    item.setHidden(True)


class TestWindow(QtWidgets.QMainWindow):
    def __init__(self, parent=None):
        super().__init__(parent=parent)

        self.setWindowTitle("ProgressBar")
        self.setGeometry(300, 200, 500, 400)
        self.project = 'sgd'
        self.main_widget = CardContainer(None, parent=self, widget_class=AssetCard)
        self.progress_query = ProgressQuery(self.project)
        self.progress_query.finished.connect(self.fill)
        self.progress_query.set_entity_prefilter('CustomEntity09',
                                                 [['sg_step.Step.code', 'in', ['Model', 'Shading']],
                                                  ['sg_delete', 'is', False],
                                                  ['sg_complete', 'is', True],
                                                  ])

        self.progress_query.set_entity_prefilter('Asset', [['sg_asset_type', 'is', 'Setprops']])

        self.progress_query.set_precache_entities(['Asset',
                                                   'CustomEntity12',
                                                   'CustomEntity09',
                                                   'CustomEntity11',
                                                   'CustomEntity10', 'Step'])

        print('precached')

        self.database = self.progress_query.database

        self.setMinimumSize(500, 800)
        self.setCentralWidget(self.main_widget)

    def start(self):
        self.progress_query.start()

    def fill(self):
        self.main_widget.view = self.database.cached_table('Asset')
        self.main_widget.update_cards()


if __name__ == '__main__':
    # thread = threaded_database_query('sgd', ['Shot', 'Sequence'])
    app = QtWidgets.QApplication(sys.argv)
    window = TestWindow()
    window.show()
    window.start()
    app.exec_()