import os
import ast
import importlib
import logging
from pprint import pprint

from typing import Any
from datetime import datetime

from numpy import ndarray as ndarray
from pandas import Series, DataFrame, to_datetime
import shotgun_api3

import library.core.config_manager as config_manager

import shotgrid_lib.shotgrid_helpers as shotgrid_helpers
import shotgrid_lib.lib.database_exceptions as database_exceptions

importlib.reload(shotgrid_helpers)
logger = logging.getLogger(__name__)
# logger.setLevel(logging.INFO)

ROOT_FOLDER = os.path.dirname(os.path.abspath(__file__))

class DataBase(object):
    """
    Database connection ORM and cache library
    """
    _instance = None
    _project_data = {}
    _last_event_id = None
    def __new__(cls, *args, **kargs):

        if cls._instance is None:
            cls._instance = super(DataBase, cls).__new__(cls)

        return cls._instance

    def fill(self, project, precatch=False, no_prefilter=False):
        """
        to show the documentation

        :param project: The project name to connect
        :param precatch: True if we want to precachhhe the basic elements
        :return: the return value
        """


        if project in self._project_data:
            logger.debug('Database instance already created, updating')
            self.get_events()

            return
        if hasattr(self, 'project') and self.project.lower() != project.lower():
            raise ('The new set project don\'t match the old project new: %s - old :%s ' % (project, self.project))

        self.project = project

        logger.debug('initializing database')

        self.config_manager = config_manager.ConfigSolver(project=self.project)

        self.config_data = self.config_manager.get_config('scheme', module='shotgrid_lib')
        self.connection_data = self.config_manager.get_config('database_connection', module='shotgrid_lib')

        self._query_time = {}
        self._sorter_keys = self.config_data.get('sorter_keys')

        self._precached_entities = []

        if no_prefilter:
            self._pre_filters = {}
            logger.debug('Precaching without prefilters')
        else:
            self._pre_filters = self.config_data.get('pre_filters', {})

        self._all_data = {}
        self._project_data[self.project] = self._all_data

        self._schema = self.config_data['entity_fields']
        self._valid_entities = list(self._schema.keys())
        self._project_filter = ['project.Project.code', 'is', self.project]

        self.sg = self._connect_to_shotgrid()
        self._full_schema = self.sg.schema_read()
        self.precatch(precatch)

        self.set_event_id()

    def _connect_to_shotgrid(self):
        conn_data = self.connection_data['connection']

        connection = shotgun_api3.Shotgun(conn_data['server'],
                                          script_name=conn_data['script_name'],
                                          api_key=conn_data['api_key'])

        logger.debug('Connected to shotgun, start!!!')
        return connection


    def precatch(self, precatch: bool):
        if not precatch:
            return
        for entity in self.config_data.get('pre_cache', []):
            logger.debug('precaching entity: %s' % entity)
            self.query_sg_database(entity, as_precache=True)

    def set_event_id(self):
        order = [{"column": "id", "direction": "desc"}]
        data = self.sg.find_one('EventLogEntry', filters=[], fields=['id'], order=order)
        self._last_event_id = data['id']

    def get_field_type(self, entity_type: str, field: str) -> dict:
        entity_schema = self._full_schema.get(entity_type, {})
        field_config = entity_schema.get(field, {})

        if not field_config and (field.endswith('_name') or field.endswith('_tp')):
            field = field.rsplit(field, 1)[0]
            field_config = entity_schema.get(field, {})

        if field_config:
            field_type = field_config['data_type']['value']
        else:
            raise database_exceptions.FieldNotInEntity(field, entity_type)

        return field_type

    def get_index_from_table(self, entity_type: str, entity_id: int, filters=None, unique=False):
        if entity_type not in self._schema:
            logger.error('The entity type %s is not in the schema' % entity_type)
            raise database_exceptions.EntityNotInShotgrid(entity_type)

        if not isinstance(entity_id, list):
            entity_id = [entity_id]

        entity_id = set(entity_id)

        id_count = len(entity_id)

        if entity_type in self._all_data and len(self._all_data[entity_type]) > 0:
            dataframe = self._all_data[entity_type].loc[self._all_data[entity_type]['id'].isin(entity_id)]
            dataframe_ids = set(self._all_data[entity_type]['id'].to_list())
            cached_count = len(dataframe)
        else:
            dataframe = []
            dataframe_ids = set()
            cached_count = 0

        logger.debug('id count %i %i' % (id_count, cached_count))

        if len(dataframe) != id_count:
            missed_ids = entity_id - dataframe_ids

            logger.debug('Not all the Ids of %s are in the table: %i - %i' % (entity_type, len(dataframe), id_count))
            logger.debug('Quering the missed %i elements' % len(missed_ids))
            self.query_sg_database(entity_type, filters=[['id', 'in', list(missed_ids)]])
            dataframe = self._all_data[entity_type].loc[self._all_data[entity_type]['id'].isin(entity_id)]
        else:
            logger.debug('loaded from already cached table')

        view = View(self, dataframe, entity_type=entity_type)

        return view

    def add_to_serie(self,
                     dataframe,
                     added_dataframe,
                     entity_type):

        if isinstance(added_dataframe, dict):
            added_dataframe = [added_dataframe]

        if len(added_dataframe) == 0:
            return dataframe

        if dataframe.empty:
            as_dict = {}
            if isinstance(added_dataframe, list):

                for key, value in added_dataframe[0].items():
                    as_dict[key] = {}
            else:
                added_dataframe.to_dict()
        else:
            as_dict = dataframe.to_dict()

        date_time_fields = self.config_data.get('datetime_fields')

        if isinstance(added_dataframe, list):

            for item in added_dataframe:
                item_id = item['id']

                for key, value in item.items():

                    if key not in as_dict:
                        as_dict[key] = {}
                    #value = item.get(key, as_dict[key][item_id])

                    if key in date_time_fields:
                        as_dict[key][item_id] = to_datetime(value, utc=True)
                    else:
                        as_dict[key][item_id] = value
        else:
            #as_dict = dataframe.to_dict()
            for key, value_data in added_dataframe.to_dict().items():


                for key_index, value in value_data.items():
                    as_dict[key] = as_dict.get(key, {})
                    as_dict[key][key_index] = value

        all_ids = list(as_dict['id'].keys())

        return DataFrame(as_dict, index=all_ids)

    def query_sg_database(self,
                          entity_type,
                          filters=None,
                          as_precache=False,
                          images=False,
                          update=False):
        logger.debug('query_sg_database: %s, image: %s' % (entity_type, str(images)))
        if entity_type not in  self._all_data or self._all_data[entity_type].empty:
            update = True
        if not images and entity_type in self._precached_entities and update is False:
            logger.debug('The entity %s is already precached, ignoring' % entity_type)
            return
        #if entity_type in self._all_data and update and not as_precache:
        #    self.get_events(entity_type=entity_type)

        start_time = datetime.now()

        if entity_type not in list(self._full_schema.keys()):
            raise database_exceptions.EntityNotInShotgrid(entity_type)

        elif entity_type not in self._schema:
            raise database_exceptions.EntityNotInConfig(entity_type)

        if images:
            query_fields = ['id', 'type', 'image']
        else:
            query_fields = self._schema[entity_type]

        if filters is None:
            filters = self._pre_filters.get(entity_type, [])

        if 'project' in self._schema[entity_type]:
            filters.append(self._project_filter)

        logger.debug('Fields: %s' % ', '.join(query_fields))
        logger.debug('Filters: %s' % str(filters))

        if not entity_type in self._query_time:
            self._query_time[entity_type] = datetime.now()

        start_query_time = datetime.now()
        new_data = self.sg.find(entity_type, filters=filters, fields=query_fields)
        start_process = datetime.now()
        logger.debug('Time quering data: %s Segs' % str(start_query_time - start_query_time))

        self._all_data[entity_type] = self.add_to_serie(self._all_data.get(entity_type, emptyView()), new_data,
                                                        entity_type)
        new_output = self.add_to_serie( emptyView(), new_data, entity_type)

        if as_precache:
            self._precached_entities.append(entity_type)

        end_time = datetime.now()

        logger.debug('Time processing data: %s Segs' % str(end_time - start_process))
        logger.debug('Time to query: %s, return %i items' % (str(end_time - start_time), len(new_data)))

        return View(self, new_output, entity_type, unique=True)


    def update_all(self):
        for entity_type in self._all_data:
            if entity_type in self._precached_entities:
                full_update = True
            else:
                full_update = False
            self.update_table(entity_type, full_update=full_update)

    def update_table(self, entity_type: str, full_update=False):
        if entity_type not in self._all_data or len(self._all_data[entity_type]) == 0:
            return

        filters = [['updated_at', 'greater_than', self._query_time[entity_type]]]

        if not full_update:
            filters.append(['id', 'in', self[entity_type].id])

        self.query_sg_database(entity_type, filters=filters, update=True)
        end_time = datetime.now()
        self._query_time[entity_type] = end_time

    def entity_view(self, entity: str):
        views = View(self, self._all_data[entity], entity)
        return views

    def __getitem__(self, entity_type):
        if entity_type not in list(self._full_schema.keys()):
            raise database_exceptions.EntityNotInShotgrid(entity_type)
        elif entity_type not in self._valid_entities:
            raise database_exceptions.EntityNotInConfig(entity_type)

        if entity_type not in self._all_data:
            return None

        views = self.entity_view(entity_type)
        return views

    def append(self, record):
        new_data = record.as_shotgun()
        entity_type = record.entity_type

        filtered_data = {}
        for key in self._schema[entity_type]:
            if key in new_data:
                filtered_data[key] = new_data[key]
            else:
                filtered_data[key] = None
        pprint(new_data)
        new_sg_record = self.sg.create(entity_type, new_data)

        index = new_sg_record['id']
        filtered_data['type'] = entity_type
        filtered_data['id'] = index
        data_frame = DataFrame([filtered_data], index=[index])
        self._all_data[entity_type] = self.add_to_serie(self._all_data.get(entity_type, emptyView()), data_frame, entity_type)

        return View(self, data_frame, entity_type, unique=True,)

    def get_events(self, entity_type=None):
        start_time = datetime.now()

        logger.info('Get events')
        entities = list(self._all_data.keys())

        valid_events = []
        for entity in entities:
            valid_events.append('Shotgun_%s_Change' % entity)
            valid_events.append('Shotgun_%s_New' % entity)
            valid_events.append('Shotgun_%s_Retirement' % entity)
            valid_events.append('Shotgun_%s_Revival' % entity)
        if not valid_events:
            return

        filters = [self._project_filter,
                   ['id', 'greater_than', self._last_event_id],
                   ['event_type', 'in', valid_events]]

        fields = ['id', 'event_type', 'attribute_name', 'meta']
        data = self.sg.find('EventLogEntry', filters=filters, fields=fields)
        if data:
            self.process_events(data, entity_type=entity_type)
        else:
            logger.debug('No events to process')

        end_time = datetime.now()
        logger.debug('event processing %s events time:  %s' % (len(data), end_time - start_time))

    def process_events(self, event_list, entity_type=None):
        events_by_type = {}
        ids = []
        for event_data in event_list:
            event_type = event_data['event_type']
            ids.append(event_data['id'])
            _, entity, action = event_type.split('_')
            if entity_type and entity != entity_type:
                continue
            events_by_type[action] = events_by_type.get(action, {})
            events_by_type[action][entity] = events_by_type[action].get(entity, [])
            events_by_type[action][entity].append(event_data)

        self.process_changed_attribute(events_by_type.get('Change', {}))

        self.process_new_attribute(events_by_type.get('New', {}))

        self._last_event_id = max(ids)

    def process_new_attribute(self, event_list):
        if not event_list:
            return
        for entity, entity_change_list in event_list.items():
            id_list = []
            for entity_changes in entity_change_list:
                id_list.append(entity_changes['meta']['entity_id'])
            filters = [['id', 'in', id_list]]
            self.query_sg_database(entity, filters=filters)

    def process_changed_attribute(self, event_list):
        if not event_list:
            return

        logger.debug('Processing changed attribute')
        for entity, entity_change_list in event_list.items():
            all_ids = [entiy['meta']['entity_id'] for entiy in  entity_change_list]

            self.query_sg_database(entity, filters=[['id', 'in', all_ids]], update=True)
            for entity_changes in entity_change_list:
                attribute_name = entity_changes['attribute_name']
                if attribute_name not in self._all_data[entity].keys():
                    continue

                changed_id = entity_changes['meta']['entity_id']

                if 'new_value' in entity_changes['meta']:
                    new_value = entity_changes['meta']['new_value']
                    self._all_data[entity].at[changed_id, attribute_name] = new_value
                else:
                    if entity_changes['meta'].get('added'):
                        added_value = entity_changes['meta']['added']
                        old_value = self._all_data[entity].at[changed_id, attribute_name]
                        if isinstance(old_value, list):
                            new_value = old_value + added_value
                        else:
                            new_value = added_value
                        self._all_data[entity].at[changed_id, attribute_name] = new_value

                    if entity_changes['meta'].get('removed'):
                        removed_value = entity_changes['meta']['added']
                        old_value = self._all_data[entity].at[changed_id, attribute_name]
                        if isinstance(old_value, list):
                            new_value = []
                            for item in old_value:
                                for removed in removed_value:
                                    if item['type'] != removed['type'] or item['id'] != removed['id']:
                                        new_value.append(item)
                            self._all_data[entity].at[changed_id, attribute_name] = new_value


class Record(object):
    def __init__(self, entity_type):
        object.__init__(self)
        self.entity_type = entity_type
        self.stored = False
        self._database = DataBase()
        self.schema = self._database._full_schema
        self._record_data = {key: None for key in self.schema[entity_type].keys()}

    def __setattr__(self, __name: str, __value: Any) -> None:
        if hasattr(self, '_record_data') and __name in self._record_data:
            self._record_data[__name] = __value
        else:
            super(Record, self).__setattr__(__name, __value)

    def get_shotgun_values(self, value):

        if isinstance(value, View):
            solved_value = value.as_shotgun()
        elif isinstance(value, list):
            solved_value = []
            for var in value:
                solved_value.append(self.get_shotgun_values(var))
        else:
            solved_value = value
        return solved_value

    def solve_pattern(self, pattern, vars_dict={}):

        solve_vars = self._record_data.copy()
        for key, value in vars_dict.items():
            solve_vars[key] = value

        if hasattr(self, 'sg_asset_type'):
            solve_vars['sg_asset_type'] = getattr(self, 'sg_asset_type')

        solved_pattern = shotgrid_helpers.solve_pattern(pattern, solve_vars)
        return solved_pattern

    def generate_publish_hash(self, publish_type):
        hash_fields = publish_type.sg_hash_fields
        print(hash_fields)
        hash = shotgrid_helpers.generate_hash(hash_fields, self.as_shotgun())
        print(hash)
        return hash

    def as_shotgun(self):
        values = {}
        for key, value in self._record_data.items():
            if value == None:
                continue
            values[key] = self.get_shotgun_values(value)
        return values


class View(object):
    def __init__(self, project_database, serie, entity_type, unique=False):
        object.__init__(self)
        self._modified = {}

        self._database = project_database
        self._filled = False
        self._serie = serie

        self._serie_size = len(serie)

        self._iterator_index = None
        self._entity_type = entity_type
        if len(serie) > 0:
            self._keys = list(serie.keys())
        else:
            self._keys = []

        self._unique = unique

    @property
    def empty(self):
        return self._serie_size == 0

    @property
    def unique(self):
        return self._unique

    def append(self, record):
        new_row = self._database.append(record)
        self._serie = self._database.add_to_serie(self._serie, new_row, self._entity_type)
        self._serie_size = self._serie.count().iloc[0]

    def find_with_filters(self, single_item=False, **kwargs):
        if self.empty:
            return self
        item = self
        serie = item._serie
        if len(serie) == 0:
            return self
        for key, value in kwargs.items():
            if value is None:
                continue
            if isinstance(value, emptyView):
                raise database_exceptions.ViewIsEmpty(value.type)
            serie = item._find_with_value(serie, key, value)

        result = View(self._database, serie, self._entity_type, unique=False)

        if single_item and not serie.empty:
            return max(result)

        return result

    def _comp_type(self, row, value):

        if isinstance(row, dict):
            key, val = list(value.items())[0]
            return row.get(key, None) == val

        if isinstance(row, list):
            for row_item in row:
                if row_item == value:
                    return True

    def _find_with_value(self, serie, field, value, single_item=False):
        if field not in self._keys:
            raise database_exceptions.FieldNotInEntity(field, self._entity_type)

        if isinstance(value, View):
            value = value.as_shotgun()
        field_type = self._database.get_field_type(self._entity_type, field)
        if field_type == 'multi_entity':
            table = serie.loc[self._serie[field].apply(self._comp_type, value=value) == True]

        elif isinstance(value, list):
            table = serie.loc[self._serie[field].isin(value)]
        else:

            table = serie.loc[self._serie[field] == value]

        return table

    def get_options(self, field: str) -> list:
        field_schema = self._database.sg.schema_field_read(self._entity_type, field)
        values = list(field_schema[field]['properties']['display_values']['value'].keys())

        return values

    def get_field_value(self, field: str, unique=False):
        if self.empty:
            field_type = self._database.get_field_type(field)
            if field_type == 'entity' or field_type == 'multi_entity':
                return emptyView()
            elif field_type == 'text' or field_type == 'list' or field_type == 'status_list':
                return ''
            elif field_type == 'number':
                return 0
            elif field_type == 'float':
                return 0.0
            else:
                return None

        schema = self._database._schema[self._entity_type]
        if field not in schema:
            raise database_exceptions.FieldNotInEntity(field, self._entity_type)

        value = getattr(self, field)

        if isinstance(value, View) and not value.empty and unique:
            value._unique = True

        return value

    def single_values(self, field: str) -> list:

        data_type = self._database.get_field_type(self._entity_type, field)

        all_items = []

        if data_type == 'entity':
            for it in self:
                value = it.get_field_value(field, unique=True)
                if not value.empty:
                    all_items.append(value.named_field)

        elif data_type == 'multi_entity':
            for it in self:
                entities = getattr(it, field)
                if entities is None:
                    continue
                for entity in entities:
                    all_items.append(entity.named_field)

        else:
            all_items = getattr(self, field)
            if isinstance(all_items, str):
                all_items = []
        all_items = list(set(all_items))
        return all_items

    def update_shotgun(self):
        logger.debug('Updating shotgun entity: %s id %s , data: %s' % (self._entity_type, int(self.id), self._modified))
        self._database.sg.update(self._entity_type, int(self.id), self._modified)

    @property
    def named_field(self):
        field = self.get_name_field(self)
        value = self.get_field_value(field)
        return value

    def get_name_field(self, view):
        if'code' in self._keys:
            field = 'code'
        elif 'content'  in self._keys:
            field = 'content'
        elif 'name'  in self._keys:
            field = 'name'
        else:
            field = 'code'

        return field

    def __add__(self, other):

        if isinstance(other, View) or isinstance(other, emptyView):
            second_serie = other._serie
        elif isinstance(other, Series) or isinstance(other, dict):
            second_serie = other
        else:
            return self

        first_serie = self._serie
        new_serie = self._database.add_to_serie(first_serie, second_serie, self._entity_type)
        new_view = View(self._database, new_serie, self._entity_type, unique=False)
        return new_view

    def __iadd__(self, other):
        if isinstance(other, View) or isinstance(other, emptyView):
            second_serie = other._serie
        elif isinstance(other, Series) or isinstance(other, dict):
            second_serie = other
        else:
            return self

        first_serie = self._serie
        new_serie = self._database.add_to_serie(first_serie, second_serie, self._entity_type)
        self._serie = new_serie
        self._serie_size = len(new_serie)
        if len(new_serie) > 1:
            self._unique = False
        return self

    def __setattr__(self, __name: str, __value: Any) -> None:
        if __name[0] == '_':
            return super(View, self).__setattr__(__name, __value)

        if not self._unique:
            return

        if isinstance(__value, View):
            self._modified[__name] = __value.as_shotgun()
            self._serie.iloc[0, self._serie.columns.get_loc(__name)] = __value.as_shotgun()

        elif isinstance(__value, list):
            new_values = []
            for val in __value:
                if isinstance(val, View):
                    new_values.append(val.as_shotgun())
                else:
                    new_values.append(val)
            self._modified[__name] = new_values
            self._serie.iloc[0, self._serie.columns.get_loc(__name)] = new_values

        else:
            self._modified[__name] = __value
            self._serie.iloc[0, self._serie.columns.get_loc(__name)] = __value


    def __getitem__(self, value):
        field = self.get_name_field(value)
        table = self._serie.loc[self._serie[field] == value]
        unique_view = View(self._database, table, self._entity_type, unique=True)
        if unique_view._serie.empty:
            return emptyView()
        return unique_view

    def __bool__(self):
        return not self.empty

    def __len__(self):
        return len(self._serie)

    def __nonzero__(self):
        return not self._serie.empty

    def __contains__(self, value):

        field = self.get_name_field(self)
        if isinstance(value, View):
            if value._entity_type != self._entity_type:
                raise database_exceptions.ViewIsDifferentEntity(value._entity_type, self._entity_type)
            value = value.named_field

        if isinstance(value, list):
            if value and isinstance(value[0], View):
                str_values = []
                for val in value:
                    str_value = val.named_field
                    if isinstance(str_value, list):
                        str_values += str_value
                    else:
                        str_values.append(str_value)
                value = str_values

            table = self._serie.loc[self._serie[field].isin(value)]
            return len(table) == len(value)

        else:
            if (isinstance(value, View) or isinstance(value, emptyView)) and value.empty:
                raise database_exceptions.ViewIsEmpty(value._entity_type)

            table = self._serie.loc[self._serie[field] == value]

        return not table.empty

    def _unique_copy(self):
        table = self._serie.iloc[[self._iterator_index]]
        unique_view = View(self._database, table, self._entity_type, unique=True)
        return unique_view

    def __str__(self):
        strings = []

        strings.append('Entity type: %s' % self._entity_type)
        for index, item in self._serie.iterrows():
            strings.append('-' * 50)
            for key, value in item.items():
                string = '> %s : ' % key
                if str(value).find('https://') == -1:
                    string += '%s' % str(value)
                else:
                    string += 'web link'
                strings.append(string)

        return '\n'.join(strings)

    def __iter__(self):
        self._iterator_index = 0
        return self

    def __next__(self):
        if self._iterator_index < len(self._serie):
            unique_view = self._unique_copy()
            unique_view._unique = True
            self._iterator_index += 1
            return unique_view
        else:
            self._iterator_index = None
            raise StopIteration

    def as_shotgun(self):
        data = []
        for d in self:
            data.append({'id': int(d.id), 'name': d.named_field, 'type': str(d.type)})

        if self._unique and data:
            return data[0]

        return data

    def as_dict(self):
        return self._serie.to_dict()

    def as_pandas(self):
        return self._serie

    def __eq__(self, other):
        if not isinstance(other, View):
            return False

        if self.id == other.id and self.type == other.type:
            return True
        return False

    def __gt__(self, other):

        if self._database._sorter_keys.get(self._entity_type):
            this_value = getattr(self, self._database._sorter_keys.get(self._entity_type))
            other_value = getattr(other, self._database._sorter_keys.get(self._entity_type))
            return this_value > other_value

        elif hasattr(self, 'content'):
            return self.content > other.content

        elif hasattr(self, 'name'):
            return self.name > other.name
        elif hasattr(self, 'firstname'):
            return self.firstname > other.firstname

        else:
            return self.code > other.code

    def _is_list_empty(self, list_to_check):
        if list_to_check is None:
            return True

        if not isinstance(list_to_check, list) and not isinstance(list_to_check, ndarray):
            return False

        if len(list_to_check) == 0:
            return True
        else:
            empty = True
            for nested_list in list_to_check:
                empty = empty and self._is_list_empty(nested_list)
            return empty

    def __getattribute__(self, field):
        if field == '__class__':
            return View
        if field[0] == '_':
            return super(View, self).__getattribute__(field)

        string_dictionaries = ['sg_files', 'sg_metadata']

        #logger.debug('field name: %s' % field)

        only_name = False
        only_type = False
        if field.endswith('_name'):
            only_name = True
            field = field.rsplit('_', 1)[0]

        if field.endswith('_tp'):
            only_type = True
            field = field.rsplit('_', 1)[0]

        if not field in self._keys:
            return object.__getattribute__(self, field)

        if field == 'type':
            var_type = 'text'
        else:
            var_type = self._database.get_field_type(self._entity_type, field)

        record = self._serie.iloc[:][field].values
        if self._is_list_empty(record):
            if var_type == 'text' or var_type == 'list' or var_type == 'status_list':
                return ''

            return emptyView()

        if var_type == 'date':
            # date_day_format = '%Y/%m/%d %I:%M %p'
            date_format = '%Y-%m-%d'
            output = []
            for item in record:
                if str(item) == 'NaT':
                    output.append(None)
                else:
                    string = str(item).split('T')[0]
                    output.append(datetime.strptime(string, date_format))

        elif var_type == 'date_time':
            date_day_format = '%Y-%m-%dT%H:%M:%S'
            output = []
            for item in record:
                if str(item) == 'NaT':
                    output.append(None)
                else:
                    time_str = str(item).split('.')[0]
                    output.append(datetime.strptime(time_str, date_day_format))

        elif var_type == 'text' and field in string_dictionaries:
            output = []
            for item in record:
                if not item:
                    output.append({})
                else:
                    try:
                        output.append(ast.literal_eval(item))
                    except:
                        output.append(item)

        elif var_type == 'text' or var_type == 'list' or var_type == 'status_list':
            output = [str(item) for item in record]
        elif var_type == 'float':
            output = []
            for out in record:
                try:
                    output.append(float(out))
                except:
                    output.append(0)

        elif var_type == 'number':
            output = []
            for out in record:
                try:
                    output.append(int(out))
                except:
                    output.append(0)

        elif var_type == 'entity':
            linked = LinkedView(self._database, record, only_name=only_name, only_type=only_type, unique=self._unique)
            output = linked.get_data()

        elif var_type == 'multi_entity':
            linked = LinkedView(self._database, record, only_name=only_name, only_type=only_type, unique=self._unique)
            output = linked.get_data()


        elif var_type == 'entity_type' or var_type == 'image':
            output = [str(item) for item in record]
        else:
            output = record

        if self._unique and isinstance(output, list) and len(output) > 0:
            output = output[0]
        return output

    def precache_dependencies(self, fields=[]):

        if len(self._serie) == 0:
            return
        record = self._serie.to_dict()

        for field in self._database.config_data['entity_fields'].get(self._entity_type, {}):
            if fields and field not in fields:
                continue
            if field == 'type':
                continue

            field_type = self._database.get_field_type(self._entity_type, field)
            entity_types = []

            if field_type == 'entity' or field_type == 'multi_entity':
                dependencies_id = {}
                entity_type = None
                for values in record[field].values():
                    if isinstance(values, dict):
                        dependencies_id[values['type']] = dependencies_id.get(values['type'], [])
                        dependencies_id[values['type']].append(values['id'])


                    elif isinstance(values, list):
                        for value in values:
                            dependencies_id[value['type']] = dependencies_id.get(value['type'], [])
                            dependencies_id[value['type']].append(value['id'])

                if not dependencies_id:
                    continue

                for entity_type, id_list in dependencies_id.items():
                    self._database.get_index_from_table(entity_type, id_list, unique=False)

    def get_thumbnails(self):

        if self.empty:
            return self

        if self.unique:
            ids = self.get_field_value('id')
            ids = [ids]

        else:
            ids = self.single_values('id')

        self_dic = self.as_dict()
        current_ids = self_dic.get('image', {})

        clean_current = []

        for c_id in current_ids:
            value = self_dic['image'][c_id]
            if not isinstance(value, float):
                clean_current.append(c_id)

        clean_current = set(clean_current)

        query_ids = set(ids) - clean_current
        if len(query_ids) == 0:
            return self

        query_ids = list(query_ids)
        filters = [['id', 'in', query_ids]]
        new_view = self._database.query_sg_database(self._entity_type, filters=filters, images=True)

        new_dict = new_view.as_dict()

        final_dict = {}

        for field, id_value in self_dic.items():
            final_dict[field] = {}
            for c_id, value in id_value.items():
                if c_id not in ids:
                    continue
                final_dict[field][c_id] = value

        for field, id_value in new_dict.items():
            final_dict[field] = final_dict.get(field, {})
            for c_id, value in id_value.items():
                final_dict[field][c_id] = value

        dataframe = DataFrame(final_dict, index=ids)

        return View(self._database, dataframe, self._entity_type, self._unique)


class LinkedView(object):
    def __init__(self, database, record, only_name=False, only_type=False, unique=False):
        self.unique = unique
        self._database = database
        self.only_name = only_name
        self.only_type = only_type
        self.record = record
        self._iterator_index = None

        self.linked_entities = {}
        self.all_data = []
        if isinstance(record[0], dict) or record[0] is None:
            record = [record]

        all_ids = {}
        grouped_ids = []
        name_output = []
        type_output = []
        entity_type = None
        for row in record:
            for linked_entity in row:
                if linked_entity is None:
                    continue
                name_output.append(linked_entity['name'])
                type_output.append(linked_entity['type'])

                all_ids[linked_entity['type']] = all_ids.get(linked_entity['type'], [])
                all_ids[linked_entity['type']].append(linked_entity['id'])

        if only_name:
            self.linked_entities['all'] = name_output
            self.all_data = name_output
        elif only_type:
            self.linked_entities['all'] = type_output
            self.all_data = type_output
        else:
            for entity_type, ids in all_ids.items():
                linked_entities = self.get_linked_entities(entity_type, ids)
                self.linked_entities[entity_type] = linked_entities
                for item in linked_entities:
                    self.all_data.append(item)

    def get_data(self):
        if len(self.linked_entities.keys()) == 1:
            output = list(self.linked_entities.values())[0]
            if isinstance(output, View) and self.unique and len(output) == 1:
                output._unique = True
            elif isinstance(output, list) and self.unique and len(output) == 1:
                output = output[0]
            return output

    def __str__(self):
        strings = []
        if len(self.linked_entities.keys()) > 1:
            strings.append('Multi type view of elemens of types: %s' % ', '.join(self.linked_entities.keys()))
        else:
            strings.append('Single type viee of elemens of types: %s' % ', '.join(self.linked_entities.keys()))

        strings.append('Total count of items: %s' % len(self.all_data))

        for entity_type, item_list in self.linked_entities.items():
            strings.append('entity: %s' % entity_type)
            strings.append(str(item_list))

        return '\n'.join(strings)

    def __len__(self):
        return len(self.all_data)

    def __iter__(self):
        self._iterator_index = 0
        return self

    def __next__(self):
        if self.only_name or self.only_type:
            name = self.all_data[self._iterator_index]
            self._iterator_index += 1
            return name

        if self._iterator_index < len(self.all_data):
            unique_view = self.all_data[self._iterator_index]
            unique_view._unique = True
            self._iterator_index += 1
            return unique_view
        else:
            self._iterator_index = None
            raise StopIteration

    def get_linked_entities(self, entity_type, entities_ids, unique=False):

        logger.debug('get linked entities %s' % entity_type)
        linked_view = self._database.get_index_from_table(entity_type, entities_ids, unique=False)

        if unique:
            table = linked_view._serie.iloc[[0]]
            unique_view = View(self._database, table, entity_type, unique=True)
            return unique_view

        return linked_view


class emptyView(object):
    def __init__(self) -> None:
        pass

    def __getattribute__(self, obj):
        if obj == 'empty':
            return True
        return emptyView()

    def __iter__(self):
        return self

    def __next__(self):
        raise StopIteration

    def __str__(self) -> str:
        return 'Empty view'

    def __len__(self):
        return 0

