import unittest

import logging

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

import datetime

import shotgrid_lib.database as database
import shotgrid_lib.lib.database_exceptions as database_exceptions


class BasicFunctions(unittest.TestCase):
    def setUp(self):
        self.database = database.DataBase()
        self.database.fill('dem', precatch=True)
        self.shot_name = 'bunny_080_0190'
        self.second_shot_name = 'bunny_080_0200'
        self.no_shot_name = 'bunny_080_invalid'
        self.first_version = 'bunny_080_0190_layout_v000'
        self.last_version = 'bunny_080_0190_layout_v001'

    def log_start(self, function_name):
        print('\n')
        logger.info('==  %s  ==' % function_name)
        logger.info('=' * (len(function_name) + 8))

    def test_aa_fill(self):
        self.log_start('test_fill')

        self.config_data = self.database.config_data

        for entity_type in self.config_data.get('pre_cache'):
            self.assertEqual(entity_type in self.database._all_data, True, '%s should be loaded ' % entity_type)
            serie = self.database._all_data[entity_type]
            logger.info('%s %i' % (entity_type, len(serie)))
        logger.info('pass')

    def test_query_fields(self):
        self.log_start('test_query_fields')

        shots = self.database['Shot']
        self.assertEqual(isinstance(shots, database.View), True, 'shots should be an instance of View')
        self.assertEqual(len(shots) > 0, True, 'The size of the shot view should be greather than 0')

        shot = self.database['Shot'][self.shot_name]

        self.assertEqual(isinstance(shot.code, str), True, 'The code field should be an instance of str')
        self.assertEqual(isinstance(shot.sg_cut_in, int), True, 'The sg_cut_in field should be an instance of int')

        self.assertEqual(isinstance(shot.sg_cut_duration, int), True,
                         'The sg_cut_duration field should be an instance of int')

        self.assertEqual(isinstance(shot.sg_sequence, database.View), True,
                         'The sequence fiend should be an instance of View')
        self.assertEqual(isinstance(shot.sg_sequence_name, str), True,
                         'The sequence_name fiend should be an instance of str')

        self.assertEqual(shot.sg_sequence._unique, True, 'The sequence view should not be unique')

        self.assertEqual(isinstance(shot.tasks, database.View), True, 'The tasks fiend should be an instance of View')
        self.assertEqual(shot.tasks._unique, False, 'The tasks view should be unique')

        self.assertEqual(isinstance(shot.created_at, datetime.datetime), True,
                         'The created_at fiend should be an instance of datetime')

        logger.info('pass')

    def test_add(self):
        self.log_start('test_add')

        first_shot = self.database['Shot'][self.shot_name]
        second_shot = self.database['Shot'][self.second_shot_name]

        sequence = first_shot + second_shot

        self.assertEqual(isinstance(sequence, database.View), True, 'sequence should be an instance of View')
        self.assertEqual(len(sequence) == 2, True, 'The size of the sequence var should be 2')
        self.assertEqual(sequence._unique, False, 'The sequence var should not be unique')

        add_itself = first_shot + first_shot

        self.assertEqual(isinstance(add_itself, database.View), True, 'sequence should be an instance of View')
        self.assertEqual(len(add_itself) == 1, True, 'The size of the sequence var should be 2')
        self.assertEqual(add_itself._unique, False, 'The sequence var should not be unique')

        logger.info('pass')

    def test_iadd(self):
        self.log_start('test_iadd')

        first_shot = self.database['Shot'][self.shot_name]
        second_shot = self.database['Shot'][self.second_shot_name]
        first_shot += second_shot
        self.assertEqual(isinstance(first_shot, database.View), True, 'first_shot should be an instance of View')
        self.assertEqual(len(first_shot) == 2, True, 'The size of the first_shot var should be 2')
        self.assertEqual(first_shot._unique, False, 'The first_shot var should not be unique')
        logger.info('pass')

    def test_iterator(self):
        self.log_start('test_iterator')

        shot = self.database['Shot'][self.shot_name]

        count = 0
        items = len(shot.sg_breakdowns)
        for breakdown_view in shot.sg_breakdowns:
            count += 1
        self.assertEqual(count, items, 'the count after the loop should be the same than the total number of items')

        logger.info('pass')

    def test_empty(self):
        self.log_start('test_empty')

        first_shot = self.database['Shot'][self.shot_name]
        second_shot = self.database['Shot'][self.no_shot_name]

        self.assertEqual(first_shot.empty, False, 'the empty value of first shot should be False')
        self.assertEqual(second_shot.empty, True, 'the empty value of second shot should be True')

        logger.info('pass')

    def test_bool(self):
        self.log_start('test_bool')

        first_shot = self.database['Shot'][self.shot_name]
        second_shot = self.database['Shot'][self.no_shot_name]

        self.assertEqual(bool(first_shot), True, 'the bool value of first shot should be True')
        self.assertEqual(bool(second_shot), False, 'the empty value of second shot should be False')
        logger.info('pass')

    def test_contains(self):
        self.log_start('test_contains')

        all_shots = self.database['Shot']
        first_shot = self.database['Shot'][self.shot_name]

        second_shot = self.database['Shot'][self.second_shot_name]

        no_valid_shot = self.database['Shot'][self.no_shot_name]
        multi_shots = [self.shot_name, self.second_shot_name]

        multi_shot_views = [first_shot, second_shot]

        self.assertEqual(self.shot_name in all_shots, True, 'the value of first shot should be True')
        self.assertEqual(first_shot in all_shots, True, 'the value of first shot should be True')
        self.assertEqual(self.no_shot_name in all_shots, False, 'the value of second shot should be False')

        try:
            a = no_valid_shot in all_shots
        except database_exceptions.ViewIsEmpty:
            logger.warning('Correct exception manage of type ViewIsEmpty')
        else:
            self.assertEqual(True, False, 'This should have been manage by a ViewIsEmpty exception')

        self.assertEqual(multi_shots in all_shots, True, 'The value of multi query shot should be True')
        self.assertEqual(multi_shot_views in all_shots, True, 'The value of multi query shot should be True')

        logger.info('pass')

    def test_greater(self):
        self.log_start('test_greater')

        first_shot = self.database['Shot'][self.shot_name]
        second_shot = self.database['Shot'][self.second_shot_name]

        versions = first_shot.sg_versions
        print(versions.code)
        first_version = versions[self.first_version]
        last_version = versions[self.last_version]

        self.assertEqual(second_shot > first_shot, True, 'The second shot is greater than first, by name')
        self.assertEqual(second_shot < first_shot, False, 'The second shot is greater than first, by name')

        self.assertEqual(first_version < last_version, True,
                         'The last version is greater than first, by version number')
        self.assertEqual(first_version > last_version, False,
                         'The first version is not greater than latest, by version number')

    def test_find_with_filter(self):
        self.log_start('test_find_with_filter')

        self.database.query_sg_database('Task')
        first_shot = self.database['Shot'][self.shot_name]

        task_view = self.database['Task'].find_with_filters(content='Comp',
                                                            entity=first_shot,
                                                            single_item=True)

        layout = first_shot.sg_versions.find_with_filters(sg_task=task_view,
                                                          single_item=False)

        self.assertEqual(isinstance(layout, database.View), True, 'The layout_approved should be a View instance')
        self.assertEqual(layout.unique, False, 'The layout_approved should not be a unique instance')

        latest_version = max(layout)

        self.assertEqual(isinstance(latest_version, database.View), True,
                         'The latest version should be a View instance')
        self.assertEqual(latest_version.empty, False, 'The latest version should not be empty')

        latest_layout = first_shot.sg_versions.find_with_filters(sg_task=task_view,
                                                                 single_item=True)

        self.assertEqual(isinstance(latest_layout, database.View), True,
                         'The latest_layout_approved should be a View instance')
        self.assertEqual(latest_layout.unique, True, 'The latest_layout_approved should be a unique instance')

    def test_get_options(self):
        self.log_start('test_get_options')

        data = self.database['Shot']

        options = data.get_options('sg_status_list')
        self.assertEqual(isinstance(options, list), True, 'The layout_approved should be a View instance')
        default_options = ['fin', 'omt', 'na', 'wtg', 'ip']
        default_options = ['ip', 'wtg', 'fin']
        self.assertEqual(default_options, options, 'The layout_approved should be a View instance')

    def test_single_values(self):
        self.log_start('test_single_values')

        data = self.database['Shot']
        sequences = data.single_values('sg_sequence')
        self.assertEqual(True, isinstance(sequences, list), 'The output should be a list')

    def test_as_shotgrid(self):
        self.log_start('test_as_shotgrid')
        shot_data = self.database['Shot'][self.shot_name]
        shot_as_shotgrid = shot_data.as_shotgun()
        self.assertEqual(isinstance(shot_as_shotgrid, dict), True, 'The output should be a dictionary')

        all_shots_as_shotgrid = self.database['Shot'].as_shotgun()
        self.assertEqual(isinstance(all_shots_as_shotgrid, list), True, 'The output should be a list')


class ErrorFunctions(unittest.TestCase):
    def setUp(self):
        self.database = database.DataBase()
        self.database.fill('dem', precatch=True)
        self.shot_name = 'bunny_080_0190'
        self.second_shot_name = 'bunny_080_0200'
        self.no_shot_name = 'bunny_080_invalid'
        self.first_version = 'bunny_080_0190_layout_v000'
        self.last_version = 'bunny_080_0190_layout_v001'

    def log_start(self, function_name):
        print('\n')
        logger.info('==  %s  ==' % function_name)
        logger.info('=' * (len(function_name) + 8))

    def test_bad_entity(self):
        self.log_start('test_bad_entity')
        try:
            data = self.database['Test']
        except database_exceptions.EntityNotInShotgrid:
            logger.warning('Raised a EntityNotInShotgrid exception as expected')
        else:
            self.assertEqual(False, True, 'The exception launched should be of type EntityNotInShotgrid')

    def test_no_config_entity(self):
        self.log_start('test_no_config_entity')
        try:
            data = self.database['Software']
        except database_exceptions.EntityNotInConfig:
            logger.warning('Raised a EntityNotInConfig exception as expected')
        else:
            self.assertEqual(False, True, 'The exception launched should be of type EntityNotInConfig')

    def test_bad_field(self):
        self.log_start('test_bad_field')

        shot = self.database['Shot'][self.shot_name]

        try:
            test = shot.get_field_value('test')
        except database_exceptions.FieldNotInEntity:
            logger.warning('Raised a FieldNotInEntity exception as expected')
        else:
            self.assertEqual(False, True, 'The exception launched should be of type EntityNotInConfig')

    def test_empty_filter(self):
        self.log_start('test_empty_filter')

        step_view = self.database['Step']['no_valid']
        first_shot = self.database['Shot'][self.shot_name]
        try:
            tasks = first_shot.tasks.find_with_filters(step=step_view)
        except database_exceptions.ViewIsEmpty:
            logger.warning('Raised a ViewIsEmpty exception as expected')
        else:
            self.assertEqual(False, True, 'The exception launched should be of type ViewIsEmpty')


if __name__ == '__main__':
    unittest.main(verbosity=0)