import ast
import importlib

from PySide import QtWidgets, QtCore, QtGui

import library.core.config_manager as config_manager
import library.ui.labeled as labeled
import shotgrid_lib.database as database


class BaseCard(QtWidgets.QFrame):
    clicked = QtCore.Signal(int)
    layouts_data = {}
    project = ''
    config_manager = None

    def __init__(self, view, parent=None, add_thumbnail=True, arrange='vertical'):
        super(BaseCard, self).__init__(parent=parent)
        self.view = view
        self.database = self.view._database
        self.setAcceptDrops(True)

        # self.setFixedHeight(100)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.Fixed)
        self.main_layout = QtWidgets.QHBoxLayout()
        self.setLayout(self.main_layout)
        self.add_thumbnail = add_thumbnail

        if self.add_thumbnail:
            self.add_thumbnail_widget(self.main_layout)

        if arrange == 'vertical':
            self.data_layout = QtWidgets.QVBoxLayout()
        else:
            self.data_layout = QtWidgets.QHBoxLayout()

        self.main_layout.addLayout(self.data_layout)

    def add_thumbnail_widget(self, layout):
        self.thumbnail_widget = labeled.ThumbnailWidget(self.view, 'Thumbnail')
        layout.addWidget(self.thumbnail_widget)

    @classmethod
    def config_card(cls):
        pass


class DataCard(QtWidgets.QFrame):
    clicked = QtCore.Signal(int)
    layouts_data = {}
    project = ''
    config_manager = None

    def __init__(self, view, parent=None):
        super(DataCard, self).__init__(parent=parent)
        self.view = view
        self.setAcceptDrops(True)

        self.setFixedHeight(100)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.Fixed)

        self.layout_config = self.layouts_data.get(self.view.type, {})
        self.widget_list = []
        self.main_layout = self.build_layout(self.layout_config)
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(4, 4, 4, 4)
        self.setLayout(self.main_layout)

    def add_mime_data(self, format, string, mime_item):
        app_prefix = 'application/shotgrid'
        encode_string = QtCore.QByteArray(str(string).encode())

        mime_format = '%s/%s' % (app_prefix, format)
        mime_item.setData(mime_format, encode_string)

    def mouseMoveEvent(self, e):

        if e.buttons() == QtCore.Qt.LeftButton:
            drag = QtGui.QDrag(self)
            mime = QtCore.QMimeData()
            mime.setText(self.view.code)

            for published in self.view.sg_published_elements:
                if published.sg_publish_type_name == 'model' and published.sg_status_list == 'cmpt':
                    files = ast.literal_eval(published.sg_files)

                    if 'abc_high' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc_high'])
                    elif 'abc' in files:
                        full_path = '%s/%s' % (published.sg_published_folder, files['abc'])

                    full_path = r'D:\work\atlantis\mock_server\projects\sgd\publish\model\LeoMesi\ModelingLow\LeoMesi_Low_4f1bb89bcf9fc3dd\v021\abc\LeoMesi_high.abc'

                    full_path = full_path.replace('\\', '/')
                    mime.setUrls([full_path])
                    self.add_mime_data('code', published.code, mime)
                    self.add_mime_data('version_number', str(published.sg_version_number), mime)
                    self.add_mime_data('asset_name', published.sg_asset_name, mime)
                    self.add_mime_data('step', published.sg_step, mime)
                    self.add_mime_data('variant', published.sg_variant, mime)
                    self.add_mime_data('hash', published.sg_hash, mime)
                    self.add_mime_data('project', published.project, mime)
                    self.add_mime_data('sg_status_list', published.sg_status_list, mime)

            drag.setMimeData(mime)

            thumbnail_widget = self
            for current_widget in self.widget_list:
                if isinstance(current_widget, labeled.ThumbnailWidget):
                    thumbnail_widget = current_widget
                    break

            pixmap = QtGui.QPixmap(thumbnail_widget.size())
            thumbnail_widget.render(pixmap)
            drag.setPixmap(pixmap)
            print(mime.formats())
            drag.exec_(QtCore.Qt.MoveAction)

    def dropEvent(self, event):
        """ """
        print("Dropped!")

    @classmethod
    def config_card(cls):
        config_manager_instance = config_manager.ConfigSolver(project='')
        config = config_manager_instance.get_config('layouts', module='breakdown_manager')

        cls.layouts_data = config

    def get_value(self, instance, variable):

        bits = variable.split('.', 1)
        if len(bits) > 1:
            new_instance = getattr(instance, bits[0])
            if isinstance(new_instance, database.View):
                output = []
                for item in new_instance:
                    output_item = self.get_value(item, bits[-1])
                    if isinstance(output_item, str) and output_item not in output:
                        output.append(output_item)
            else:
                output = self.get_value(new_instance, bits[-1])

            return output
        else:
            return getattr(instance, bits[0])

    def init_widget(self, widget, init_function_name, view, value_name, value):
        module_name, function_name = init_function_name.rsplit('.', 1)
        module = importlib.import_module(module_name)
        function = getattr(module, function_name)
        values = function(view, value_name)
        widget.setValues(values)
        widget.setValue(value)

    def query_variant(self, query_value):
        value = ''
        if query_value == 'self':
            value = self.view
        elif query_value is not None:
            value = self.get_value(self.view, query_value)

        if isinstance(value, database.emptyView):
            value = ''

        return value

    def build_layout(self, data):
        if data['arrange'] == 'vertical':
            layout = QtWidgets.QVBoxLayout()
        else:
            layout = QtWidgets.QHBoxLayout()

        for line in data['items']:
            if 'arrange' in line.keys():
                h_layout = self.build_layout(line)
                layout.addLayout(h_layout)
            else:
                label = line.get('label')

                query_value = line.get('value')
                query_values = line.get('values')
                value = self.query_variant(query_value)
                values = self.query_variant(query_values)

                if len(values) == 1:
                    value = values[0]

                widget = getattr(labeled, line['widget'])

                line['args'] = line.get('args', {})

                if values:
                    line['args']['values'] = values
                for key, arg_value in line['args'].items():
                    if arg_value == 'self':
                        line['args'][key] = self
                    if arg_value == 'view':
                        line['args'][key] = self.view

                new_widget = widget(value, label, parent=self, **line['args'])

                layout.addWidget(new_widget)
                self.widget_list.append(new_widget)
                var_name = label.lower().replace(' ', '_')
                setattr(self, var_name, new_widget)

        if data.get('connect'):
            for connection in data['connect']:
                function = getattr(self, connection['function'])
                emitter = getattr(self, connection['emitter'])
                signal = getattr(emitter, connection['signal'])
                signal.connect(function)

        return layout

    def on_select(self):
        self.clicked.emit(self.view.id)