import getpass
import logging
import os
import platform

import deadline_lib.lib.deadline_submitter as deadline_submitter
import publisher.lib.processes_node as process_node

DEFAULT_SCRIPT_FOLDER = 'publisher/resources/scripts'
logger = logging.getLogger()

exception_list = ['ModuleNotFoundError'] 


class RunCommand(process_node.ProcessesNode):

    def do_it(self):
        self.get_job_names()
        out = self.launch_in_deadline()
        return out

    def get_full_path(self, relative_path):
        logger.info('----- get full path -----')
        tools_root = self.company_config['root']
        relative_path = relative_path.replace('\\', '/')

        if relative_path.find('/') > -1:
            module, local_path = relative_path.split('/', 1)
            logger.debug('Module: %s, rest path : %s' % (module, local_path))

            version = None
            for tools_group in self._manager.tools_config.values():
                if module in tools_group:
                    version = tools_group[module]
            logger.debug('Version: %s' % version)
            if version:
                if version == 'local':
                    full_path = 'c:/dev/%s' % relative_path
                    return full_path
                else:
                    full_path = '%s/%s/%s/%s' % (tools_root, module, version, local_path)
                    logger.debug('full version path: %s' % full_path)
                    if os.path.exists(full_path):
                        return full_path
                
            full_path = '%s/%s' % (tools_root, relative_path)
            logger.debug('no version: %s' % full_path)
            if os.path.exists(full_path):
                return full_path
            
        return relative_path

    def get_command_line(self):
        logger.debug('----- get command line -----')
        command_full = str(self.command_line['command'])
        logger.debug('original: %s' % command_full)
        if not os.path.exists(command_full):
            logger.debug('dont exists, search local path')
            if command_full[1] != ':':
                command_full = self.get_full_path(command_full)

        cmd = [command_full]

        for label, arg in self.command_line['arguments'].items():
            if label == 'project':
                arg = self.project
            if label.startswith('keyarg'):
                pass
            elif len(label) > 2 and not self.command_line.get('short_args', False):
                cmd.append('--%s' % label)
            else:
                cmd.append('-%s' % label)

            arg = str(arg)
            if arg.find(' ') > -1:
                arg = '"%s"' % str(arg)
            arg = arg.replace('\\', '/')
            cmd.append(arg)
        return cmd

    def get_environment(self):

        if hasattr(self, 'environ'):
            environ = self.environ
        else:
            environ = {}

        environ['PROJECT'] = self.project
        logger.debug('ENVIRONMENT')
        for key in sorted(environ.keys()):
            logger.debug('ENV %s: %s' % (key, environ[key]))
        return environ

    def get_dependencies(self):

        job_dependencies = []
        node_dependencies = self._manager.dependencies[self.name]

        for node_name in node_dependencies['inputs']:
            node = self._manager.nodes.get(node_name, None)
            if node is None:
                continue
            if hasattr(node, 'deadline_job_id') and getattr(node, 'deadline_job_id'):
                job_dependencies.append(getattr(node, 'deadline_job_id'))

        return job_dependencies

    def get_job_names(self):
        bits = self.name.split('_')
        if len(bits) > 1:
            start = bits[0]
            others = [bit.title() for bit in bits[1:]]
            compact_name = '%s%s' % (start, ''.join(others))
        else:
            compact_name = self.name

        if self.asset_name:
            self.job_name = 'job_%s_%s_%s' % (self.asset_name, self.variant, compact_name)
            self.batch_name = 'publish_%s_%s_%s_v%03d' % (self.asset_name, self.variant, self.step, self.version_number)
        else:
            if hasattr(self, 'subset') and self.subset:
                self.job_name = 'job_%s_%s_%s' % (self.shot_name, compact_name, self.subset)
                self.batch_name = 'publish_%s_%s_%s_v%03d' % (
                self.shot_name, self.subset, self.step, self.version_number)
            else:
                self.job_name = 'job_%s_%s' % (self.shot_name, compact_name)
                self.batch_name = 'publish_%s_%s_v%03d' % (self.shot_name, self.step, self.version_number)

    def local_job(self):
        tools_group = self._manager.tools_config.get('libraries').values()
        if 'local' in tools_group:
            return True
        return False


    def get_base_job_options(self):
        job_options = {'Plugin': 'AACommandLine'}

        python_path = 'PYTHONPATH=%s'

        job_options.update({"Name": self.job_name,
                            "BatchName": self.batch_name,
                            "Pool": self.pool,
                            "Group": "on",
                            "Priority": 100,
                            'OutputFilename0': os.path.basename(self.output_path),
                            'OutputDirectory0': os.path.dirname(self.output_path),
                            'EnvironmentKeyValue0': python_path % self.company_config['tools_root'],
                            'EnvironmentKeyValue1': 'COMPANY_ROOT=%s' % os.environ.get('COMPANY_ROOT'),
                            'EnvironmentKeyValue2': 'LAUNCHER_VERSION=%s' % os.environ.get('LAUNCHER_VERSION', '1.0.0'),
                            'EnvironmentKeyValue3': 'LIBRARY_VERSION=%s' % os.environ.get('LIBRARY_VERSION', '1.0.0'),
                            'EnvironmentKeyValue4': 'TOOLS_ROOT=%s' % os.environ.get('TOOLS_ROOT', 'V:/company/tool'),
                            'EnvironmentKeyValue5': 'LOCAL_PIPE=%s' % os.environ.get('LOCAL_PIPE', '0'),
                            'UserName': getpass.getuser()
                            })

        if self.local_job():
            machine = platform.node()
            job_options['Pool'] = 'development'
            job_options['Allowlist'] = machine


        elif hasattr(self, 'limits'):
            if isinstance(self.limits, str):
                job_options['LimitGroups'] = self.limits
            else:
                job_options['LimitGroups'] = ','.join(self.limits)

        return job_options

    def launch_in_deadline(self):
        try:
            deadline_connection = deadline_submitter.init_connection(self.project)
        except:
            logger.error('Can\'t connect to deadline webservice')
            self.error = True
            return -1

        if not hasattr(self, 'pool'):
            self.pool = 'system_jobs'
            
        logger.debug('pool: %s' % self.pool)

        cmd = self.get_command_line()
        executable = cmd[0]

        if len(cmd) > 1:
            argument_list = [str(arg) for arg in cmd[1:]]
            arguments = ' '.join(argument_list)
        else:
            arguments = ''

        job_options = self.get_base_job_options()

        if hasattr(self, 'job_first_frame') and hasattr(self, 'job_first_frame'):
            if hasattr(self, 'job_total_frames'):
                first = int(self.job_first_frame)
                last = int(self.job_last_frame)
                total_frames = last - first
                number_of_frames = float(self.job_total_frames)
                step = total_frames / (number_of_frames - 1)
                frames = []
                for index in range(int(number_of_frames)):
                    frames.append(str(int(first + step * index)))
                frames.append(str(last))
                job_options['Frames'] = ','.join(frames)

            else:
                job_options['Frames'] = '%i-%i' % (int(self.job_first_frame), int(self.job_last_frame))
        else:
            job_options['Frames'] = '1'

        plugin_options = {'Executable': str(executable),
                          'Arguments': arguments,
                          'Project': self.project,
                          'overrides': os.environ.get('SETTING_OVERRIDES', '')}

        dependencies = self.get_dependencies()

        if dependencies:
            job_options['JobDependencies'] = ','.join(dependencies)
        
        if hasattr(self, 'application') and self.application:
            plugin_options['Standalone'] = self.application

        logger.info('command line: %s' % executable)

        logger.info('Arguments: %s' % arguments)

        logger.debug('Plugin options:')
        for key in sorted(plugin_options.keys()):
            logger.debug('Plugin options: %s: %s' % (key, plugin_options[key]))

        logger.debug('Job options:')
        for key in sorted(job_options.keys()):
            logger.debug('Job options: %s: %s' % (key, job_options[key]))

        try:
            new_job = deadline_connection.Jobs.SubmitJob(job_options, plugin_options)

        except Exception as e:
            for line in str(e).split('\n'):
                logger.error(line)

            self.error = True
            return -1

        if not isinstance(new_job, dict):
            logger.error(new_job)
            return -1

        self.deadline_job_id = new_job['_id']

        logger.info('Job launched to deadline')

        return 1

    def save_job_info(self, data, path):

        with open(path, 'w') as job_file:
            for key, value in data.items():
                if isinstance(value, str):
                    value = value.replace('\\', '/')
                job_file.write('%s=%s\n' % (key, value))

        return path

