import os
import sys
import importlib
import logging

from pprint import pformat, pprint

import library.core.references_solver as references_solver

logger = logging.getLogger()

this_path = os.path.dirname(os.path.abspath(__file__))

RESOURCES_PATH = this_path

sys.path.append(RESOURCES_PATH)
libraries = os.listdir(RESOURCES_PATH)

DEFAULT_NODES_MODULE = 'publisher.resources.nodes'


def to_class_name(module_name):
    last_name = module_name.split('.')[-1]
    bits = last_name.split('_')
    class_name =''.join([bit.title() for bit in bits])
    return class_name


class ProcessBuilder(references_solver.ReferencesSolver):

    def __new__(cls, *args, **kwargs):
        class_name = ProcessesNode
        instance_name = args[0]
        if 'node_type' in kwargs:
            module_name = kwargs['node_type']
            
            clas_name = to_class_name(module_name)
            if module_name.find('.') == -1:
                module_name = '%s.%s' % (DEFAULT_NODES_MODULE, module_name)
            mymodule = importlib.import_module(module_name)
            importlib.reload(mymodule)
            if hasattr(mymodule, clas_name):
                class_name = getattr(mymodule, clas_name)
                
        new_node = class_name(*args, **kwargs)

        return new_node


class ProcessesNode(references_solver.ReferencesSolver):

    connections = []
    def __init__(self, name, data, manager, published_version, node_type=None, level=0, start_delimiter='<', end_delimiter='>'):
        super().__init__(self)
        self.name = name
        self.data = data
        self.type = self.data['type']

        self._manager = manager
        self.project = self._manager.project
        self.project_config = self._manager.project_config
        self.company_config = self._manager.company_config

        self.published_version = published_version
        self.node_type = node_type
        self.sg_connection = self.published_version._database.sg

        self.relative_path = ''
        self.error = False

        for key, value in self._manager.definition.get('constants', {}).items():
            setattr(self, key, value)

        self.level = level
        self.output_files = {}
        self.metadata = {}
        self.children_error = False
        self.multi_frame = False

        logger.debug('%sCreating node %s of type %s' % (' ' * self.level, self.name, self.type))

        self._solved_vars = {}
        self._vars_to_solve = self.data['args']

        self.create_children()

        if self.children_error:
            self.error = True
            logger.warning('Not running the node %s of type %s because a child returned an error code' % (self.name, self.node_type))
            return 
        
        self.pre_job()
        logger.info('%s %s - %s %s'  % ('-' * 5, self.node_type, self.name, '-' * 5))

        output = self.do_it()
        
        if output == -1:
            logger.debug('The node returned an error code')
            self.error = True
        else:
            self.post_job()

    def pre_job(self):
        self.init_values()

    def post_job(self):
        self.update_shotgrid()

    def update_dictionary(self, dict_a, dict_b):
        for key, value in dict_b.items():
            if isinstance(value, dict):
                if key in dict_a:
                    dict_a[key] = self.update_dictionary(dict_a[key], dict_b[key])
                else:
                    dict_a[key] = value
            else:
                dict_a[key] = value

        return dict_a

    def update_shotgrid(self):
        if not self.output_files and not self.metadata:
            return

        if not self.sg_files:
            updated_dict = self.output_files
        else:
            updated_dict = self.sg_files.copy()
            updated_dict = self.update_dictionary(updated_dict, self.output_files)

        if not self.sg_metadata:
            metadata_dict = self.metadata
        else:
            metadata_dict = self.sg_metadata.copy()
            metadata_dict = self.update_dictionary(metadata_dict, self.metadata)

        data = {'sg_files': pformat(updated_dict)}
        if self.metadata:
            data['sg_metadata'] = pformat(metadata_dict)

        self.published_version.sg_files = str(updated_dict)
        self.published_version.sg_metadata= str(metadata_dict)
        self.published_version.update_shotgun()

    def set_version_values(self):
        self.published_version.precache_dependencies(fields=['sg_asset'])

        version = self.published_version

        self.asset_name = version.sg_asset_name
        self.asset_type = version.sg_asset.sg_asset_type

        self.shot_name = version.sg_context_name
        if version.sg_context.type == 'Shot':
            self.start_frame = version.sg_context.sg_cut_in
            self.end_frame = version.sg_context.sg_cut_out
        else:
            self.start_frame = 1001
            self.end_frame = 1001
        
        self.published_folder = version.sg_published_folder

        self.step = version.sg_task.step_name
        self.task_code = version.sg_task_name
        self.user_login = version.sg_artist.login
        self.output_path = '%s/<relative_path>' % self.published_folder
        self.version_number = version.sg_version_number
        self.variant = version.sg_variant_name
        self.version_number_str = '%03d' % version.sg_version_number
        self.hash = version.sg_hash
        self.shotgrid_id = int(version.id)
        self.sg_files = version.sg_files
        self.sg_metadata = version.sg_metadata

    def init_values(self):

        for variable, value in self._manager.constants.items():
            self._vars_to_solve[variable] = value

        for variable, value in self._manager.parameters.items():

            setattr(self, variable, value)
            if isinstance(value, str) and os.path.exists(value):
                extension = value.split('.')[-1]
                setattr(self, '%s_extension' % variable, extension)

        self.set_version_values()

        self._vars_to_solve['output_path'] = self.output_path

        for variable, value in self._vars_to_solve.items():
            setattr(self, variable, value)

    def get_children(self):

        node_dependencies = self._manager.dependencies[self.name]

        children = node_dependencies['inputs']
        return children

    def get_node_data(self, node_name):
        return self._manager.definition['element_process'][node_name]

    def create_children(self):
        children = self.get_children()
        if not children:
            return
        
        for child_name in children:
            
            if child_name in self._manager.nodes or child_name =='project_config':
                continue
            child_data = self.get_node_data(child_name)

            child = ProcessBuilder(child_name,
                                   child_data,
                                   self._manager,
                                   self.published_version,
                                   node_type=child_data['type'],
                                   level=self.level + 1)

            self._manager.nodes[child_name] = child
            if child.error:
                logger.debug('The child node returned an error code')
                self.children_error = True
                return -1

        return None


    def do_it(self):
        print('\t' * self.level, 'Now Work', self.name, self.type)
