import time
from pprint import pprint
import traceback
start = time.time()
import os
import sys
import importlib

from PySide import QtWidgets, QtCore

import packages_io
import library.ui.interface_builder as interface_builder
import library.ui.labeled as labeled
import library.ui.context_widget as context_widget
import library.ui.logging_dialog as logging_dialog
import library.ui.dialog_widgets as dialog_widgets

import publisher.lib.publish_cmd as publish_cmd
import publisher.lib.ui.sanity_check as sanity_check

#importlib.reload(context_widget)
#importlib.reload(sanity_check)
#importlib.reload(logging_dialog)
importlib.reload(publish_cmd)


class PublisherWindow(packages_io.MainWindow):
    """
    Main publisher for all the pipeline steps

    publish_type: The type of the publish to be created
    project: The code of the Shotgrid project to work on it

    """
    def __init__( self, publish_type='Animatic', project=None, parent=None):

        self.publish_type = publish_type

        self.title = 'Publisher: %s : %s' % (publish_type, project)
        super( PublisherWindow, self ).__init__(self.title, parent=parent)

        self.setContentsMargins(0,0, 0,0)
        self.main_widget = publisherWidget(self.publish_type, project=project, parent=self)
        self.setMinimumSize(500, 800)
        self.setCentralWidget(self.main_widget)

        self.main_widget.start()
        self.apply_style(self)

    def closeEvent(self, event):
        print('close_event')
        self.main_widget.close_window()
        event.accept()

class publisherWidget(context_widget.ContextWidget):

    precached_entities = ['Step', 'HumanUser', 'Project', 'CustomEntity10', 'CustomEntity11']
    main_module = 'publisher'

    def __init__(self, definition_type, parent=None, project=None):
        super(publisherWidget, self).__init__(definition_type, parent=parent, project=project)
        self.log_dialog = None
        self.setObjectName('Builder: %s' % definition_type)
        self.definition_context = self.definition.get('definition_context', 'shot')
        self.definition_app = self.definition.get('definition_app', 'maya')
        self.pipeline_step = self.definition.get('pipeline_step')
        self.publish_type = self.definition.get('publish_type')

        self.scene_path = packages_io.file_name()
        self.add_path_widget(self.main_layout)
        if self.definition_context == 'shot':
            self.add_framerange_widget(self.main_layout)

        self.interface_widget = interface_builder.InterfaceWidget(self.definition.get('interface', {}), label_width=100)
        self.main_layout.addWidget(self.interface_widget)

        self.add_comment_widget(self.main_layout)
        self.add_inputs_widget(self.main_layout)
        self.add_sanity_widget(self.main_layout)

        self.launch = QtWidgets.QPushButton('Publish')
        self.launch.setFixedHeight(25)
        self.app_layout.addStretch(1)
        self.app_layout.addWidget(self.launch)
        self.launch.clicked.connect(self.launch_accept)
        self.launch.setEnabled(True)
        self.add_inputs()

    def enable_publish(self, status):

        self.launch.setEnabled(status)
        
    def add_inputs(self):
        config = packages_io.scan_file()
        pprint(config)
        self.inputs_widget.update(config=config)


    def add_framerange_widget(self, layout):
        self.start_widget = labeled.LabeledInt(101, 'First frame')
        self.start_widget.set_width(100)

        self.end_widget = labeled.LabeledInt(101, 'Last frame')
        self.end_widget.set_width(100)

        layout.addWidget(self.start_widget)
        layout.addWidget(self.end_widget)

    def add_comment_widget(self, layout):
        self.comment_widget = labeled.LabeledMultiLineText('', 'Comment')
        self.comment_widget.setFixedHeight(125)
        layout.addWidget(self.comment_widget)

    def add_log_widget(self, layout):
        self.log_widget = labeled.LabeledMultiLineText('', 'Log')
        self.log_widget.setFixedHeight(125)
        layout.addWidget(self.log_widget)


    def add_inputs_widget(self, layout):
        import library.ui.usd_scene as usd_scene
        importlib.reload(usd_scene)

        self.inputs_widget = usd_scene.UsdLayersTree()
        #self.inputs_widget.setFixedHeight(300)
        layout.addWidget(self.inputs_widget)

    def add_sanity_widget(self, layout):
        self.sanity_check_widget = sanity_check.SanityCheckWidget(self.definition.get('sanity_checks', {}),
                                                                  interface_widget=self)
        self.sanity_check_widget.launch_run.connect(self.open_log_window)

        layout.addWidget(self.sanity_check_widget)

    def add_path_widget(self, layout):
        if self.definition_app == 'maya':
            label = 'Maya scene'
            filter = 'Maya ascii (*.ma);;Maya binary (*.mb)'
        elif self.definition_app == 'nuke':
            label = 'Nuke script'
            filter = 'Nuke script (*.nk)'

        else:
            label = 'all files'
            filter = 'all files (*.*)'
        self.path_widget = labeled.LabeledPath(self.scene_path, label, filter=filter)
        self.path_widget.set_width(100)
        layout.addWidget(self.path_widget)

    def log_error(self, error_str):
        dialog = dialog_widgets.CustomDialog(self, 'ERROR!!!', error_str)
        dialog.exec_()
        return

    def get_parameters(self):
        parameters = self.interface_widget.getValue()
        context = self.context_widget.getValue()
        pprint(context)
        if not context.get(self.definition_context) and not context.get('name'):
            self.log_error('Cant find context')
            return None
    
        if self.definition_context == 'shot':
            parameters['shot_name'] = context['shot']
            parameters['entity_name'] = parameters['shot_name']
            parameters['sequence_name'] = context['sequence']
            parameters['episode_name'] = context['episode']
            parameters['start_frame'] = self.start_widget.getValue()
            parameters['end_frame'] = self.end_widget.getValue()
            parameters['entity_type'] = 'Shot'
        elif self.definition_context == 'sequence':
            parameters['entity_name'] = context['sequence']
            parameters['sequence_name'] = context['sequence']
            parameters['episode_name'] = context['episode']
            parameters['entity_type'] = 'Sequence'

        elif self.definition_context == 'episode':
            parameters['entity_name'] = context['episode']
            parameters['episode_name'] = context['episode']
            parameters['entity_type'] = 'Episode'

        elif self.definition_context == 'asset':
            parameters['asset_name'] = context['name']
            parameters['asset_type'] = context['type']
            parameters['entity_name'] = parameters['asset_name']
            parameters['entity_type'] = 'Asset'


        parameters['comment'] = self.comment_widget.getValue()
        parameters['dependencies'] =  self.inputs_widget.getValue()
        parameters['source_path'] = self.path_widget.getValue()
        parameters['task_code'] = self.task_widget.getValue()

        """
        if not parameters['source_path'] or not os.path.exists(parameters['source_path']):
            self.log_error('Can\'t find path')
            return

        if not parameters['comment']:
            self.log_error('A comment should be added to the publish')
            return
        """
        return parameters

    def open_log_window(self):
        try:
            print(self.log_dialog)
        except:
            #if not hasattr(self, 'log_dialog') or isinstance(self.log_dialog, logging_dialog.LoggingDialog):
            self.log_dialog = None

        if self.log_dialog is None:
            self.log_dialog = logging_dialog.LoggingDialog(self, 'Publishing: %s' % self.definition_type)

        if not self.log_dialog.isVisible():
            self.log_dialog.show()
            self.log_dialog.init_handler()

    def launch_accept(self):
        QtWidgets.QApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.open_log_window()
        self.sanity_check_widget.run()

        comment = self.comment_widget.getValue()

        if not comment:
            self.log_error('Need to add some comment to the publish')
            QtWidgets.QApplication.restoreOverrideCursor()

            return

        source_path = self.path_widget.getValue()

        if not source_path:
            QtWidgets.QApplication.restoreOverrideCursor()

            self.log_error('Can\'t find source file path')

            return

        check_sanity = self.sanity_check_widget.getValue()
        check_passed = True
        ignored_checks = []

        for check_name, check_output in check_sanity.items():
            if check_output is False:
                check_passed = False
            elif check_output == 'ignored':
                ignored_checks.append(check_name)

        if not check_passed:
            QtWidgets.QApplication.restoreOverrideCursor()
            self.log_error('Some sanity check had fail')
            return

        parameters = self.get_parameters()
        if parameters is None:
            QtWidgets.QApplication.restoreOverrideCursor()

            self.log_error('Some parameter are missed')
            return
        
        mandatory_options = {}
        if self.definition:
            for asset in self.definition.get('interface', {}):
                for key, value in asset.items():
                    mandatory_options[key] = value.get('mandatory', False)

        mandatory_empty = []
        for name, value in parameters.items():
            if value == '' and mandatory_options.get(name):
                mandatory_empty.append(name)

        if mandatory_empty:
            QtWidgets.QApplication.restoreOverrideCursor()

            dlg = dialog_widgets.CustomDialog(self, 'Error!!',  'All the mandatory fields must be completed: %s' % ','.join(mandatory_empty))
            dlg.exec_()
            return

        parameters['ignored_checks'] = ignored_checks

        try:
            published_version = publish_cmd.PublishCmd(self.definition_type,
                                                       parameters,
                                                       self.project)

            QtWidgets.QApplication.restoreOverrideCursor()
            dlg = dialog_widgets.CustomDialog(self, 'Done!!', 'The task finished correctly')
            dlg.exec_()

        except Exception as e:
            trace_error = traceback.format_exc()
            for line in str(trace_error).split('\n'):
                self.log_dialog.append(line, level='ERROR')

            for line in str(e).split('\n'):
                self.log_dialog.append(line, level='ERROR')

            QtWidgets.QApplication.restoreOverrideCursor()

            dlg = dialog_widgets.CustomDialog(self, 'ERROR!!', 'Some tasks didn\'t finished correctly')
            dlg.exec_()

        QtWidgets.QApplication.restoreOverrideCursor()


    def close_window(self):
        if self.log_dialog:
            self.log_dialog.close()
        self.parent().close()

def open_publisher(pipeline_step, project, parent=None):
    print('open publisher', project)
    publisher_window = PublisherWindow(pipeline_step, project=project, parent=parent)
    publisher_window.show()

def open_publisher_substance(project, task, parent=None):
    entity = task.entity
    pipeline_step = task.step_name

    os.environ['PIPE_CONTEXT'] = 'Asset'
    os.environ['PIPE_ASSET_NAME'] = entity.code

    os.environ['PIPE_ASSET_TYPE'] = entity.sg_asset_type

    print('open publisher substance', project)
    publisher_window = PublisherWindow(pipeline_step, project=project, parent=parent)
    publisher_window.show()


if __name__ == '__main__':
    print('-'*50)

    import library.core.parser_function as parser_function
    importlib.reload(parser_function)

    QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_ShareOpenGLContexts)
    app = QtWidgets.QApplication(sys.argv)
    window = parser_function.parse_function(PublisherWindow)
    window.show()
    app.exec_()

