import logging
import re

logger = logging.getLogger()

import maya.cmds as cmds

transform_name_pattern = '([xlrud])_([a-zA-Z]*)_([0-9]{3})_([a-z]{3})'
geo_name_pattern = '([xlrud])_([a-zA-Z]*)_([0-9]{3})_([a-z]{3})Shape'

valid_sufix = {'msh': 'transform',
               'geo': 'transform'}

def check_group(group_name,  allow_empty=False):
    shapes = ['mesh', 'aiVolume']
    logger.info('Checking group : %s' % group_name)
    if not cmds.objExists(group_name):
        logger.error('Can\'t find node: %s' % group_name)
        return False

    children = cmds.listRelatives(group_name, ad=True, f=True)
    if not children:
        if allow_empty:
            logger.info('The group %s is empty but allowed:' % group_name)
            return True
        else:
            return False

    parent = ''
    success = True

    for child in sorted(children):
        short_name = child.split('|')[-1]
        if cmds.nodeType(child) not in shapes:
            match = re.fullmatch(transform_name_pattern, short_name)
            parent = short_name
        else:
            match = re.match(geo_name_pattern, short_name)
            if short_name[:-5] != parent:
                success = False
                logger.error('%s under %s don\'t match parent name %s' % (child, group_name, parent))

        if not match:
            logger.error(f'Node: {child} under group {group_name} don\'t match the naming conventions')
            success = False

    return success


def check_geo_names(geometry='', proxy_geo=''):
    render_success = check_group(geometry)
    if proxy_geo and cmds.objExists(proxy_geo):
        proxy_success = check_group(proxy_geo, allow_empty=True)
    else:
        proxy_success = True

    success = proxy_success and render_success

    if not success:
        logger.warning('Some node get errors in naming conventions. The corret structure of the nomenclature is: ' \
                       +'"side_name_index_msh" for the meshes and "side_name_index_grp" for the groups')

    return success

def check_fx_names(geometry='', fx_geo=''):
    logger.info('Checking naming conventions geometry :')
    logger.info('geometry: %s ' % geometry)
    logger.info('proxy_node: %s ' % fx_geo)

    render_success = check_group(geometry)

    proxy_success = check_group(fx_geo, allow_empty=True)

    success = proxy_success and render_success

    return success


def check_shading_names():
    logger.info('Running sanity check : check_shading_names' )
    render_name_start = {'redshift': 'Redshift',
                         'arnold': 'ai'}

    sg_name_pattern = '([xlrud])_([a-zA-Z0-9]*)_shd_SG'
    shader_name_pattern = '([xlrud])_([a-zA-Z0-9]*)_shd_([a-zA-Z]*)'

    ignore_sg_nodes = ['initialParticleSE', 'initialShadingGroup', 'lambert1SG']
    ignore_shaders = ['lambert1', 'particleCloud1', 'standardSurface1']
    logger.info('Checking naming conventions shaders:')

    shading_groups = cmds.ls(type='shadingEngine')
    success = True
    for sg_node in shading_groups:
        if sg_node in ignore_sg_nodes:
            logger.info('Ignoring the node %s ' % sg_node)
            continue

        match = re.fullmatch(sg_name_pattern, sg_node)
        if not match:
            logger.error('The node %s don\'t follow the naming convention x_Name_shd_SG' % sg_node)
            success = False
    shader_list = cmds.listNodeTypes('shader')
    for shader_type in shader_list:
        logger.debug('Checking shaders of type: %s' % shader_type)

        all_type_shaders = cmds.ls(type=shader_type)
        if not all_type_shaders:
            continue

        for shader_name in all_type_shaders:
            logger.debug('Check shader; %s' % shader_name)
            if shader_name in ignore_shaders:
                logger.info('Ignoring the shader %s ' % shader_name)
                continue

            match = re.fullmatch(shader_name_pattern, shader_name)
            if not match:
                logger.error('The node %s don\'t follow the naming convention: x_Name_shd_sufix' % shader_name)
                success = False

    return success