import logging
import os
import sys
import importlib
from pprint import pprint


from PySide import QtCore, QtGui, QtWidgets

import library.core.references_solver as references_solver

import library.core.dynamic_run as dynamic_run
logger = logging.getLogger(__name__)
importlib.reload(references_solver)

this_file = os.path.abspath(__file__)
this_folder = os.path.dirname(this_file)
icon_folder = os.path.join(this_folder, '../../icons')


class CheckWidget(QtWidgets.QFrame):
    launch_run = QtCore.Signal()

    font_style_sheet = "color: rgb(210,210,210);"
    def __init__(self, name, check_data, parent=None, interface_widget=None):

        super(CheckWidget, self).__init__(parent=parent)
        self.name = name
        self.label = check_data['label']
        self.optional = check_data.get('optional', False)
        self.check_data = check_data
        self.interface_widget = interface_widget

        self.status = None
        self.info = 'Waiting to be Checked'
        self.setFixedHeight(20)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.main_layout = QtWidgets.QGridLayout()

        self.check_widget = QtWidgets.QCheckBox()
        self.check_widget.setFixedWidth(20)
        self.check_widget.setChecked(True)
        if not self.optional:
            self.check_widget.setEnabled(False)

        self.label_widget = QtWidgets.QLabel(self.label)

        self.status_button_widget = QtWidgets.QPushButton('')
        self.status_button_widget.setFixedWidth(18)
        self.status_button_widget.setFixedHeight(18)
        play_icon = QtGui.QPixmap('%s/play.png' % icon_folder)
        self.status_button_widget.setIcon(play_icon)
        self.status_button_widget.setStyleSheet('background-color: rgb(50, 50, 50')
        self.fix_button_widget = QtWidgets.QPushButton('')
        self.fix_button_widget.setFixedWidth(18)
        self.fix_button_widget.setFixedHeight(18)

        path = '%s/tools.png' % icon_folder
        path = path.replace('\\', '/')

        tool_icon = QtGui.QPixmap(path)
        self.fix_button_widget.setIcon(tool_icon)

        self.fix_button_widget.setHidden(True)

        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(20, 1, 2, 1)

        self.main_layout.addWidget(self.check_widget, 0, 0)
        self.main_layout.addWidget(self.label_widget, 0, 1)
        self.main_layout.addWidget(self.fix_button_widget, 0, 2)
        self.main_layout.addWidget(self.status_button_widget, 0,3)

        self.status_button_widget.clicked.connect(self.run)
        self.fix_button_widget.clicked.connect(self.fix)

        self.setLayout(self.main_layout)

    def fix(self):
        self.launch_run.emit()
        logger.info('Run Fix check: %s' % self.check_data['label'])

        parameters = self.get_current_parameters()
        script_run_config = self.get_script_config(parameters, script_type='fix')
        self.status = dynamic_run.run_scripts(script_run_config)
        logger.info('Done: %s' % self.check_data['label'])
        self.run()
        self.set_status_check()

        return self.status

    def get_current_parameters(self):
        parameters = self.interface_widget.get_parameters()
        if parameters is None:
            parameters = {}

        parameters['project_config'] = self.interface_widget.project_config
#        parameters['variant'] = self.interface_widget.current_task.sg_variant_name

        return parameters


    def get_script_config(self, parameters, script_type='run'):

        if script_type == 'fix':
            script_run_config = self.check_data['fix_script'].copy()
        else:
            script_run_config = self.check_data['check_script'].copy()

        args = script_run_config.get('args', {})
        args = references_solver.ReferencesSolver.parse_dict(args, aditional_values=parameters)


        for arg, value in args.items():
            logger.debug('argument: %s = %s' % (arg, str(value)))

        script_run_config['args'] = args

        return script_run_config

    def run(self):
        self.launch_run.emit()
        self.set_running()

        init_string = 'Run sanity check: %s' % self.check_data['label']
        logger.info(init_string)

        parameters = self.get_current_parameters()
        script_run_config = self.get_script_config(parameters)

        self.status = dynamic_run.run_scripts(script_run_config)
        logger.debug('Sanity check finished with code %s' % str(self.status))
        if self.status:
            logger.info('Passed...')
        else:
            logger.error('Failed...')
        self.set_status_check()
        return self.status

    def set_running(self):
        current_icon = QtGui.QPixmap('%s/icons8-next-32.png' % icon_folder)
        self.status_button_widget.setIcon(current_icon)
        self.info = 'Running...'

    def set_status_check(self):

        if self.status is None:
            current_icon = QtGui.QPixmap('%s/error.png' % icon_folder)
            if self.check_data.get('fix_script'):
                self.fix_button_widget.setHidden(False)
        elif self.status:
            current_icon = QtGui.QPixmap('%s/check.png' % icon_folder)
            self.fix_button_widget.setHidden(True)
        else:
            current_icon = QtGui.QPixmap('%s/error.png' % icon_folder)
            if self.check_data.get('fix_script'):
                self.fix_button_widget.setHidden(False)

        self.status_button_widget.setIcon(current_icon)

        return self.status


    def set_paused(self):
        self.paused = not self.paused
        if self.paused:
            pause_icon = QtGui.QPixmap('%s/pause_off.png' % icon_folder)
            self.pause_button_widget.setIcon(pause_icon)
        else:
            pause_icon = QtGui.QPixmap('%s/pause_on.png' % icon_folder)
            self.pause_button_widget.setIcon(pause_icon)

class SanityCheckWidget(QtWidgets.QFrame):
    launch_run = QtCore.Signal()

    def __init__(self, check=None, parent=None, interface_widget=None):
        super(SanityCheckWidget, self).__init__(parent=parent)
        if check:
            self.checks_data = check
        else:
            self.checks_data = {}
        self.interface_widget = interface_widget

        self.check_widgets = []
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_widget = QtWidgets.QWidget()

        self.scroll_layout = QtWidgets.QVBoxLayout()
        self.scroll_layout.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignTop)
        self.scroll_layout.setSpacing(1)
        self.scroll_layout.setContentsMargins(5, 2, 5, 5)

        self.title_widget = QtWidgets.QLabel('Sanity checks:')
        self.main_layout.addWidget(self.title_widget)
        for name, check_data in self.checks_data.items():
            new_widget = CheckWidget(name, check_data, interface_widget=self.interface_widget)
            new_widget.launch_run.connect(self.emit_launch_run)

            self.scroll_layout.addWidget(new_widget)
            self.check_widgets.append(new_widget)

        self.scroll_layout.addStretch(1)
        self.main_widget.setLayout(self.scroll_layout)
        self.scroll_area = QtWidgets.QScrollArea(self)
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setWidget(self.main_widget)
        self.scroll_area.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.main_layout.addWidget(self.scroll_area)
        self.setLayout(self.main_layout)

    def emit_launch_run(self):
        self.launch_run.emit()

    def run(self, all_data=None):

        logger.debug('RUN sanity checks main')
        status = True
        for check_widget in self.check_widgets:

            logger.debug(check_widget.label_widget.text())
            run_this = check_widget.check_widget.isChecked()
            logger.debug('Run this: %s' % str(run_this))

            if run_this:
                this_check_status = check_widget.run()
                status = this_check_status and status

        self.status = status

        return status

    def getValue(self):
        return self.status

class SanityToolWidget(QtWidgets.QWidget):

    def __init__(self, definition_type, project, parent=None):
        super(SanityToolWidget, self).__init__(arrange='horizontal', parent=parent)
        self.definition_type = definition_type
        self.project = project
        self.definition_data = get_definition_sanity_checks(self.definition_type, self.project)
        self.sanity_check_widget = SanityCheckWidget(check=self.definition_data)

         #self.log_window = log_widget.LoggingWidget()
        self.main_layout.addWidget(self.sanity_check_widget)
        self.main_layout.addWidget(self.log_window)
        #self.signal_handler = log_widget.SignalHandler(self.log_window)

        self.sanity_check_widget.run()

def get_definition_sanity_checks(definition_type, project):
    import library.core.config_manager as config_manager
    config_manager = config_manager.ConfigSolver(project=project)
    definition = config_manager.get_config('definitions/%s' % definition_type,
                                           module='publisher',
                                           merge=False)
    return definition.get('sanity_checks', {})

if __name__ == '__main__':

    app = QtWidgets.QApplication(sys.argv)
    data = {'checka': 'none', 'checkb':'other'}
    window = SanityCheckWidget(data)
    window.show()
    app.exec_()