import os
import logging

import packages_io
import library.sandbox as sandbox_builder
import library.core.config_manager as config_manager

logger = logging.getLogger()

import maya.cmds as cmds


def is_valid_context(asset_name='',
                     asset_type=''):
    result = True

    if not 'PIPE_ASSET_NAME' or not 'PIPE_ASSET_TYPE' in os.environ:
        logger.error('Context not set. Please click on a task to set the context.')
        result = False

    context_asset_type = os.environ.get('PIPE_ASSET_TYPE', '')
    context_asset_name = os.environ.get('PIPE_ASSET_NAME', '')

    if asset_type != context_asset_type:
        msg = f'You are trying to publish the asset type {context_asset_type} in the asset type {asset_type}. '
        msg += f'This is not mandatory but, if you want to do this, uncheck this sanity check and launch the publish again.'
        logger.warning(msg)
        result = False
    elif asset_name != context_asset_name:
        msg = f'You are trying to publish the asset {context_asset_name} in the asset {asset_name}. '
        msg += f'This is not mandatory but, if you want to do this, uncheck this sanity check and launch the publish again.'
        logger.warning(msg)
        result = False
    
    if result:
        logger.info(f"Asset type: {asset_type} and Asset name: {asset_name}, has correct context.")

    return result


def is_valid_scene(extensions=None):
    result = True

    config_solver = config_manager.ConfigSolver(project=os.environ['PROJECT'])
    project_data = config_solver.get_config('project')    

    scene_path = packages_io.file_name()
    logger.info(f"Scene name: {scene_path}")
    extension = scene_path.split('.')[-1]

    if not scene_path:
        logger.error('This is not a work scene. Please open a work scene to can publish.')
        result = False
    elif not 'PIPE_ASSET_NAME' or not 'PIPE_ASSET_TYPE' in os.environ:
        logger.error('Context not set. Please click on a task to set the context or choose one.')
        result = False
    elif not extension in extensions:
        logger.error('This is not a valid scene to publish. Please open a work scene to publish the scene.')
        result = False
    elif scene_path.startswith(project_data['paths']['publish_server'].replace('\\', '/')):
        logger.error('You are trying to publish an alrerady published scene. \n' \
                     + 'Please use the fix to save this scene in your asset sandbox.')
        result = False
    elif scene_path.startswith(project_data['paths']['images_server'].replace('\\', '/')):
        logger.error('You are trying to publish a scene from renders server. \n' \
                     + 'Please go to your sandbox to open any wip scene or use the builder.')
        result = False
    
    return result


def fix_scene_location(sandbox_env=None,
                       step='',
                       variant=''):             
    
    if not sandbox_env or not step or not variant:
        logger.error('Args of this sanity are not defined, please contact with pipline team to solve this issue.')
        return False
    elif not 'PIPE_ASSET_NAME' or not 'PIPE_ASSET_TYPE' in os.environ:
        logger.error('Context not set. Please click on a task to set the context.')
        return False
    
    sandbox_solver = sandbox_builder.Sandbox(project=os.environ['PROJECT'])
    sandbox_solver.set_context(asset_name=os.environ['PIPE_ASSET_NAME'], 
                               asset_type=os.environ['PIPE_ASSET_TYPE']
                               )
    sandbox_path = sandbox_solver.build()

    extension = packages_io.file_name().split('.')[-1]

    sandbox_solver.content = step
    sandbox_solver.variant = variant
    sandbox_solver.extension = extension if extension else 'ma'

    filename = sandbox_solver.generate_filename()
    final_path = sandbox_solver.get_version_path(sandbox_path + 
                                                 sandbox_env + 
                                                 filename)


    packages_io.file_save(rename_to=final_path)


