import logging

logger = logging.getLogger()

import maya.cmds as cmds

def get_scene_cameras():
    cameras  = list()
    for asset in cmds.ls(type='RigAssembly'):
        if cmds.getAttr('{}.assetName'.format(asset)) != 'Camera':
            continue

        cameras.append(asset)
    
    return cameras


def cam_exists():
    logger.info("Checking scene cameras")
    result = True

    scene_cams = get_scene_cameras()

    if not scene_cams:
        logger.error('There is not camera in the scene. This asset is mandatory to can publish in this step.' \
                     + 'Please add a cam assembly to this scene.')
        result = False
    elif len(scene_cams) > 1:
        logger.error('There is more than one camera in the scene. Please use just one.')
        result = False

    return result


def get_cam_animation_range(first_frame=False, last_frame=False):
    animated_cam_frames = list()
    nim_value = 101
    max_value = 102

    cam_assembly = get_scene_cameras()

    if not cam_assembly and first_frame:
        return nim_value
    elif not cam_assembly and last_frame:
        return max_value
    elif not cam_assembly and not first_frame and not last_frame:
        return [nim_value, max_value]

    nameSpace = cmds.getAttr("{}.representationNamespace".format(cam_assembly[0]))

    cam_animation_curves = cmds.ls(cmds.listConnections(
                                                    '{}:root'.format(nameSpace)), 
                                                    typ='animCurve')
    for cc in cam_animation_curves:
        animated_cam_frames.extend(cmds.keyframe(cc, q=True))

    if animated_cam_frames:
        nim_value = int(min(animated_cam_frames))
        max_value = int(max(animated_cam_frames))
    
    if first_frame:
        return nim_value
    elif last_frame:
        return max_value

    return [nim_value, max_value]


def cam_anim_duration(max_duration=None):
    result = True

    if max_duration is None:
        logger.error('Max duration is not defined, please contact with pipline team to solve this issue.')
        return False    
    
    cam_anim_range = get_cam_animation_range()
    
    if not cam_anim_range[0] or not cam_anim_range[1]:
        logger.error('No animation found in the camera. You must set any keyframe in the root ctrl of the camera rig,' \
                     + ' to get the setDressing QC.')
        return False
    
    len_cam_anim = len(range(cam_anim_range[0], cam_anim_range[1]))

    if len_cam_anim > max_duration:
        logger.error('The camera animation duration is greater than the maximum duration allowed. ' \
                     + 'The camera animation has {} consecutive frames and the maximum duration allowed is {} frames'.format(len_cam_anim, 
                                                                                                                            max_duration)
                                                                                                                            )
        result = False

    return result