import glob
import importlib
import os
import shutil
import logging

from pprint import pformat

import maya.cmds as cmds

import library.core.config_manager as config_manager
import shotgrid_lib.database as database
import shotgrid_lib.lib.publish_helpers as publish_helpers
import shotgrid_lib.shotgrid_helpers as shotgrid_helpers

importlib.reload(shotgrid_helpers)
#importlib.reload(database)
importlib.reload(publish_helpers)


logger = logging.getLogger()


texture_node_types = {'file': 'fileTextureName'}
channel_codes = ['MSK', 'DFC', 'MTL', 'NRM', 'RLR', 'HGT']


def get_udim(path):
    basename = os.path.basename(path)
    basename, ext = basename.rsplit('.', 1)
    udim = basename.split('_')[-1]
    if udim.isdigit():
        udim = int(udim)
    return udim


def get_udims(path, udim):
    import glob

    pattern = path.replace(udim, '*')

    all_files = glob.glob(pattern)

    all_udims = []
    for udim_file in all_files:

        udim = get_udim(udim_file)
        check_file = pattern.replace('*', '%04d' % udim)
        udim_file = udim_file.replace('\\', '/')
        check_file = check_file.replace('\\', '/')
        if check_file != udim_file:
            continue

        all_udims.append(udim)

    return all_udims, pattern

class TexturePublisher():
    def __init__(self, asset_name, task_code, project):
        self.asset_name = asset_name
        self.task_code = task_code
        self.project = project
        self.database = database.DataBase()
        self.database.fill(project=self.project)
        self.database.query_entity('Asset', as_precache=True)
        self.database.query_entity('CustomEntity10', as_precache=True)
        self.asset_view = self.database.entity_view('Asset')[self.asset_name]
        self.texture_publish_type =  self.database.entity_view('CustomEntity10')['texture']
        self.config_solver = config_manager.ConfigSolver(project=self.project)
        project_data = self.config_solver.get_config('project')
        self.server_path = project_data['paths']['publish_server']

        self.texture_data = self.get_texture_data()

    def get_mask(self, path):
        pattern_path = path.replace('/', '_')
        for bit in pattern_path.split('_'):
            if bit.upper() in channel_codes:
                return bit.upper()

    def get_texture_data(self):
        all_textures = {}

        for node_type, texture_attribute in texture_node_types.items():
            texture_nodes = cmds.ls(type=node_type)
            for node in texture_nodes:
                texture_path = cmds.getAttr('%s.%s' % (node, texture_attribute))
                all_textures[node] = {'source_path': texture_path}

        sorted_textures = {}

        asset_published_elements = self.asset_view.sg_published_elements
        texture_view = asset_published_elements.find_with_filters(sg_publish_type=self.texture_publish_type)

        for node, path_data in all_textures.items():

            path = path_data['source_path']
            published_file_data = None
            for published in texture_view:
                if path.startswith(published.sg_published_folder):
                    published_file_data = published
                    break

            if published_file_data and not published_file_data.empty:
                path_bits = path.split('/')
                channel_mask = path_bits[-2]
                name = published_file_data.sg_subset
                sorted_textures[name] = sorted_textures.get(name, {})

                metadata = published_file_data.sg_metadata
                udims = metadata['channels'][channel_mask]['udims']

                pattern = published_file_data.sg_files[channel_mask]
                sorted_textures[name][channel_mask] = {'udims': udims,
                                                       'relative_path': pattern,
                                                       'published_folder': published_file_data.sg_published_folder,
                                                       'path': '%s/%s' % (published_file_data.sg_published_folder, pattern),
                                                       'node': node,
                                                       'published_data': published_file_data}

            else:
                basename = os.path.basename(path)
                basename, ext = basename.rsplit('.', 1)
                bits = basename.split('_')
                if bits[-1].isdigit():
                    udim = bits[-1]

                channel = self.get_mask(path)

                name = bits[0]
                sorted_textures[name] = sorted_textures.get(name, {})
                udims, pattern = get_udims(path, udim)
                pattern = pattern.replace('*', '<UDIM>')
                sorted_textures[name][channel] = {'udims': udims,
                                                  'path': pattern,
                                                  'node': node,
                                                  'published_data': False}
        return sorted_textures

    def to_local(self):
        local_path = 'V:/SGD/assets/Main_Characters/LeoMesi/maya/sourceimages'
        for subset_name, subset_data in self.texture_data.items():
            for mask_name, mask_data in subset_data.items():
                if not mask_data['published_data']:
                    continue

                full_source_path = '%s/%s' %(mask_data['published_folder'], mask_data['relative_path'])
                full_target_path = '%s/%s' % (local_path, mask_data['relative_path'])
                full_target_folder = os.path.dirname(full_target_path)
                full_source_pattern = full_source_path.replace('<UDIM>', '*')

                all_texture_file_path = glob.glob(full_source_pattern)
                if not os.path.exists(full_target_folder):
                    os.makedirs(full_target_folder)

                for texture_file_path in all_texture_file_path:
                    basename = os.path.basename(texture_file_path)
                    full_target_path = '%s/%s' %(full_target_folder, basename)
                    logger.debug('to local: %s' % texture_file_path)
                    logger.debug('local path: %s' % full_target_path)
                    shutil.copy2(texture_file_path, full_target_path)
                    node_type = cmds.nodeType(mask_data['node'])
                    path_attribute = texture_node_types.get(node_type, '')
                    cmds.setAttr('%s.%s' % (mask_data['node'], path_attribute), full_target_path, type='string')


    def publish(self):

        textures_published = []

        for texture_name, data in self.texture_data.items():
            all_mask_data = {}

            published = True
            publish_data = None
            for mask_code, mask_data in data.items():
                if not mask_data['published_data']:
                    published = False
                    break
                else:
                    publish_data = mask_data['published_data']

            if published:
                textures_published.append(publish_data)
                continue

            for mask_code, mask_data in data.items():
                all_mask_data[mask_code] = {'node': mask_data['node'],
                                            'udims': mask_data['udims'],
                                            'source_path': mask_data['path']}

            extra_data = {'sg_metadata': {'channels': all_mask_data}}

            #publish_data = publish_entity.create_new_version(sg_connection, 'texture', task_code, 'fernando.vizoso',
            #                                                 asset_name=asset_name, subset=texture_name,
            #                                                 extra_data=extra_data, server=server_path, ignore_variant=True)


            parameters = {}

            parameters['task_code'] = self.task_code
            parameters['entity_name'] = self.asset_name
            parameters['comment'] = 'Auto publish textures'
            parameters['subset'] = texture_name
            parameters['metadata'] = extra_data


            publish_data = publish_helpers.create_new_publish(self.database, 'texture', parameters, server=self.server_path)


            publish_folder = publish_data.sg_published_folder
            files = {}
            for mask_code, mask_data in data.items():
                mask_folder = '%s/%s' % (publish_folder, mask_code)

                if not os.path.exists(mask_folder):
                    os.makedirs(mask_folder)
                pattern = mask_data['path']
                extension = pattern.split('.')[-1]

                relative_path = '%s/%s_<UDIM>.%s' % (mask_code, texture_name, extension)
                files[mask_code] = relative_path

                for udim in mask_data['udims']:
                    source = pattern.replace('<UDIM>', str(udim))

                    target = '%s/%s_%s.%s' % (mask_folder, texture_name, str(udim), extension)
                    shutil.copy2(source, target)

                cmds.setAttr('%s.fileTextureName' % mask_data['node'], target, type='string')

            self.database.sg.update('CustomEntity09', int(publish_data.id), {'sg_files': pformat(files)})

            textures_published.append({'id': int(publish_data.id), 'type': 'CustomEntity09'})

        shader_extra_data = {'sg_down_dependencies': textures_published}
        cmds.file(save=True, f=True)
        return shader_extra_data


def launch_publish(asset_name, task_code, project=None, server_path = None):
    if server_path is None:
        config_solver = config_manager.ConfigSolver(project=project)
    project_data = config_solver.get_config('project')

    server_path = project_data['paths']['server']

    texture_data = get_texture_data(server_path=server_path)
    connection = publish_entity.init_connection()
    output = publish_data(connection, asset_name, texture_data, task_code, server_path=server_path, project=project)
    return output


def check_published_textures():
    project = os.environ['PROJECT']
    config_mng = config_manager.ConfigSolver(project=project)
    project_config = config_mng.project_config

    publish_root = project_config['paths']['publish']['root']
    logger.debug('Publish server root: %s' % publish_root)

    publish_root = publish_root.replace('\\', '/').lower()

    all_files = cmds.ls(type='file')

    non_published_files = []
    for file_node in all_files:

        path = cmds.getAttr('%s.fileTextureName' % file_node)

        check_path = path.replace('\\', '/').lower()

        if not check_path.startswith(publish_root):
            logger.error('Texture file(s) not loaded from publish: %s' % path)
            non_published_files.append(path)
        else:
            logger.debug('Texture file() loaded from publish: %s' % path)

    if not non_published_files:
        logger.info('All the texture are loaded from a previous publish')
        return True

    return False


if __name__ == '__main__':
    import argparse
    logger.debug('running export shaders')
    parser = argparse.ArgumentParser(description='Shotgun event launcher')
    parser.add_argument('-s', '-source', '--source-file', dest='source', help='Source file path')
    parser.add_argument('-o', '-output', '--output-file', dest='output', help='Output file path')
    parser.add_argument('-n', '-name', '--asset-name', dest='asset_name', help='Asset name')
    parser.add_argument('-t', '-task', '--task-code', dest='task_code', help='Task name')


    args = parser.parse_args()

    import maya.standalone as standalone

    standalone.initialize(name='python')
    cmds.loadPlugin(r'C:\ProgramData\pipeline\taskbrowser_migration\databased\maya_assemblies\plugins\geo_assembly.py')
    cmds.loadPlugin('AbcImport')

    cmds.file(args.source, o=True)

    texture_data = get_texture_data()
    connection = publish_entity.init_connection()

    folder = os.path.dirname(args.output)
    publish_data(connection, args.asset_name, texture_data, args.task_code)
    if not os.path.exists(folder):
        os.makedirs(folder)
    cmds.file(rename=args.output)
    cmds.file(save=True, f=True)




