import getpass
import logging

from pprint import pprint

import packages_io
import shotgrid_lib.database as database

import launcher.lib.set_environment as set_environment

import publisher.lib.processes_manager as processes_manager

logger = logging.getLogger(__name__)

class PublishCmd():
    def __init__(self,
                 definition_type,
                 parameters,
                 project,
                 force_save=True):

        self.project = project
        self.force_save = force_save
        self.edit_step = parameters.get('edit_step', None)

        self.project_environment = set_environment.ProjectEnvironment(project=self.project)

        self.project_config = self.project_environment.project_config
        self.company_config = self.project_environment.company_config
        self.tools_config = self.project_environment.project_tools

        self.publisher_config = self.project_environment.get_config_file('publisher', module='publisher')
        self.definition = self.project_environment.get_config_file('definitions/%s' % definition_type, module='publisher', merge=False)

        self.publish_type = self.definition['publish_type']

        self.get_parameters = parameters
        self.definition_context = self.definition['definition_context']

        if self.force_save:
            packages_io.file_save()

        published_version = self.create_new_publish(parameters)
        manager = processes_manager.ProcessesManager(self.definition,
                                                     parameters,
                                                     published_version,
                                                     project=self.project,
                                                     project_config=self.project_config,
                                                     company_config=self.company_config,
                                                     tools_config=self.tools_config
                                                     )
        self.error = manager.error
        self.log_resume(published_version)

    def log_resume(self, published_version):
        logger.info('*' * 100)
        logger.info('Publish launched' )
        logger.info('*' * 100)
        logger.info(f'Name: {published_version.code}')
        logger.info(f'Version: {published_version.sg_version_number}')
        logger.info(f'Artist: {published_version.sg_artist_name}')
        logger.info(f'Publish type: {published_version.sg_publish_type_name}')
        logger.info(f'Folder: {published_version.sg_published_folder}')
        for tag, local_path in published_version.sg_files.items():
            print(f'File: {tag} - {local_path}')


    def get_latest_version(self, context_view, publish_type_view, variant_view, subset):
        version_number = 1
        logger.info('-' * 100)
        logger.info(context_view.type)
        logger.info(context_view.code)
        logger.info(publish_type_view.code)
        logger.info(variant_view.code)
        logger.info(subset)
        logger.info('-' * 100)
        context_view.precache_dependencies(fields=['sg_published_elements'])

        if context_view.sg_published_elements.empty:
            return version_number

        if context_view.type == 'Asset':
            published_view = context_view.sg_published_elements.find_with_filters(sg_asset=context_view,
                                                                                  sg_publish_type=publish_type_view,
                                                                                  sg_variant=variant_view,
                                                                                  sg_subset=subset)

        else:
            published_view = context_view.sg_published_elements.find_with_filters(sg_context=context_view,
                                                                                   sg_publish_type=publish_type_view,
                                                                                   sg_variant=variant_view,
                                                                                   sg_subset=subset)
        if not published_view.empty:
            latest = max(published_view)
            version_number = int(latest.sg_version_number + 1)
            logger.info('New version set to: %s' % version_number)
        else:
            logger.info('No last version found, initiating version list')

        return version_number

    def get_entity_view(self, parameters):

        entity_name = parameters['entity_name']
        entity_type = parameters.get('entity_type')
        asset_filters = [['code', 'is', entity_name],
                         ['project.Project.code', 'is', self.project]]

        self.database.query_sg_database(entity_type, filters=asset_filters, update=True)

        entity_view = self.database[entity_type].find_with_filters(code=entity_name,
                                                               single_item=True)


        return entity_view

    def get_publish_task(self, task_name, entity_view):
        task_filters = [['content', 'is', task_name],
                        ['project.Project.code', 'is', self.project],
                        ['entity', 'name_is', entity_view.code]]

        self.database.query_sg_database('Task', filters=task_filters, update=True)

        task = self.database['Task'].find_with_filters(content=task_name, entity=entity_view, single_item=True)
        task.precache_dependencies()

        task_status = task.sg_status_list
        if task_status in ['pld', 'wtg']:
            task.sg_status_list = 'ip'
            task.update_shotgun()

        return task

    def get_added_data(self, context_view):
        added_data = {}
        if self.definition_context == 'shot':
            added_data['sg_episode'] = context_view.sg_sequence.episode.code
            added_data['sg_sequence'] = context_view.sg_sequence.code

        elif self.definition_context == 'sequence':
            added_data['sg_episode'] = context_view.sg_episode.code

        elif self.definition_context == 'episode':
            added_data['sg_episode'] = context_view.code

        elif self.definition_context == 'asset':
            added_data['sg_asset_type'] = context_view.sg_asset_type.replace(' ', '_')

        return added_data

    def get_published_folder(self, publish_record, context_view, task_view, publish_type_view):
        publish_folder = publish_type_view.sg_path_pattern
        added_data = self.get_added_data(context_view)
        publish_folder = publish_record.solve_pattern(publish_folder, vars_dict=added_data)
        server = publish_type_view.sg_server
        server_path = self.project_config['paths'].get(server)
        published_folder = '%s/%s' % (server_path, publish_folder)
        return published_folder


    def get_publish_views(self, task_name):
        self.database = database.DataBase()
        self.database.fill(self.project, precatch=False)

        self.database.get_events()
        self.database.query_sg_database('CustomEntity10')
        publish_type_view = self.database['CustomEntity10'][self.publish_type]

        self.database.query_sg_database('HumanUser', filters=[])
        user_view = self.database['HumanUser'].find_with_filters(login=getpass.getuser(), single_item=True)

        self.database.query_sg_database('Step')
        entity_view = self.get_entity_view(self.get_parameters)
        task_view = self.get_publish_task(task_name, entity_view)

        return task_view, entity_view, user_view, publish_type_view

    def create_new_publish(self, parameters):
        print('*' * 50)
        logger.info(' %s Create new publish %s' %( '-' * 10, '-' * 10))

        task_name = parameters['task_code']
        subset = parameters.get('subset')

        task_view, entity_view, user_view, publish_type_view = self.get_publish_views(task_name)

        context_view = task_view.entity
        project_view = task_view.project
        variant_view = task_view.sg_variant
        if variant_view.empty:
            self.database.query_sg_database('CustomEntity11', filters=[['code', 'is', 'Master']])
            variant_view = self.database['CustomEntity11']['Master']


        version_number = self.get_latest_version(context_view, publish_type_view, variant_view, subset)
        code_pattern = publish_type_view.sg_code_pattern

        publish_record = database.Record('CustomEntity09')

        if self.definition_context == 'shot' or self.definition_context == 'sequence' or self.definition_context == 'episode':
            publish_record.sg_context = context_view
            publish_record.sg_asset = None
        elif self.definition_context == 'asset':
            publish_record.sg_asset = context_view
            publish_record.sg_context = None

        if self.edit_step:
            step_view = self.database['Step'][self.edit_step]
            publish_record.sg_step = step_view
        else:
            publish_record.sg_step = task_view.step

        publish_record.sg_publish_type = publish_type_view
        publish_record.sg_task = task_view
        publish_record.sg_version_number = version_number
        publish_record.description = parameters.get('comment')
        publish_record.project = project_view.as_shotgun()
        publish_record.sg_artist = user_view
        publish_record.sg_variant = variant_view
        publish_record.sg_subset = parameters.get('subset', variant_view.code)
        publish_record.sg_status_list = 'ip'
        dependencies = self.get_as_shotgrid(parameters.get('dependencies', {}))
        publish_record.sg_down_dependencies = dependencies
        publish_record.sg_hash = publish_record.generate_publish_hash(publish_type_view)
        publish_record.sg_published_folder = self.get_published_folder(publish_record,
                                                                       context_view,
                                                                       task_view,
                                                                       publish_type_view)

        publish_record.code = publish_record.solve_pattern(code_pattern)

        latest_version_view = self.database.append(publish_record)

        logger.info('%s New publish with id: %s %s %s' % ('-' * 10,
                                                          latest_version_view.id,
                                                          latest_version_view.code,
                                                          '-' * 10))
        return latest_version_view

    def get_as_shotgrid(self, inputs_data):

        logger.debug('Get dependencies data')

        new_data = []

        asset_list = []
        for asset_name, asset_data in inputs_data.items():
            for alias, alias_data in asset_data.items():

                for step, item_data in alias_data.items():

                    asset_list.append(item_data['code'])

        filters = [['code', 'in', asset_list]]
        if not asset_list:
            return []
        self.database.query_sg_database('Asset', filters=filters)
        self.database.query_sg_database('CustomEntity11')

        for asset, asset_data in inputs_data.items():
            for alias, alias_data in asset_data.items():
                logger.debug('alias name: %s' % alias)
                print('Get shotgrid publishs for %s' % alias)
                for step, item_data in alias_data.items():
                    if step == self.publish_type:
                        continue
                    print('Step: ', step)
                    if item_data['type'] == 'Asset' and item_data.get('asset_name'):
                        asset_name = item_data['code']
                        asset_view = self.database['Asset'][asset_name]
                        published_items = asset_view.sg_published_elements
                    else:
                        print('shot:', item_data['code'])
                        pprint(item_data)
                        shot_name = item_data['code']
                        print('set shot_name to: ', shot_name)
                        shot_view = self.database['Shot'][shot_name]
                        published_items = shot_view.sg_published_elements

                    if published_items.empty:
                        continue

                    step_view = self.database['Step'][step]
                    if step_view.empty or 'variant' not in item_data or 'version' not in item_data:

                        continue

                    find_filters = {'sg_step': step_view}
                    logger.debug('variant name: %s' % item_data['variant'])

                    variant_view = self.database['CustomEntity11'][item_data['variant']]

                    find_filters['sg_variant'] = variant_view

                    if isinstance(item_data['version'], int) or item_data['version'].isdigit():
                        find_filters['sg_version_number'] = int(item_data['version'])

                        published_item = published_items.find_with_filters(single_item=True,
                                                                           **find_filters
                                                                           )
                    elif item_data['version'] == 'latest':
                        published_item = published_items.find_with_filters(single_item=True,
                                                                           **find_filters
                                                                           )
                    elif item_data['version'] == 'approved':
                        find_filters['sg_status_list'] = 'apr'
                        published_item = published_items.find_with_filters(single_item=True,
                                                                           **find_filters
                                                                           )
                    elif item_data['version'] == 'recommended':
                        base_filter = find_filters.copy()
                        find_filters['sg_status_list'] = 'apr'
                        published_item = published_items.find_with_filters(single_item=True,
                                                                           **find_filters
                                                                           )
                        if published_item.empty:
                            published_item = published_items.find_with_filters(single_item=True,
                                                                               **base_filter
                                                                               )

                    if not published_item.empty:
                        new_data.append({'type': 'CustomEntity09', 'id': published_item.id})
                    else:
                        print('cant find:')
                        pprint(item_data)
        return new_data


if __name__ == '__main__':

    definition_type = 'Model'
    definition_type = 'Render'

    parameters = {'task_code':'Model',
                  'entity_name': 'LeoMesi',
                  'source_path': 'C:/Users/fernando.vizoso/Documents/maya/projects/default/scenes/LeoMesi.ma',
                  'geometry_high': 'render',
                  'geometry_low': 'proxy',
                  }

    parameters = {'task_code': 'Render',
                  'entity_name': 's00_ep01_sq100_sh100',
                  #'source_path': 'C:/Users/fernando.vizoso/Documents/maya/projects/default/scenes/LeoMesi.ma',
                  'renderpass': 'Characters',
                  'start_frame': 1001,
                  'end_frame': 1198,
                  'job_frames': 198,
                  'subset': 'Characters'
                  }

    publish = PublishCmd(definition_type, parameters, 'tpt')

    parameters['renderpass'] = 'FX'
    parameters['subset'] = 'FX'
