import logging
import re

import publisher.lib.processes_node as processes_node

logger = logging.getLogger()


class ProcessesManager():

    def __init__(self,
                 definition,
                 parameters,
                 published_version,
                 project=None,
                 project_config=None,
                 company_config=None,
                 tools_config=None,
                 ):

        self.project = project

        if project_config:
            self.project_config = project_config
        else:
            self.project_config = {}

        if company_config:
            self.company_config = company_config
        else:
            self.company_config = {}

        if tools_config:
            self.tools_config = tools_config
        else:
            self.tools_config = {}

        self.error = False
        self.definition = definition.copy()
        self.dependencies = self.get_processes_dependencies()

        self.parameters = parameters.copy()
        self.constants = self.definition.get('constants', {})
        self.published_version = published_version
        self.end_node = self.get_finish_publish_node()
        self.nodes = {}

        self.build_node_tree()

    def as_path(self, inputs_data):
        new_data = {}
        for pipeline_step, item_data in inputs_data.items():
            new_data[item_data['pipeline_step']] = new_data.get(item_data['pipeline_step'], [])
            new_data[item_data['pipeline_step']].append(item_data['path'])

        clean_dict = {}
        for key, value in new_data.items():
            if len(value) == 1:
                clean_dict[key.lower()] = value[0]
            else:
                clean_dict[key.lower()] = value
        return clean_dict

    def get_str_connections(self, argument_line):
        
        search_pattern = r'<[.0-9a-z_]*>' 
            
        all_matchs = re.findall(search_pattern, argument_line)
        nodes = []
        for match in all_matchs:
            match = match[1:-1]
            if match.find('.') > -1:
                node = match.split('.')[0]
                nodes.append(node)
        return nodes


    def get_connections(self, arguments):
        connections = []
        if isinstance(arguments, str):
            return self.get_str_connections(arguments)
            
        elif isinstance(arguments, list):
            for item in arguments:
                connections += self.get_connections(item)

        elif isinstance(arguments, dict):
            for item, item_value in arguments.items():
                connections += self.get_connections(item_value)

        return connections

    def get_processes_dependencies(self):
        dependencies = {}
        ignore_nodes = ['project_config', 'dependency_publish', 'asset']
        for node_name, node_data in self.definition['element_process'].items():
            arguments = node_data['args']
            inputs = self.get_connections(arguments)

            clean_inputs = [input_node for input_node in inputs if input_node not in ignore_nodes]
            clean_inputs = [input_node for input_node in clean_inputs if input_node != node_name]
            dependencies[node_name] = {'inputs': clean_inputs, 'outputs':[]}
        
        for node_name, node_data in dependencies.items():
            for input_node in node_data.get('inputs'):

                dependencies[input_node]['outputs'].append(node_name)

        return dependencies

    def build_node_tree(self):
        logger.debug('Building node tree..')
        root_node_data = self.definition['element_process'][self.end_node]
        logger.debug('Root node: %s' % self.end_node)
        self.root_node = processes_node.ProcessBuilder(self.end_node,
                                                       root_node_data,
                                                       self,
                                                       node_type=root_node_data['type'],
                                                       published_version=self.published_version
                                                       )
        logger.debug('Building node tree finished')

        self.error = self.root_node.error


    def get_farm_jobs(self):
        farm_jobs = []
        for job_name, job_data in self.definition['element_process'].items():
            if job_data['type'] == 'run_command':
                farm_jobs.append(job_name)

        return farm_jobs

    def add_finish_node(self, all_roots, node_name='finish_publish'):

        finish_node_dict = {}

        finish_node_dict['dependencies'] = all_roots

        constants = self.definition

        arguments = {'project': self.project}

        if self.get_farm_jobs():
            finish_node_dict['type'] = 'run_command'
            finish_node_dict['args'] = {'command_line': {'command': 'publisher/resources/scripts/finish_publish.py'},
                                        'application': 'python'
                                        }

            arguments['shotgrid-id'] = '<shotgrid_id>'
            if 'version_movie' in constants:
                arguments['movie'] = constants['version_movie']

            if 'start_frame' in self.parameters:
                arguments['start-frame'] = self.parameters['start_frame']
            elif 'start_frame' in constants:
                arguments['start-frame'] = constants['start_frame']

            if 'end_frame' in self.parameters:
                arguments['end-frame'] = self.parameters['end_frame']

            elif 'end_frame' in constants:
                arguments['end-frame'] = constants['end_frame']
            finish_node_dict['args']['command_line']['arguments'] = arguments

        else:
            finish_node_dict['type'] = 'run_python_script'
            finish_node_dict['args'] = {'script': {'module': 'publisher.resources.scripts.finish_publish',
                                                   'function': 'finish_publish'},
                                        }

            arguments['shotgrid_id'] = '<shotgrid_id>'
            if 'version_movie' in constants:
                arguments['movie'] = constants['version_movie']
            if 'start_frame' in self.parameters:
                arguments['start-frame'] = self.parameters['start_frame']
            elif 'start_frame' in constants:
                arguments['start_frame'] = constants['start_frame']

            if 'end_frame' in self.parameters:
                arguments['end-frame'] = self.parameters['end_frame']
            elif 'end_frame' in constants:
                arguments['end_frame'] = constants['end_frame']
            finish_node_dict['args']['script']['arguments'] = arguments


        self.definition['element_process'][node_name] = finish_node_dict
        
        self.dependencies[node_name] = {'inputs': all_roots, 'outputs': []}
        return finish_node_dict
         

    def get_finish_publish_node(self, node_name='finish_publish'):
        all_roots = []
        for current_node_name, node_data in self.dependencies.items():
            if not node_data['outputs']:
                all_roots.append(current_node_name)

        if not all_roots:
            logger.error('Can\'t find the end node')
            return None

        finish_node = self.add_finish_node(all_roots, node_name=node_name)

        return node_name
