import library.core.config_manager as config_manager
import os
import shutil
from pathlib import Path
from datetime import datetime

class FileHandler:
    def __init__(self, path):
        self.file = None
        self.path = path
        self.open_file()

    def open_file(self):
        try:
            self.file = open(self.path, 'a')
        except (FileNotFoundError, PermissionError, Exception) as e:
            self.file = None
            return

    def write(self, content):
        if self.file:
            self.file.write(content + "\n")

    def close_file(self):
        if self.file:
            self.file.close()
            self.file = None

def episodePack(project, database, episode):

    config_solver = config_manager.ConfigSolver(project=project)
    project_data = config_solver.get_config('project')
    server_path = project_data['paths']['pack_server']
    if not server_path or not episode:
        return False, "Server Pack : path not found !"

    episode_path = os.path.join(server_path, episode)
    if not os.path.exists(episode_path):
        os.makedirs(episode_path)

    # Feedback file
    file_handler = FileHandler(os.path.join(episode_path, "EpisodePack.txt"))
    current_datetime = datetime.now()
    formatted_datetime = current_datetime.strftime("%Y-%m-%d at %H:%M:%S")
    handler_line="EPISODE: " + episode + "\nDATE: " + formatted_datetime
    file_handler.write(handler_line)

    assets = database['Asset']
    for asset in assets:
        for ep in asset.episodes:
            if ep.code == episode:
                copy_asset(database, episode_path, asset, file_handler)

    #
    file_handler.close_file()

    return True, file_handler.path

def copy_asset(database, episode_path, asset_view, file_handler):

    asset_view.precache_dependencies(fields='sg_published_elements')

    handler_line="************** Asset: " + asset_view.code + " **************"
    file_handler.write(handler_line)

    #TODO : brief / design / concept / color

    #modeling
    step_modeling = database['Step']['Modeling'].find_with_filters(entity_type='Asset', single_item=True)
    copy_turntable(database, episode_path, asset_view, step_modeling, file_handler)

    #shading
    step_shading = database['Step']['Shading'].find_with_filters(entity_type='Asset', single_item=True)
    copy_turntable(database, episode_path, asset_view, step_shading, file_handler)

    return

def copy_turntable(database, episode_dest_path, asset_view, step_view, file_handler):

    dest_asset = asset_view.sg_asset_type + '/' + asset_view.code
    dest_folder_path = os.path.join(episode_dest_path, dest_asset)

    #find the latest approved version of the step
    asset_versions = database['CustomEntity09'].find_with_filters(sg_asset=asset_view, sg_step=step_view, sg_status_list='apr')
    if not asset_versions:
        handler_line = 'No approved version for ' + asset_view.code
        file_handler.write(handler_line)
        return

    for version in asset_versions:
        handler_line = version.code
        file_handler.write(handler_line)

        turntable = version.sg_files.get('turntable', 'None')
        if turntable == 'None':
            handler_line = '   Turntable not found'
            file_handler.write(handler_line)
            continue

        # is src file exist ?
        src_turntable_path = os.path.join(version.sg_published_folder, turntable)
        if not os.path.isfile(src_turntable_path):
            handler_line = "   Src file does not exist :" + src_turntable_path
            file_handler.write(handler_line)
            continue

        # create dest directory
        if not os.path.exists(dest_folder_path):
            os.makedirs(dest_folder_path)

        # rename dest_file with "step.filename.version"
        file_path = Path(src_turntable_path)
        file_name = file_path.name
        file_name_without_ext, file_extension = os.path.splitext(file_name)
        new_file_name = f"{version.code}{file_extension}"
        new_full_file_path = os.path.join(dest_folder_path, new_file_name)

        try:
            if os.path.exists(new_full_file_path):
                handler_line = "   The file " + new_full_file_path + " already exists."
                file_handler.write(handler_line)
                # os.remove(new_full_file_path)
                continue
            shutil.copy2(src_turntable_path, new_full_file_path)
            handler_line = "   Copy success " + src_turntable_path + " in " + new_full_file_path
            file_handler.write(handler_line)
        except FileNotFoundError:
            file_handler.write("   Src file does not exist")
        except PermissionError:
            file_handler.write("   Permission copy denied.")
        except Exception as e:
            handler_line = "   Copy error: " + e
            file_handler.write(handler_line)

    return
