import library.core.config_manager as config_manager
import os
import sys
import importlib
import packages_io
from PySide import QtCore, QtGui, QtWidgets

import shotgrid_lib.database as database
import production_tools.production_tools_pack_episode as packEpisode

class PackWindow(packages_io.MainWindow):
    def __init__(self, project):
        super(PackWindow, self).__init__('Pack Window', style_sheet='pack_window')
        self.apply_style(self)
        self.setWindowTitle("Episode Pack")

        icon_path = "icons/package.png"
        self.setWindowIcon(QtGui.QIcon(icon_path))

        self.project = project
        self.database_from_project(project)

        vlayout = QtWidgets.QVBoxLayout()

        label_project_name = QtWidgets.QLabel()
        #label_project_name.setText(self.project.name)
        label_project_name.setText("Project.Name from DATABASE")
        label_project_name.setEnabled(False)

        layout_episode = QtWidgets.QHBoxLayout()
        label = QtWidgets.QLabel("Episode")
        label.setFixedWidth(50)
        self.episode_combo = QtWidgets.QComboBox()
        layout_episode.addWidget(label)
        layout_episode.addWidget(self.episode_combo)

        self.label_episode_description = QtWidgets.QLabel()
        self.label_episode_description.setEnabled(False)

        layout_actions = QtWidgets.QHBoxLayout()
        button_build = QtWidgets.QPushButton("Build")
        button_cancel = QtWidgets.QPushButton("Cancel")
        layout_actions.addWidget(button_build)
        layout_actions.addWidget(button_cancel)

        vlayout.addWidget(label_project_name, alignment=QtCore.Qt.AlignCenter)
        vlayout.addLayout(layout_episode)
        vlayout.addWidget(self.label_episode_description, alignment=QtCore.Qt.AlignCenter)
        vlayout.addLayout(layout_actions)

        self.init_episode_combo()

        self.episode_combo.currentIndexChanged.connect(self.on_episode_changed)
        button_build.clicked.connect(self.on_build_clicked)
        button_cancel.clicked.connect(self.on_cancel_clicked)

        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.addLayout(vlayout)

        central_widget = QtWidgets.QWidget()
        central_widget.setLayout(self.main_layout)

        self.setCentralWidget(central_widget)
        self.resize(300, 100)
        return

    def database_from_project(self, project):
        self.project = project
        self.database = database.DataBase()
        self.database.fill(self.project, True)
        return

    def refresh_episode_description(self):

        episode_code = self.episode_combo.currentText()
        episode = self.database['Episode'][episode_code]

        #TODO : description in scheme.yaml ok for database
        #self.label_episode_description.setText(episode.description)
        self.label_episode_description.setText(episode.code)
        return

    def init_episode_combo(self):

        self.episode_combo.blockSignals(True)
        self.episode_combo.clear()

        episode_view = self.database['Episode']
        for episode in episode_view:
            if episode.empty:
                continue
            print(episode.code)
            self.episode_combo.addItems([episode.code])

        last_index = self.episode_combo.count() - 1
        if last_index >= 0:
            self.episode_combo.setCurrentIndex(last_index)

        self.refresh_episode_description()

        self.episode_combo.blockSignals(False)
        return

    def on_episode_changed(self, index):
        self.refresh_episode_description()
        return

    def on_build_clicked(self):

        episode = self.episode_combo.currentText()

        config_solver = config_manager.ConfigSolver(project=self.project)
        project_data = config_solver.get_config('project')
        server_path = project_data['paths']['pack_server']

        message_question = f'Pack creation for the <span style="color: #2dbeff;">{episode}</span> episode<br>in: <span style="color: #2dbeff;">{server_path}</span><br><br>Do you want to continue?'
        response_question = QtWidgets.QMessageBox.question(
            self,
            "Build",
            message_question,
            QtWidgets.QMessageBox.Ok | QtWidgets.QMessageBox.Cancel
        )
        if response_question == QtWidgets.QMessageBox.Ok:
            result, message = packEpisode.episodePack(self.project, self.database, episode)

            msg_box = QtWidgets.QMessageBox(self)
            msg_box.setIcon(QtWidgets.QMessageBox.Information)
            msg_box.setWindowTitle("Pack info")
            msg_box.setText(message)
            msg_box.setStandardButtons(QtWidgets.QMessageBox.Ok)

            if result:
                if os.path.exists(message):
                    url_file_path = self.convert_to_file_url(message)
                    self.clickable_path = f'<a href="{url_file_path}" style="color: #2dbeff;">{url_file_path}</a>'
                    text = f'Episode pack built! <br>Click here to find the result: <br>{self.clickable_path}'
                    msg_box.setText(text)
                    msg_box.setTextInteractionFlags(QtCore.Qt.TextBrowserInteraction)
                    msg_box.buttonClicked.connect(self.open_link)
            response_info = msg_box.exec()
            if response_info:
                self.close()
        return

    def convert_to_file_url(self, path):
        path = path.replace('\\', '/')
        return f"file:{path}"

    def open_link(self, button):
        QtGui.QDesktopServices.openUrl(QtCore.QUrl.fromLocalFile(self.clickable_path))
        return

    def on_cancel_clicked(self):
        window.close()
        return


def open_tools_pack(project):
    print('open pack tools', project)
    tool_window = PackWindow(project=project)
    tool_window.show()


if __name__ == '__main__':

    import library.core.parser_function as parser_function
    importlib.reload(parser_function)
    print(sys.argv)
    QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_ShareOpenGLContexts)
    app = QtWidgets.QApplication(sys.argv)
    window = parser_function.parse_function(PackWindow)
    window.show()
    app.exec_()