from pprint import pprint
from PySide import QtWidgets, QtCore, QtGui

import shotgrid_lib.lib.database_card as database_card

import library.ui.labeled as labeled


class BreakdownCard(database_card.BaseCard):
    clicked = QtCore.Signal(int)

    def __init__(self, view, parent=None):
        super(BreakdownCard, self).__init__(view, parent=parent, add_thumbnail=False)
        self.max_published_geometry = None
        self.height = 38
        # self.setFixedHeight(self.height)
        self.use_alias = False
        self.view = view
        asset = self.view.sg_asset
        self.publish_elements = asset.sg_published_elements


        self.publish_widgets = {}

        # self.code_widget = labeled.LabeledText(view.code, 'Breakdown', as_label=True)

        self.breakdown_layout = QtWidgets.QHBoxLayout()

        self.asset_name_widget = labeled.LabeledText(view.sg_asset_name, 'Asset', as_label=True)
        self.asset_name_widget.label_widget.setFixedWidth(40)
        self.asset_name_widget.value_widget.setFixedWidth(100)

        # self.data_layout.addWidget(self.code_widget)
        self.breakdown_layout.addWidget(self.asset_name_widget)

        if not view.sg_alias:
            self.use_alias = True
            self.instance_widget = labeled.LabeledText(str(view.sg_instance), 'Instance', as_label=True)
            self.instance_widget.label_widget.setFixedWidth(60)
            self.instance_widget.value_widget.setFixedWidth(100)

            self.breakdown_layout.addWidget(self.instance_widget)
            self.alias_widget = None
        else:
            self.alias_widget = labeled.LabeledText(str(view.sg_alias), 'Alias', as_label=True)
            self.alias_widget.label_widget.setFixedWidth(60)
            self.alias_widget.value_widget.setFixedWidth(100)
            self.breakdown_layout.addWidget(self.alias_widget)
            self.instance_widget = None

        self.geometry_variant_widget = labeled.LabeledText(view.sg_geometry_variant_name, 'Geometry', as_label=True)
        self.geometry_variant_widget.label_widget.setFixedWidth(80)
        self.geometry_variant_widget.value_widget.setFixedWidth(100)

        self.shading_variant_widget = labeled.LabeledText(view.sg_shading_variant_name, 'Shading', as_label=True)
        self.shading_variant_widget.label_widget.setFixedWidth(80)
        self.shading_variant_widget.value_widget.setFixedWidth(100)

        self.breakdown_layout.addWidget(self.geometry_variant_widget)
        self.breakdown_layout.addWidget(self.shading_variant_widget)

        self.data_layout.addLayout(self.breakdown_layout)
        self.children_widget = QtWidgets.QWidget(parent=self)
        self.children_layout = QtWidgets.QVBoxLayout()
        self.children_layout.setSpacing(3)
        self.children_layout.setContentsMargins(20, 0, 0, 0)

        self.children_widget.setLayout(self.children_layout)

        self.data_layout.addWidget(self.children_widget)
        self.data_layout.addStretch(1)

        self.breakdown_layout.addStretch(1)

    def fill_child_type(self, pipeline_step, maya_node):
        modeling_step = self.view._database['Step'][pipeline_step]

        if modeling_step.empty:
            return
        publish_step = self.publish_elements.find_with_filters(sg_step=modeling_step,
                                                               sg_delete=False,
                                                               sg_complete=True
                                                               )
        if publish_step.empty:
            return

        publish_widget = PublishCard(publish_step, maya_node=maya_node, asset=self.view)

        self.children_layout.addWidget(publish_widget)
        self.publish_widgets[pipeline_step] = publish_widget
        self.height += 36
        return publish_widget


class PublishCard(database_card.BaseCard):
    clicked = QtCore.Signal(int)

    def __init__(self, view, maya_node, asset, parent=None):
        super(PublishCard, self).__init__(view, parent=parent, add_thumbnail=False)
        self.view = view
        self.maya_node = maya_node
        self.asset = asset
        self.selected_published = max(self.view)

        self.setFixedHeight(30)
        self.data_layout.setSpacing(0)
        self.data_layout.setContentsMargins(10, 0, 0, 0)
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        short_maya_node = self.maya_node.split('|')[-1]

        self.maya_node_widget = labeled.LabeledText(short_maya_node, 'Node', as_label=True)
        self.maya_node_widget.setToolTip(self.maya_node)
        self.maya_node_widget.label_widget.setFixedWidth(60)
        self.maya_node_widget.value_widget.setFixedWidth(200)

        self.pipeline_step_widget = QtWidgets.QLabel(self.selected_published.sg_step_name)
        self.pipeline_step_widget.setFixedWidth(100)

        variants = self.view.single_values('sg_variant')

        self.variant_widget = labeled.LabeledCombo('', 'variant', values=variants)
        self.variant_widget.currentIndexChanged.connect(self.update_versions)
        self.variant_widget.label_widget.setFixedWidth(60)
        self.variant_widget.value_widget.setFixedWidth(100)

        versions = ['%03d' % ver for ver in sorted(self.view.sg_version_number)]
        self.version_widget = labeled.LabeledCombo('', 'Version', values=versions)
        self.version_widget.currentIndexChanged.connect(self.update_status)

        self.version_widget.label_widget.setFixedWidth(60)
        self.version_widget.value_widget.setFixedWidth(60)

        if isinstance(self.selected_published.sg_status_list, str):
            status = str(self.selected_published.sg_status_list)
        else:
            status = 'n/a'

        self.status_widget = labeled.LabeledText(status, 'Status', as_label=True)

        self.status_widget.label_widget.setFixedWidth(60)
        self.status_widget.value_widget.setFixedWidth(60)

        self.update_button = QtWidgets.QPushButton('Set recommended')
        self.update_button.clicked.connect(self.set_recomended)
        self.update_button.setFixedWidth(100)

        self.main_layout.addWidget(self.maya_node_widget)
        self.main_layout.addWidget(self.pipeline_step_widget)
        self.main_layout.addWidget(self.variant_widget)

        self.main_layout.addWidget(self.version_widget)
        self.main_layout.addWidget(self.status_widget)
        self.main_layout.addWidget(self.update_button)

        self.main_layout.addStretch(1)


    def set_recomended(self):
        variant_view = self.asset.sg_geometry_variant
        pipeline_step = self.pipeline_step_widget.text()
        pipeline_view = self.database['Step'][pipeline_step]
        publish_versions = self.view.find_with_filters(sg_variant=variant_view,
                                                       sg_step=pipeline_view)

        recomended_version = self.get_recomended_version(publish_versions)
        self.setValue(recomended_version)

    def get_recomended_version(self, approved_list):
        approvec_version = approved_list.find_with_filters(sg_status_list='cmpt')
        if not approvec_version.empty:
            self.exists_approved = True
            return max(approvec_version)

        self.exists_approved = False
        return max(approved_list)

    def update_status(self):
        pipeline_step = self.pipeline_step_widget.text()
        pipeline_view = self.database['Step'][pipeline_step]
        variant_name = self.variant_widget.getValue()
        variant_view = self.database['CustomEntity11'][variant_name]

        version_number_str = self.version_widget.getValue()
        if not version_number_str:
            return
        version_number = int(version_number_str)
        self.recomended_versions = self.view.find_with_filters(sg_variant=variant_view,
                                                               sg_step=pipeline_view,
                                                               sg_version_number=version_number,
                                                               single_item=True)

        if isinstance(self.recomended_versions.sg_status_list, str):
            status = str(self.recomended_versions.sg_status_list)
        else:
            status = 'n/a'
        self.status_widget.setValue(status)

        if status == 'cmpt':
            self.status_widget.value_widget.setStyleSheet("color: green;")
            self.update_button.setHidden(True)
        else:
            self.status_widget.value_widget.setStyleSheet("color: red;")
            self.update_button.setHidden(False)

    def setValue(self, publish):
        self.set_version = publish
        self.variant_widget.setValue(publish.sg_variant_name)
        self.version_widget.setValue('%03d' % publish.sg_version_number)

    def update_versions(self):
        pipeline_step = self.pipeline_step_widget.text()
        pipeline_view = self.database.entity_view('Step')[pipeline_step]

        variant_name = self.variant_widget.getValue()
        variant_view = self.database.entity_view('CustomEntity11')[variant_name]

        publish_versions = self.view.find_with_filters(sg_variant=variant_view,
                                                       sg_step=pipeline_view)
        if publish_versions.empty:
            self.version_widget.setValues([])
            return

        self.versions = ['%03d' % ver for ver in sorted(publish_versions.sg_version_number)]
        self.version_widget.setValues(self.versions)
        self.recomended_versions = self.get_recomended_version(publish_versions)
        self.version_widget.setValue('%03d' % self.recomended_versions.sg_version_number)

