import sys
import importlib
from pprint import pprint

from PySide import QtWidgets, QtCore

import packages_io

import maya_scene_manager.lib.ui.publish_card as publish_card
import shotgrid_lib.lib.progress_query_widget as progress_query

import maya_scene_manager.lib.scene_manager_core as scene_manager_core

importlib.reload(packages_io)

importlib.reload(scene_manager_core)
importlib.reload(publish_card)
importlib.reload(progress_query)

#import library.ui.window as maya_window


class CardContainer(QtWidgets.QWidget):
    on_click = QtCore.Signal(int, bool)

    def __init__(self, attribute, view=None, parent=None, widget_class=None):
        super(CardContainer, self).__init__(parent=parent)

        self.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.card = parent
        self.view = view
        self.attribute = attribute
        widget_class.config_card()
        self.widget_class = widget_class

        self.selected_widget_position = None
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(0)
        self.main_layout.setContentsMargins(0, 0, 0, 0)
        self.visible = False
        self.widgets = {}

        self.main_widget = QtWidgets.QFrame()
        self.main_widget.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.scroll_layout = QtWidgets.QVBoxLayout()
        self.scroll_layout.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignTop)

        self.scroll_layout.setSpacing(4)
        self.scroll_layout.setContentsMargins(5, 5, 5, 5)

        self.main_widget.setLayout(self.scroll_layout)

        self.scroll_area = QtWidgets.QScrollArea(self)
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setWidget(self.main_widget)
        self.scroll_area.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.main_layout.addWidget(self.scroll_area)
        self.setLayout(self.main_layout)

    def clean(self):
        #while ((child := self.scroll_layout.takeAt(0)) != None):
        #    child.widget().deleteLater()
         for child in self.widgets.values():
             child.widget().deleteLater()
         self.widgets = {}

    def update_cards(self):
        if self.widgets:
            self.clean()

        if self.attribute is None:
            loop = self.view
        else:
            loop = getattr(self.view, self.attribute)

        for index, item in enumerate(loop):
            self.widgets[item.id] = self.widget_class(item, parent=self)
            self.scroll_layout.addWidget(self.widgets[item.id])
            self.widgets[item.id].clicked.connect(self.clicked)
            self.update()

    def clicked(self, selected_id):
        current_item = self.widgets[selected_id]
        if not current_item.is_open():
            for item_id, item in self.widgets.items():
                item.setHidden(False)
                item.close_children()

        else:
            for item_id, item in self.widgets.items():
                if item_id == selected_id:
                    item.setHidden(False)
                    item.open_children()
                else:
                    item.setHidden(True)


class SceneManagerWindow(packages_io.MainWindow):
    def __init__(self, shot=None, asset=None, project='sgd'):
        super().__init__( title='Scene manager')

        self.setMinimumSize(1000, 500)

        self.project = project
        self.shot = shot
        self.asset = asset

        if self.shot:
            self.entity = self.shot
            self.entity_type = 'Shot'
        elif self.asset:
            self.entity = self.asset
            self.entity_type = 'Asset'
        else:
            return
        self.main_widget = QtWidgets.QWidget()
        self.main_layout = QtWidgets.QVBoxLayout()

        self.container_widget = CardContainer(None, parent=self, widget_class=publish_card.BreakdownCard)

        self.button_layout = QtWidgets.QHBoxLayout()

        self.apply_button = QtWidgets.QPushButton('Apply')
        self.apply_button.setFixedWidth(100)
        self.apply_button.clicked.connect(self.apply_changes)

        self.cancel_button = QtWidgets.QPushButton('Cancel')
        self.cancel_button.setFixedWidth(100)

        self.main_layout.addWidget(self.container_widget)
        self.button_layout.addStretch(1)

        self.button_layout.addWidget(self.apply_button)
        self.button_layout.addWidget(self.cancel_button)

        self.main_layout.addLayout(self.button_layout)
        self.main_widget.setLayout(self.main_layout)
        self.setStyleSheet("background-color: rgb(10, 10, 10);")



        self.apply_style(self)

        self.scene_manager = scene_manager_core.SceneManagerCore(self.entity_type, self.shot, self.project)

        self.progress_query = progress_query.ProgressQuery(self.project, parent=self)
        self.progress_query.set_precache_entities(['Asset', 'Step', 'CustomEntity11'])
        self.progress_query.precache_dependencies = {'Asset': ['sg_published_elements']}
        self.progress_query.finished.connect(self.fill())
        self.database = self.progress_query.database

        self.setCentralWidget(self.main_widget)

    def start(self):
        self.progress_query.start()

    def get_root_widget(self, asset_name):
        for widget in self.container_widget.widgets.values():
            if widget.view.sg_asset_name == asset_name:
                return widget

    def apply_changes(self):
        for asset_name, breakdown_widget in self.container_widget.widgets.items():
            for step_name, publish_widget in breakdown_widget.publish_widgets.items():
                current_publish = publish_widget.recomended_versions
                maya_node = publish_widget.maya_node
                self.scene_manager.check_update_node(maya_node, current_publish)

    def fill_shot(self, scene_data):
        shot_view = self.database['Shot'][self.shot]
        breakdown = shot_view.sg_breakdowns
        asset_list = breakdown.sg_asset_name

        asset_filters = [['code', 'in', asset_list]]
        self.database.query_sg_database('Asset', filters=asset_filters)

        list_codes = []
        for asset in breakdown.sg_asset:
            elements = asset.sg_published_elements_name
            list_codes += elements

        publish_filters = [['code', 'in', list_codes]]
        self.database.query_sg_database('CustomEntity09', filters=publish_filters)

        self.container_widget.view = breakdown

        QtWidgets.QApplication.restoreOverrideCursor()

        self.container_widget.update_cards()

        for asset_name, asset_publish in scene_data.items():

            asset_widget = self.get_root_widget(asset_name)
            for pipeline_step, publish_data in asset_publish.items():
                publish_widget = asset_widget.fill_child_type(pipeline_step, publish_data['maya_node'])
                publish_data = publish_data['publish_view']

                publish_widget.setValue(publish_data)
                publish_widget.update_status()

    def fill_asset(self, scene_data):

        QtWidgets.QApplication.restoreOverrideCursor()
        self.container_widget.update_cards()

        asset_widget = self.get_root_widget(self.asset)
        for asset_name, asset_entities in scene_data.items():
            for isntance, publish_data in asset_entities['instances'].items():
                publish_widget = asset_widget.fill_child_type(pipeline_step, publish_data['maya_node'])

                publish_widget.setValue(publish_data)
                publish_widget.update_status()

    def fill(self):
        print('FILL')

        scene_data = self.scene_manager.inspect_scene()

        if self.entity_type == 'Asset':
            self.fill_asset(scene_data)
        else:
            self.fill_shot(scene_data)

def open_scene_manager(task=None):
    database = task._database
    entity = task.entity

    project_view = task.project
    project = project_view.code

    print(project, entity)
    if entity.type == 'Shot':
        window_test = SceneManagerWindow(shot=entity.code, project=project)
    else:
        window_test = SceneManagerWindow(asset=entity.code, project=project)

    window_test.start()
    window_test.show()

if __name__ == '__main__':
    # thread = threaded_database_query('sgd', ['Shot', 'Sequence'])

    app = QtWidgets.QApplication(sys.argv)
    window_test = SceneManagerWindow('s00_ep00_sq010_sh020', 'sgd')
    window_test.show()
    window_test.start()
    app.exec_()