import os
import sys
import logging
import importlib
from collections import OrderedDict

from pprint import pprint
from PySide import QtCore, QtGui, QtWidgets

import library.core.dynamic_run as dynamic_run
import library.core.config_manager as config_manager
import shotgrid_lib.database as database
import usd.lib.usd_manager as usd_manager
import packages_io

importlib.reload(usd_manager)

logger = logging.getLogger(__name__)
logger.setLevel(logging.WARNING)

class SceneTreeItem(QtWidgets.QTreeWidgetItem):
    def __init__(self, layer_name, pipeline_step, config, active=True, parent=None):
        super().__init__()
        self.setFlags(self.flags() & QtCore.Qt.ItemIsSelectable)

        self.tree_widget = parent
        self.layer_name = layer_name
        self.pipeline_step = pipeline_step
        self.editable = self.tree_widget.editable
        self.all_variants = {}
        self.config = config
        print('self published_elements')
        self.published_elements = config.get('published_items', {})
        print(self.published_elements)
        self.current_variant = config.get('variant', 'Master')
        self.current_version = config.get('version_number', 'None')
        self.status = 5
        self.approved_version = None
        self.latest_version = None
        self.recommended_version = None
        self.init_variant = config.get('variant', 'Master')
        self.init_version = config.get('version_number', 'None')

        self.entity_type = self.config.get('type')
        self.asset_name = config.get('code')
        self.setText(0, layer_name)
        if self.editable:
            self.variant_widget = QtWidgets.QComboBox()
            self.version_widget = QtWidgets.QComboBox()
        else:
            self.variant_widget = QtWidgets.QLabel()
            self.version_widget = QtWidgets.QLabel()

        self.active = active
        if self.editable:
            if self.active:
                self.setCheckState(0, QtCore.Qt.Checked)
            else:
                self.setCheckState(0, QtCore.Qt.Unchecked)

    def set_variants(self, variants_data):
        print('set variants')
        pprint(variants_data)
        self.all_variants = variants_data
        variants = list(self.all_variants.keys())
        versions = []
        for int_version in self.all_variants[self.current_variant].get('versions', []):
            if isinstance(int_version, str):
                versions.append(int_version)
            else:
                versions.append('%03d' % int_version)

        if not versions:
            versions = ['None']
            self.current_version = 'None'

        if self.all_variants[self.current_variant].get('latest'):
            versions.insert(0, 'latest')
            self.latest_version = '%03d' % self.all_variants[self.current_variant]['latest']

        if self.all_variants[self.current_variant].get('recommended'):
            versions.insert(0, 'recommended')
            self.recommended_version = '%03d' % self.all_variants[self.current_variant]['recommended']

        if self.all_variants[self.current_variant].get('approved'):
            versions.insert(0, 'approved')
            self.approved_version = '%03d' % self.all_variants[self.current_variant]['approved']
        else:
            self.approved_version = None

        versions = sorted(versions)
        variants = sorted(variants)

        self.set_combo_value(self.variant_widget, self.current_variant, values=variants)
        self.set_combo_value(self.version_widget, self.current_version, values=versions)


    def set_combo_value(self, combo_widget, value, values=None):
        if self.editable:
            if values is None and value == combo_widget.currentText():
                return

            if values is not None:
                combo_widget.clear()
                combo_widget.addItems(values)
            if isinstance(value, int):
                value = '%03d' % value
            selected_index = combo_widget.findText(value)
            if selected_index is -1:
                if value in ['approved', 'recommended', 'latest', 'none']:
                    combo_widget.addItem(value)
                    selected_index = combo_widget.findText(value)
                else:
                    return
            combo_widget.setCurrentIndex(selected_index)
        else:
            combo_widget.setText(value)

    def set_combo(self):
        variant_name = '%s_variant' % self.layer_name

        variants = self.config.get('variants', ['None'])
        if not variants:
            variants = self.config.get('published_items', {}).keys()
        if not variants:
            variants = [self.current_variant]

        self.set_combo_value(self.variant_widget, self.current_variant, values=variants)
        if self.editable:
            self.variant_widget.currentTextChanged.connect(self.update_version)
            self.version_widget.currentTextChanged.connect(self.set_version)

        self.tree_widget.setItemWidget(self, 1, self.variant_widget)
        self.tree_widget.setItemWidget(self, 2, self.version_widget)

        self.setText(3, self.config.get('loader_type', 'NaN'))
        self.setText(4, self.config.get('type', '----'))

        self.update_version(self.current_variant)

        if self.tree_widget.editable:
            self.update_button = QtWidgets.QPushButton('Update')
            self.update_button.clicked.connect(self.update_asset)
            self.tree_widget.setItemWidget(self, 5, self.update_button)
    def update_asset(self):
        print('update_asset')
        variant = str(self.variant_widget.currentText())
        version = str(self.version_widget.currentText())
        if version.isdigit():
            version = int(version)
        else:
            version = self.all_variants[variant].get(version, 'None')

        if self.init_version == version and self.init_variant == variant:
            print('same')
            return

        database = self.tree_widget.database

        entity = database[self.config['type']][self.config['code']]
        published_elements = entity.sg_published_elements
        variant_view = database['CustomEntity11'][variant]
        step_view = database['Step'][self.layer_name]
        published_version = published_elements.find_with_filters(sg_variant=variant_view,
                                                                 sg_version_number=version,
                                                                 sg_step=step_view,
                                                                 single_item=True)

        if published_version.empty:
            return

        entity_data = self.tree_widget.app_config['entities'].get(self.config['loader_type'])
        if not entity_data:
            return
        script_data = entity_data['setter']
        script_data['args'] = {'node_data': self.config, 'publish_view':published_version}
        dynamic_run.run_scripts(script_data)
        self.init_version = version
        self.init_variant = variant
        self.current_version = version
        self.current_variant = variant

    def update_version(self, variant_name):
        print('>' * 100)
        print('update version')
        self.current_variant = variant_name
        print(self.current_variant)

        if isinstance(self.current_version, int):
            versions = ['%03d' % self.current_version]
        else:
            versions = [self.current_version]

        print(self.all_variants)

        if self.all_variants:
            print('check published_versions')
            variant_data = self.all_variants.get(self.current_variant, {})

            versions_view = variant_data.get('versions')
            if versions_view:
                versions = ['%03d' % ver for ver in versions_view]

        variant_name = '%s_version' % self.layer_name
        selected = self.config.get('version_number', '')
        print(selected)
        if not selected:
            selected = 'None'

        if isinstance(selected, int):
            selected = '%03d' % selected
        print('>>>')
        self.set_combo_value(self.version_widget, selected, values=versions)


    def set_variant(self, variant):
        self.current_variant = variant
        self.set_combo_value(self.variant_widget, variant)

    def set_version(self, version):

        if isinstance(version, int):
            version = '%03d' % version
        self.current_version = version
        self.set_combo_value(self.version_widget, version)


        if version == 'approved' or version == self.approved_version:
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5

        elif version == 'None':
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("grey")))
            self.status = 0

        elif version == self.recommended_version or version == 'recommended':
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5

        elif self.approved_version is None and (version != 'latest' and version != self.latest_version):
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("red")))
            self.status = 1

        elif self.approved_version and version == 'latest':
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("orange")))
            self.status = 3

        elif self.approved_version and (version != 'approved' or version != self.approved_version):
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("red")))
            self.status = 1

        elif self.approved_version is None and version == 'latest' or version == 'recommended':
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5

        else:
            self.setForeground(0, QtGui.QBrush(QtGui.QColor("orange")))
            self.status = 3

        self.tree_widget.check_status()

    def set_active(self, active):
        self.active = active
        if self.active:
            self.setCheckState(0, QtCore.Qt.Checked)
        else:
            self.setCheckState(0, QtCore.Qt.Unchecked)


class SceneTree(QtWidgets.QTreeWidget):
    def __init__(self, parent=None, project=None, editable=True):
        super().__init__(parent=parent)
        self.setColumnCount(6)
        self.editable = editable

        self.project = project
        self.database = database.DataBase()
        self.database.fill(self.project, precatch=False)
        header = self.headerItem()
        header.setText(0, 'Layer')
        header.setText(1, 'Variant')
        header.setText(2, 'Version')
        header.setText(3, 'Loader')
        header.setText(4, 'Type')
        header.setText(5, 'Action')
        self.get_config()
        self.update_dependencies()

    def get_config(self):
        host_app = packages_io.get_application()

        self.config_solver = config_manager.ConfigSolver(project=self.project)
        self.project_config = self.config_solver.get_config('project')
        self.app_config = self.config_solver.get_config(host_app, module='maya_scene_manager')


    def scan_scene(self):
        all_inputs = {}
        for entity, entity_scripts in self.app_config['entities'].items():
            function_data = entity_scripts['getter']
            if 'project' in function_data.get('args', {}):
                function_data['args']['project'] = self.project
            entity_data = dynamic_run.run_scripts(function_data)
            all_inputs.update(entity_data)

        return all_inputs

    def get_scene_versions(self, dependencies):
        asset_list = list(dependencies.keys())
        asset_filters = [['code', 'in', asset_list]]
        self.database.query_sg_database('Asset', filters=asset_filters, as_precache=False)
        self.database.query_sg_database('Step', as_precache=True)

        assets_view = self.database['Asset'].find_with_filters(code=asset_list)
        assets_view.precache_dependencies(fields=['sg_published_elements'])

        for asset_index in range(self.topLevelItemCount()):
            asset_item = self.topLevelItem(asset_index)
            asset_name = asset_item.layer_name
            if not asset_name:
                continue
            asset_view = self.database['Asset'][asset_name]
            for instance_index in range(asset_item.childCount()):
                instance_item = asset_item.child(instance_index)
                instance_name = instance_item.layer_name
                for step_index in range(instance_item.childCount()):
                    step_item = instance_item.child(step_index)
                    pipeline_step = step_item.layer_name

                    #step_view = self.database['Step'][pipeline_step]
                    step_view = self.database['Step'].find_with_filters(code=pipeline_step,
                                                                        entity_type='Shot',
                                                                        single_item=True
                                                                        )
                    print(asset_view)
                    print(pipeline_step)
                    print(step_view)
                    published_elements = self.database['CustomEntity09'].find_with_filters(sg_asset=asset_view,
                                                                                           sg_step=step_view,
                                                                                           sg_complete=True,
                                                                                           sg_delete=False,)
                    variants_data = {}

                    for published_element in published_elements:
                        publish_variant = published_element.sg_variant_name
                        variants_data[publish_variant] = variants_data.get(publish_variant, {'versions': [],
                                                                                             'recommended': None,
                                                                                             'latest': None,
                                                                                             'approved': None,
                                                                                             })
                        variants_data[publish_variant]['versions'].append(published_element.sg_version_number)
                        if published_element.sg_status_list == 'apr':
                            if variants_data[publish_variant][
                                'approved'] is None or published_element.sg_version_number > \
                                    variants_data[publish_variant]['approved']:
                                variants_data[publish_variant]['approved'] = published_element.sg_version_number

                        if variants_data[publish_variant]['latest'] is None or published_element.sg_version_number > \
                                variants_data[publish_variant]['latest']:
                            variants_data[publish_variant]['latest'] = published_element.sg_version_number

                    for variant_key, variant_data in variants_data.items():
                        if variant_data['approved']:
                            variant_data['recommended'] = variant_data['approved']
                        else:
                            variant_data['recommended'] = variant_data['latest']

                    if not variants_data:
                        variants_data = {'Master': {}}
                    pprint(variants_data)
                    step_item.set_variants(variants_data)

    def update_dependencies(self):
        dependencies = self.scan_scene()

        self.update_layers(config=dependencies)
        if self.editable:
            self.get_scene_versions(dependencies)

    def check_item_status(self, item):
        child_count = item.childCount()
        if child_count == 0:
            return item.status

        child_status_list = []
        for index in range(child_count):
            child_item = item.child(index)
            child_status = self.check_item_status(child_item)
            child_status_list.append(child_status)

        child_status_list = list(set(child_status_list))
        if len(child_status_list) == 1:
            status = child_status_list[0]
        else:
            status = 3

        if status == 5:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('green')))
        elif status == 3:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('orange')))
        elif status == 1:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('red')))


        return status



    def check_status(self):
        item_count = self.topLevelItemCount()

        for index in range(item_count):
            item = self.topLevelItem(index)
            self.check_item_status(item)

    def clear_widget(self):
        iterator = QtWidgets .QTreeWidgetItemIterator(self, QtWidgets .QTreeWidgetItemIterator.All)
        while iterator.value():
            iterator.value().takeChildren()
            iterator += 1
        i = self.topLevelItemCount()
        while i > -1:
            self.takeTopLevelItem(i)
            i -= 1

    def update_layers(self, config=None):
        if not config:
            config = {}
        self.config = config
        self.clear_widget()
        for asset_name_instance, asset_layers in self.config.items():
            self.add_layers(asset_name_instance, asset_layers)

    def get_layer_from_step(self, step):
        all_steps, layer_data = self.manager.check_layer_in_assembly(step)
        return all_steps, layer_data

    def add_layers(self, asset_name, config, root=None, entity_type='Asset'):
        root = SceneTreeItem(asset_name, None, config,  parent=self)
        self.addTopLevelItem(root)
        for instance_name, instance_data in config.items():
            if 'id' not in instance_data :
                instance_item = SceneTreeItem(instance_name, None, instance_data, parent=self)
                root.addChild(instance_item)

                for pipeline_step, step_data in instance_data.items():
                    item = SceneTreeItem(pipeline_step, pipeline_step, step_data, parent=self)
                    instance_item.addChild(item)

                    item.set_combo()
            else:
                item = SceneTreeItem(instance_name, instance_name, instance_data, parent=self)
                root.addChild(item)

                item.set_combo()

    def getValue(self):
        data = OrderedDict()
        item_count = self.topLevelItemCount()

        for index in range(item_count):
            item = self.topLevelItem(index)
            asset_name = item.layer_name
            data[asset_name] = {}
            if not item.childCount():
                continue
            for instance_index in range(item.childCount()):
                instance_item = item.child(instance_index)
                asset_alias = instance_item.layer_name
                if not instance_item.childCount():
                    continue
                data[asset_name][asset_alias] = {}

                for child_index in range(instance_item.childCount()):
                    child = instance_item.child(child_index)

                    pipeline_step = child.layer_name
                    data[asset_name][asset_alias][pipeline_step] = {'active': child.active,
                                                                    'code': asset_name,
                                                                    'step': pipeline_step,
                                                                    'type': child.entity_type}
                    if child.current_variant:
                        data[asset_name][asset_alias][pipeline_step]['variant'] = child.current_variant
                    if child.current_version:
                        data[asset_name][asset_alias][pipeline_step]['version'] = child.current_version

        return data


def test_builder():
    import builder.lib.builder_core as builder_core
    importlib.reload(builder_core)
    app = QtWidgets.QApplication(sys.argv)
    project = 'SGD'

    path = 'V:/SGD/publish/usd/assets/Main_Characters/LeoMesi/asset_assembly.usda'
    asset_name = 'LeoMesi'
    entity_name = 's00_ep00_sq010_sh010'
    entity_type = 'Shot'
    pipeline_step = 'Layout'
    builder_core = builder_core.AssetBuilder(pipeline_step, project=project)
    variant_name = 'Master'
    dependencies = builder_core.set_context(entity_name, variant=variant_name)

    window = SceneTree()
    window.update(config=dependencies)
    window.show()
    app.exec_()

def test_publisher():
    import packages_io
    importlib.reload(packages_io)

    project = 'SGD'
    dependencies = packages_io.scan_file()
    pprint(dependencies)
    window = SceneTree()
    window.update(config=dependencies)
    #pprint(window.getValue())
    window.show()


if __name__ == '__main__':
    test_builder()