import maya.cmds as cmds
import maya.mel as mel

import maya_assemblies.lib.shader_helpers as shader_helpers
from pprint import pprint

def get_asset_transform():
    node = '|asset'
    all_inputs = {}
    if cmds.objExists(node) and cmds.attributeQuery('asset_name', n='|asset', exists=True):

        asset_name = cmds.getAttr('|asset.asset_name')
        asset_type = cmds.getAttr('|asset.asset_type')
        model_version = cmds.getAttr('|asset.model_version')
        if model_version:
            model_version = int(model_version)
        else:
            model_version = 1
        variant = cmds.getAttr('|asset.model_variant')
        all_inputs[asset_name] = {asset_name: {'Model': {'version_number': model_version,
                                                         'variant': variant,
                                                         'asset_name': asset_name,
                                                         'code': asset_name,
                                                         'asset_type': asset_type,
                                                         'loader_type': 'import',
                                                         'type': 'Asset',
                                                         'node': node

                                                         }
                                               }
                                  }
    return all_inputs


def set_asset_transform(node_data, publish_view):
    pprint(node_data)
    root_node = '|asset'
    all_meshes = {}
    variant_name = publish_view.sg_variant_name
    version_number = publish_view.sg_version_number
    project = publish_view.project.code
    asset_name = node_data['asset_name']
    asset_type = node_data['asset_type'].replace(' ', '_')

    render_node = '|asset|render'
    proxy_node = '|asset|proxy'


    prim_vars = [['/asset', 'model_variant', variant_name],
                 ['/asset', 'model_version', '%03d' % version_number]]
    shadered_meshes = []
    for shader_group in cmds.ls(type='shadingEngine'):
        print(shader_group)
        meshes = cmds.listConnections(shader_group, type="mesh")

        if not meshes:
            continue
        all_meshes[shader_group] = []
        for mesh in meshes:
            new_mesh = cmds.listRelatives(mesh, s=True, f=True)
            all_meshes[shader_group].append(new_mesh[0])
            shadered_meshes.append(new_mesh[0])

    if cmds.objExists(render_node):
        cmds.delete(render_node)
    if cmds.objExists(proxy_node):
        cmds.delete(proxy_node)

    full_path = 'V:/%s/publish/usd/assets/%s/%s/geometry/%s_geometry.usda' % (project, asset_type, asset_name, asset_name)

    render_prim = '/asset/render'
    proxy_prim = '/asset/proxy'

    cmds.mayaUSDImport(file=full_path, primVariant=prim_vars, primPath=render_prim, parent=root_node)
    cmds.mayaUSDImport(file=full_path, primVariant=prim_vars, primPath=proxy_prim, parent=root_node)

    loaded_meshes =[]
    for mesh in cmds.listRelatives(root_node, ad=True, f=True):
        if cmds.nodeType(mesh) != 'mesh':
            continue
        loaded_meshes.append(mesh.split('|')[-1])

    shadered_meshes = [m.split('|')[-1] for m in shadered_meshes]

    print(loaded_meshes)
    print(shadered_meshes)
    wrong_assign = list(set(shadered_meshes) - set(loaded_meshes))
    wrong_name = list(set(loaded_meshes) - set(shadered_meshes))
    pprint(wrong_assign)
    pprint(wrong_name)
    translator = shader_helpers.get_translator(wrong_name, wrong_assign)
    pprint(translator)

    for shading_group, meshes in all_meshes.items():
        for mesh in meshes:
            cmds.sets(mesh, e=True, forceElement=shading_group)

    mel.eval('generateAllUvTilePreviews')