import os
import sys
import shutil
import importlib


import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds

from pxr import Usd, UsdGeom, Sdf

import maya_assemblies.lib.master_assembly as master_assembly
import maya_assemblies.lib.helpers as helpers

import maya_assemblies.lib.representations.locator_representation as locator_representation

import maya_assemblies.lib.representations.geo_usd_representation as geo_usd_representation
import maya_assemblies.lib.representations.usd_stage_representation as usd_stage_representation

importlib.reload(master_assembly)
importlib.reload(helpers)
importlib.reload(usd_stage_representation)

importlib.reload(geo_usd_representation)


class UsdAssembly(master_assembly.MasterAssembly):
    typename = "UsdAssembly"
    id = OpenMaya.MTypeId(70004)
    icon_name = "out_assemblyReference.png"
    tracking_dict = {}

    def __init__(self):
        self.representations = {}
        self.active_rep = None
        self.is_updating_namespace = False
        self.representations = {}
        self.allow_shaders = False
        self.is_loaded = False
        self.loader_attributes_callback = None
        self.set_attributes_callback = None
        self.connect_attributes_callback = None
        self.attribute_set_edits = {}
        self.attribute_connect_edits = {}
        self.applying_edit = False
        self.force_refresh = False
        self.variant_sets = {}
        self._default_representation = 'Geometry'
        self.is_activating = False
        super(UsdAssembly, self).__init__()

    @property
    def geometry_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.usd_file)
        value = plug.asString()
        return value

    @property
    def show_proxy(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.show_proxy_attr)
        value = plug.asBool()
        return value



    @classmethod
    def initialize(cls):
        cls.active_representation = cls.add_string_attribute('activeRepresentation', 'arep', category='representation')
        cls.representation_namespace = cls.add_string_attribute('representationNamespace', 'rns')

        cls.asset_name = cls.add_string_attribute('assetName', 'an', category='asset_data')
        cls.asset_type = cls.add_string_attribute('assetType', 'at', category='asset_data')
        cls.instance_number = cls.add_integer_attribute('instanceNumber', 'in', min=0, default=0, category='asset_data')
        cls.usd_file = cls.add_string_attribute('usdPath', 'usp', as_path=True, category='refresh_geo')
        cls.show_proxy_attr = cls.add_boolean_attribute('showProxy', 'shp', default=True, category='refresh_geo')

    def canRepApplyEdits(self, representation):
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        parent = afn.getParentAssembly()
        if parent and not parent.isNull():
            return False

        return True

    def initialize_assembly(self):

        geometry_path = self.geometry_path
        if not os.path.exists(geometry_path):
            return
        self.representations['Geometry'] = geo_usd_representation.GeoUsdRepresentation(self, 'Geometry', geometry_path)
        self.representations['USD'] = usd_stage_representation.UsdStageRepresentation(self, 'USD', geometry_path)

        self.representations['Disabled'] = locator_representation.LocatorRepresentation(self, 'Disabled', None)


        self._default_representation = 'Geometry'


    def add_callbacks(self):
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        parent = afn.getParentAssembly()
        if parent and not parent.isNull():
            return
        assembly_node = self.get_assembly_name()
        nodes = cmds.listRelatives(assembly_node, ad=True, f=True)
        if nodes is None:
            return
        for node in nodes:
            mobject_node = self.get_mobject_from_name(node)
            self.set_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(mobject_node,
                                                                                             self.add_shading_edit)


    def add_shading_edit(self, msg, plug, other_plug, data):
        if not msg:
            return

        if plug.isLocked():
            return

        if self.applying_edit:
            return


        if msg & OpenMaya.MNodeMessage.kConnectionMade:  # or  msg & OpenMaya.MNodeMessage.kOtherPlugSet :
            attribute = plug.name()
            self.attribute_connect_edits[attribute] = other_plug.name()
            if attribute in self.attribute_set_edits:
                self.attribute_set_edits.pop(attribute)




    def getRepNamespace(self):

        namespace_plug = OpenMaya.MPlug(self.thisMObject(), self.representation_namespace)
        namespace = namespace_plug.asString()
        if not namespace:
            namespace = self.get_default_namespace()
            namespace_plug.setString(namespace)

        afn = OpenMaya.MFnAssembly(self.thisMObject())
        parent = afn.getParentAssembly()
        if parent and not parent.isNull():
            parent_assembly = OpenMaya.MFnAssembly(parent)
            parent_namespace = parent_assembly.getRepNamespace()
            namespace = '%s:%s' % (parent_namespace, namespace)

        return namespace

    def postConstructor(self):
        self.loader_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(self.thisMObject(),
                                                                                            self.add_assembly_node_callbacks)
        super(UsdAssembly, self).postConstructor()

    def postLoad(self, force=False):
        afn = OpenMaya.MFnAssembly(self.thisMObject())
        if not afn.isTopLevel() and force is False:
            return
        representation_plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)
        representation_type = representation_plug.asString()
        if not representation_type:
            representation_type = self._default_representation

        self.performActivate(representation_type)

    def add_assembly_node_callbacks(self, msg, plug, other_plug, data):
        if self.is_activating:
            return

        if msg & OpenMaya.MNodeMessage.kAttributeSet and self.is_activating == False:
            fnattr = OpenMaya.MFnAttribute(plug.attribute())
            categories = []
            fnattr.getCategories(categories)

            if 'representation' in categories:
                representation = plug.asString()
                active_rep = self.getActive()

                if representation in self.representations and representation != active_rep:
                    self.activateRep(representation)

            if 'refresh_geo' in categories:
                self.save_edits()
                out = self.initialize_assembly()
                if out == -1:
                    return

                if helpers.in_io() and not self.is_loaded:
                    return

                self.force_refresh = True
                self.postLoad(force=True)
                self.force_refresh = False


def initializePlugin(mobject):
    ''' Initialize the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.registerNode(UsdAssembly.typename,
                             UsdAssembly.id,
                             UsdAssembly.creator,
                             UsdAssembly.initialize,
                             OpenMayaMPx.MPxNode.kAssembly,
                             "drawdb/geometry/transform")  # , kPluginNodeClassify)

        cmds.assembly(edit=True, type=UsdAssembly.typename, label=UsdAssembly.typename)

        cmds.assembly(edit=True, type=UsdAssembly.typename, repTypeLabelProc=UsdAssembly.representation_label)

        cmds.assembly(edit=True, type=UsdAssembly.typename, listRepTypesProc=UsdAssembly.list_representation_types)

        # Register the assembly nodes to the filePathEditor

        # cmds.filePathEditor(registerType=UsdAssembly.typename, typeLabel=UsdAssembly.typename, temporary=True)

        # if not cmds.about(batch=True):
        #    import maya_assemblies.lib.templates.AEAssemblyTemplate as AEAssemblyTemplate

    except:
        sys.stderr.write("Failed to register node: " + UsdAssembly.typename)
        raise


def uninitializePlugin(mobject):
    ''' Uninitializes the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.deregisterNode(UsdAssembly.id)
        cmds.assembly(edit=True, deregister=UsdAssembly.typename)


    except:
        sys.stderr.write("Failed to deregister node: " + UsdAssembly.typename)
        raise
