import os
import sys
import ast
import shutil
import importlib

import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds

import maya_assemblies.lib.master_assembly as master_assembly
import maya_assemblies.lib.helpers as helpers

import maya_assemblies.lib.representations.set_usd_representation as set_usd_representation
import maya_assemblies.lib.representations.locator_representation as locator_representation
import maya_assemblies.lib.representations.geo_usd_representation as geo_usd_representation


importlib.reload(master_assembly)
importlib.reload(helpers)

importlib.reload(set_usd_representation)


class SetAssembly(master_assembly.MasterAssembly):
    typename = "SetAssembly"
    id = OpenMaya.MTypeId(70002)
    icon_name = "out_assemblyReference.png"
    tracking_dict = {}

    def __init__(self):
        self.representations = {}
        self.active_rep = None
        self.is_updating_namespace = False
        self.representations = {}
        self.children_assemblies = {}
        self.localized = True
        self.is_loaded = False
        self.loader_attributes_callback = None
        self.set_attributes_callback = None
        self.connect_attributes_callback = None
        self.attribute_set_edits = {}
        self.attribute_connect_edits = {}
        self.applying_edit = False
        self.force_refresh = False
        self.is_activating = False

        super(SetAssembly, self).__init__()

    @property
    def set_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.set_file)
        value = plug.asString()
        return value

    @property
    def subset_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.subset_file)
        value = plug.asString()
        return value


    @property
    def lod(self):
        lod_plug = OpenMaya.MPlug(self.thisMObject(), self.level_of_detail)
        lod = lod_plug.asInt()
        return lod

    @classmethod
    def initialize(cls):
        cls.active_representation = cls.add_string_attribute('activeRepresentation', 'arep', category='representation')
        cls.representation_namespace = cls.add_string_attribute('representationNamespace', 'rns')

        cls.asset_name = cls.add_string_attribute('assetName', 'an', category='asset_data')
        cls.instance_number = cls.add_integer_attribute('instanceNumber', 'in', min=0, default=0, category='asset_data')

        cls.set_sg_id = cls.add_string_attribute('setSgId', 'sid')

        cls.set_file = cls.add_string_attribute('setPath', 'sep', as_path=True, category='refresh_geo')

        cls.set_hash = cls.add_string_attribute('setHash', 'shs', category='asset_data')



    def canRepApplyEdits(self, representation):
        return True

    def initialize_assembly(self):

        set_path = self.set_path

        self.representations['Set'] = set_usd_representation.SetUsdRepresentation(self, 'Set', set_path)
        #self.representations['Set'] = geo_usd_representation.GeoUsdRepresentation(self, 'Set', set_path)


        self.representations['Disabled'] = locator_representation.LocatorRepresentation(self, 'Disabled', None)
        self._default_representation = 'Set'


    def add_callbacks(self):
        return
        assembly_node = self.get_assembly_name()
        nodes = cmds.listRelatives(assembly_node, ad=True, f=True)

        if not nodes:
            return

        for node in nodes:
            if cmds.nodeType(node) == 'transform' or cmds.nodeType(node) == 'GeometryAssembly':
                mobject_node = self.get_mobject_from_name(node)
                self.set_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(mobject_node,
                                                                                                 self.add_set_edit)

    def add_set_edit(self, msg, plug, other_plug, data):
        if not msg:
            return

        if msg & OpenMaya.MNodeMessage.kAttributeSet:
            attribute = plug.name()
            connected = cmds.listConnections(attribute, p=True, c=True, s=True)
            if connected:
                return

            data = []
            plug.getSetAttrCmds(data, OpenMaya.MPlug.kAll, True)
            setAttr_cmd = data[0].split(' ')[2:]

            raw_value = cmds.getAttr(attribute)
            setAttr_cmd[-1] = str(raw_value)
            value = ' '.join(setAttr_cmd)

            if not self.applying_edit and attribute not in self.attribute_connect_edits and self.attribute_set_edits.get(
                    attribute) != value:
                self.attribute_set_edits[attribute] = value
    def beforeSave(self):
        pass
        #self.save_child_representation()
        #self.save_edits()



    def getRepNamespace(self):
        namespace_plug = OpenMaya.MPlug(self.thisMObject(), self.representation_namespace)
        namespace = namespace_plug.asString()
        if not namespace:
            namespace = self.get_default_namespace()
            namespace_plug.setString(namespace)
        return namespace

    def postConstructor(self):
        self.loader_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(self.thisMObject(),
                                                                                            self.add_assembly_node_callbacks)
        super(SetAssembly, self).postConstructor()


    def postLoad(self):
        print('post load set')
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        current_representation = self.current_representation

        if not current_representation:
            current_representation = self._default_representation

        print('perform activate', current_representation)
        self.activate(current_representation)



    def add_assembly_node_callbacks(self, msg, plug, other_plug, data):
        if self.is_activating:
            return
        attribute_name = plug.name().split('.', 1)[-1]
        if attribute_name in ['activeRepresentation']:
            return

        if msg & OpenMaya.MNodeMessage.kAttributeSet:
            fnattr = OpenMaya.MFnAttribute(plug.attribute())

            categories = []
            fnattr.getCategories(categories)

            if 'representation' in categories:
                representation = plug.asString()
                active_rep = self.getActive()
                if representation in self.representations and representation != active_rep:
                    self.activateRep(representation)


            if 'refresh_geo' in categories:

                out = self.initialize_assembly()
                if out == -1:
                    return

                if helpers.in_io() and not self.is_loaded:
                    return

                self.force_refresh = True
                self.postLoad()
                self.force_refresh = False




def initializePlugin(mobject):
    ''' Initialize the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.registerNode(SetAssembly.typename,
                             SetAssembly.id,
                             SetAssembly.creator,
                             SetAssembly.initialize,
                             OpenMayaMPx.MPxNode.kAssembly,
                             "drawdb/geometry/transform")  # , kPluginNodeClassify)

        cmds.assembly(edit=True, type=SetAssembly.typename, label=SetAssembly.typename)

        cmds.assembly(edit=True, type=SetAssembly.typename, repTypeLabelProc=SetAssembly.representation_label)

        cmds.assembly(edit=True, type=SetAssembly.typename, listRepTypesProc=SetAssembly.list_representation_types)

        # Register the assembly nodes to the filePathEditor

        #cmds.filePathEditor(registerType=GeometryAssembly.typename, typeLabel=GeometryAssembly.typename, temporary=True)

        #if not cmds.about(batch=True):
        #    import maya_assemblies.lib.templates.AEAssemblyTemplate as AEAssemblyTemplate

    except:
        sys.stderr.write("Failed to register node: " + SetAssembly.typename)
        raise

def uninitializePlugin(mobject):
    ''' Uninitializes the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.deregisterNode(SetAssembly.id)
        cmds.assembly(edit=True, deregister=SetAssembly.typename)


    except:
        sys.stderr.write("Failed to deregister node: " + SetAssembly.typename)
        raise
