import os
import sys
import logging
import importlib


import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds
from pxr import Usd, UsdGeom, Sdf

import maya_assemblies.lib.master_assembly as master_assembly
import maya_assemblies.lib.helpers as helpers

import maya_assemblies.lib.representations.scene_representation as scene_representation
import maya_assemblies.lib.representations.locator_representation as locator_representation


importlib.reload(master_assembly)
importlib.reload(helpers)

importlib.reload(scene_representation)

logger = logging.getLogger(__name__)
# logger.setLevel(logging.WARNING)

class RigAssembly(master_assembly.MasterAssembly):
    typename = "RigAssembly"
    id = OpenMaya.MTypeId(70003)
    icon_name = "out_assemblyReference.png"
    tracking_dict = {}

    def __init__(self):
        self.representations = {}
        self.active_rep = None
        self.is_updating_namespace = False
        self.representations = {}
        self.allow_shaders = False
        self.localized = False
        self.is_loaded = False
        self.loader_attributes_callback = None
        self.set_attributes_callback = None
        self.connect_attributes_callback = None
        self.attribute_set_edits = {}
        self.attribute_connect_edits = {}
        self.applying_edit = False
        self.force_refresh = False

        super(RigAssembly, self).__init__()

    @property
    def rig_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.rig_file)
        value = plug.asString()
        return value

    @property
    def geometry_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.geometry_file)
        value = plug.asString()
        return value

    @property
    def shader_path(self):
        project = os.environ['PROJECT']
        asset_name_plug = OpenMaya.MPlug(self.thisMObject(), self.asset_name)
        asset_name = asset_name_plug.asString()

        asset_type_plug = OpenMaya.MPlug(self.thisMObject(), self.asset_type)
        asset_type = asset_type_plug.asString()
        asset_type = asset_type.replace(' ', '_')

        usd_asset_shading_path = r'\\project/SGD/%s/publish/usd/assets/%s/%s/shader/%s_shader.usda' % (project,
                                                                                                       asset_type,
                                                                                                       asset_name,
                                                                                                       asset_name)

        print('ASSET PATH', usd_asset_shading_path)
        if not os.path.exists(usd_asset_shading_path):
            return

        self.stage = Usd.Stage.Open(usd_asset_shading_path)
        asset_prim = self.stage.GetPrimAtPath('/asset')

        if not asset_prim.IsValid():
            return

        attribute = asset_prim.GetAttribute('atlantis:shading_maya_path')
        if not attribute.IsValid():
            return

        shader_asset_path = attribute.Get()
        if not shader_asset_path:
            return

        path = shader_asset_path.path
        print('Loading textures from: %s' % path)
        return path

    @property
    def lod(self):
        return 1


    @classmethod
    def initialize(cls):
        cls.geometry_sg_id = cls.add_string_attribute('rigSgId', 'rgi')
        cls.active_representation = cls.add_string_attribute('activeRepresentation', 'arep', category='representation')
        cls.representation_namespace = cls.add_string_attribute('representationNamespace', 'rns')

        cls.asset_name = cls.add_string_attribute('assetName', 'an', category='asset_data')
        cls.asset_type = cls.add_string_attribute('assetType', 'atp', category='asset_data')
        cls.instance_number = cls.add_integer_attribute('instanceNumber', 'in', min=0, default=0, category='asset_data')

        cls.rig_file = cls.add_string_attribute('rigPath', 'rgp', as_path=True, category='refresh_geo')
        cls.rig_variant = cls.add_string_attribute('rigVariant', 'rvar', category='asset_data')

        cls.rig_hash = cls.add_string_attribute('rigHash', 'rhs', category='asset_data')
        cls.rig_version = cls.add_string_attribute('rigVersion', 'rver', category='asset_data')


    def canRepApplyEdits(self, representation):
        return True

    def initialize_assembly(self):

        rig_path = self.rig_path

        self.representations['Rig'] = scene_representation.SceneRepresentation(self, 'Rig', rig_path)
        self.representations['Disabled'] = locator_representation.LocatorRepresentation(self, 'Disabled', None)
        self._default_representation = 'Rig'


    def add_callbacks(self):
        valid_nodes = ['mesh', 'camera', 'controller', 'transform', 'joint']
        assembly_node = self.get_assembly_name()
        nodes = cmds.listRelatives(assembly_node, ad=True, f=True)
        if not nodes:
            return
        for node in nodes:
            if cmds.nodeType(node) not in valid_nodes:
                continue

            mobject_node = self.get_mobject_from_name(node)
            self.set_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(mobject_node,
                                                                                             self.add_set_edit)

    def add_set_edit(self, msg, plug, other_plug, data):
        #print('callback')
        if not msg:
            return

        if plug.isLocked():
            return

        if self.applying_edit:
            return

        if msg & OpenMaya.MNodeMessage.kAttributeSet:

            attribute = plug.name()

            attr_name = attribute.split('.')[-1]
            data = []
            plug.getSetAttrCmds(data, OpenMaya.MPlug.kAll, True)
            setAttr_cmd = data[0].split(' ')[2:]
            try:
                raw_value = cmds.getAttr(attribute)
            except:
                return
            setAttr_cmd[-1] = str(raw_value)

            value = ' '.join(setAttr_cmd)
            logger.debug('Set attribute %s' % attribute)
            if attr_name in ['translate', 'rotate', 'scale']:
                raw_value = raw_value[0]
                attr_sufix = ['X', 'Y', 'Z']
                for sufix, child_value in zip(attr_sufix, raw_value):
                    child_attr = '%s%s' % (attribute, sufix)
                    setAttr_cmd[-1] = str(child_value)
                    value = ' '.join(setAttr_cmd)
                    logger.debug('Set child attribute %s' % child_attr)
                    print('Set child attribute %s' % child_attr)

                    self.attribute_set_edits[child_attr] = value



            else:

                self.attribute_set_edits[attribute] = value

        if msg & OpenMaya.MNodeMessage.kConnectionMade:  # or  msg & OpenMaya.MNodeMessage.kOtherPlugSet :
            attribute = plug.name()
            self.attribute_connect_edits[attribute] = other_plug.name()
            logger.debug('Connect attribute %s' % attribute)
            print('Connect attribute %s' % attribute)
            if attribute in self.attribute_set_edits:
                self.attribute_set_edits.pop(attribute)
                logger.debug('clean from attribute set')


    def getRepNamespace(self):

        namespace_plug = OpenMaya.MPlug(self.thisMObject(), self.representation_namespace)
        namespace = namespace_plug.asString()
        if not namespace:
            namespace = self.get_default_namespace()
            namespace_plug.setString(namespace)

        afn = OpenMaya.MFnAssembly(self.thisMObject())
        parent = afn.getParentAssembly()
        if parent and not parent.isNull():
            parent_assembly = OpenMaya.MFnAssembly(parent)
            parent_namespace = parent_assembly.getRepNamespace()
            namespace = '%s:%s' % (parent_namespace, namespace)

        return namespace

    def postConstructor(self):
        self.loader_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(self.thisMObject(),
                                                                                            self.add_assembly_node_callbacks)
        super(RigAssembly, self).postConstructor()


    def postLoad(self):
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        representation_plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)
        representation_type = representation_plug.asString()
        if not representation_type:
            representation_type = self._default_representation

        self.activate(representation_type)

    def load_shading_data(self):
        print('load shading data')
        active_rep = self.getActive()
        if not active_rep:
            return
        activated_representation = self.representations[active_rep]
        shader_path = self.shader_path
        #if not shader_path or not os.path.exists(shader_path):
        #    return
        #activated_representation.apply_shaders(shader_path)


    def add_assembly_node_callbacks(self, msg, plug, other_plug, data):

        if msg & OpenMaya.MNodeMessage.kAttributeSet:
            fnattr = OpenMaya.MFnAttribute(plug.attribute())
            categories = []
            fnattr.getCategories(categories)

            if 'representation' in categories:
                representation = plug.asString()
                active_rep = self.getActive()

                if representation in self.representations and representation != active_rep:
                    self.activateRep(representation)

            if 'refresh_geo' in categories or 'refresh_shading' in categories:
                out = self.initialize_assembly()
                if out == -1:
                    return

                if helpers.in_io() and not self.is_loaded:
                    return

                self.force_refresh = True
                self.postLoad()

                self.force_refresh = False


def initializePlugin(mobject):
    ''' Initialize the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.registerNode(RigAssembly.typename,
                             RigAssembly.id,
                             RigAssembly.creator,
                             RigAssembly.initialize,
                             OpenMayaMPx.MPxNode.kAssembly,
                             "drawdb/geometry/transform")  # , kPluginNodeClassify)

        cmds.assembly(edit=True, type=RigAssembly.typename, label=RigAssembly.typename)

        cmds.assembly(edit=True, type=RigAssembly.typename, repTypeLabelProc=RigAssembly.representation_label)

        cmds.assembly(edit=True, type=RigAssembly.typename, listRepTypesProc=RigAssembly.list_representation_types)

        # Register the assembly nodes to the filePathEditor

        #cmds.filePathEditor(registerType=RigAssembly.typename, typeLabel=RigAssembly.typename, temporary=True)

        #if not cmds.about(batch=True):
        #    import maya_assemblies.lib.templates.AERigAssemblyTemplate as AERigAssemblyTemplate

    except:
        sys.stderr.write("Failed to register node: " + RigAssembly.typename)
        raise

def uninitializePlugin(mobject):
    ''' Uninitializes the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.deregisterNode(RigAssembly.id)
        cmds.assembly(edit=True, deregister=RigAssembly.typename)


    except:
        sys.stderr.write("Failed to deregister node: " + RigAssembly.typename)
        raise
