import os
import sys
import shutil
import importlib

import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds

import maya_assemblies.lib.master_assembly as master_assembly
import maya_assemblies.lib.helpers as helpers

import maya_assemblies.lib.representations.abc_representation as abc_representation
import maya_assemblies.lib.representations.locator_representation as locator_representation
import maya_assemblies.lib.representations.vfx_representation as vfx_representation
import maya_assemblies.lib.representations.gpu_representation as gpu_representation



importlib.reload(master_assembly)
importlib.reload(helpers)

importlib.reload(abc_representation)
importlib.reload(vfx_representation)
importlib.reload(gpu_representation)


class GeometryAssembly(master_assembly.MasterAssembly):
    typename = "GeometryAssembly"
    id = OpenMaya.MTypeId(70001)
    icon_name = "out_assemblyReference.png"
    tracking_dict = {}

    def __init__(self):
        self.representations = {}
        self.active_rep = None
        self.is_updating_namespace = False
        self.representations = {}
        self.allow_shaders = False
        self.localized = True
        self.is_loaded = False
        self.loader_attributes_callback = None
        self.set_attributes_callback = None
        self.connect_attributes_callback = None
        self.attribute_set_edits = {}
        self.attribute_connect_edits = {}
        self.applying_edit = False
        self.force_refresh = False

        super(GeometryAssembly, self).__init__()

    @property
    def geometry_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.geometry_file)
        value = plug.asString()
        return value

    @property
    def shader_path(self):
        shading_plug = OpenMaya.MPlug(self.thisMObject(), self.shading_file)
        shading = shading_plug.asString()
        return shading

    @property
    def fx_path(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.fx_file)
        value = plug.asString()
        return value


    @property
    def lod(self):
        lod_plug = OpenMaya.MPlug(self.thisMObject(), self.level_of_detail)
        lod = lod_plug.asInt()
        return lod

    @classmethod
    def initialize(cls):
        cls.geometry_sg_id = cls.add_string_attribute('geometrySgId', 'gi')
        cls.active_representation = cls.add_string_attribute('activeRepresentation', 'arep', category='representation')
        cls.representation_namespace = cls.add_string_attribute('representationNamespace', 'rns')

        cls.asset_name = cls.add_string_attribute('assetName', 'an', category='asset_data')
        cls.asset_type = cls.add_string_attribute('assetType', 'atp', category='asset_data')
        cls.instance_number = cls.add_integer_attribute('instanceNumber', 'in', min=0, default=0, category='asset_data')
        cls.level_of_detail = cls.add_integer_attribute('LevelOfDetail', 'ld', min=0, default=4, category='subdivision')

        cls.geometry_file = cls.add_string_attribute('geometryPath', 'gep', as_path=True, category='refresh_geo')
        cls.geometry_variant = cls.add_string_attribute('geometryVariant', 'gvar', category='asset_data')
        cls.geometry_version = cls.add_string_attribute('geometryVersion', 'gver', category='asset_data')
        cls.geometry_file_tag = cls.add_string_attribute('geometryFileTag', 'gft', category='asset_data')
        cls.geometry_step = cls.add_string_attribute('geometryStep', 'gs', category='asset_data')
        cls.geometry_hash = cls.add_string_attribute('geometryHash', 'ghs', category='asset_data')

        cls.shading_file = cls.add_string_attribute('shadingPath', 'shpa', as_path=True, category='refresh_shading')
        cls.shading_variant = cls.add_string_attribute('shadingVariant', 'svar', category='asset_data')
        cls.shading_version = cls.add_string_attribute('shadingVersion', 'sver', category='asset_data')
        cls.shading_file_tag = cls.add_string_attribute('shadingFileTag', 'sft', category='asset_data')
        cls.shading_step = cls.add_string_attribute('shadingStep', 'ss', category='asset_data')
        cls.shading_hash = cls.add_string_attribute('shadingHash', 'shs', category='asset_data')

        cls.fx_file = cls.add_string_attribute('fxPath', 'fxp', as_path=True, category='force_fx')
        cls.fx_variant = cls.add_string_attribute('fxVariant', 'fxvar', category='asset_data')
        cls.fx_version = cls.add_string_attribute('fxVersion', 'fxver', category='asset_data')
        cls.fx_file_tag = cls.add_string_attribute('fxFileTag', 'fxft', category='asset_data')
        cls.fx_step = cls.add_string_attribute('fxStep', 'fxs', category='asset_data')
        cls.fx_hash = cls.add_string_attribute('fxHash', 'fxhs', category='asset_data')

    def localize_file(self, path):
        local_root = 'C:/projects'
        if not os.path.exists(path):
            return None

        local_path = '%s/%s' % (local_root, path.split('/', 1)[-1])
        if not os.path.exists(local_path):
            folder = os.path.dirname(local_path)
            if not os.path.exists(folder):
                os.makedirs(folder)
            shutil.copy2(path, local_path)

        return local_path

    def canRepApplyEdits(self, representation):
        return False

    def initialize_assembly(self):

        geometry_path = self.geometry_path
        shader_path = self.shader_path
        fx_path = self.fx_path

        if self.localized:
            geometry_path = self.localize_file(geometry_path)
            shader_path = self.localize_file(shader_path)
            fx_path = self.localize_file(fx_path)

        #self.representations['Geometry'] = abc_representation.AbcRepresentation(self, 'Geometry', geometry_path)
        self.representations['Geometry'] = gpu_representation.GpuRepresentation(self, 'Geometry', geometry_path)
        self.representations['vfx'] = vfx_representation.VFXRepresentation(self, 'vfx', fx_path)
        self.representations['Disabled'] = locator_representation.LocatorRepresentation(self, 'Disabled', None)
        self._default_representation = 'Geometry'


    def load_shading_data(self):

        active_rep = self.getActive()
        if not active_rep:
            return
        activated_representation = self.representations[active_rep]
        shader_path = self.shader_path
        if not shader_path or not os.path.exists(shader_path):
            return
        activated_representation.apply_shaders(shader_path)

    def add_callbacks(self):
        return
        assembly_node = self.get_assembly_name()
        nodes = cmds.listRelatives(assembly_node, ad=True, f=True)
        if not nodes or not self.allow_shaders:
            return
        for node in nodes:
            if cmds.nodeType(node) == 'mesh':
                mobject_node = self.get_mobject_from_name(node)
                self.set_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(mobject_node, self.add_shading_edit)

    def add_shading_edit(self, msg, plug, other_plug, data):
        if not msg:
            return
        if msg & OpenMaya.MNodeMessage.kConnectionMade:  # or  msg & OpenMaya.MNodeMessage.kOtherPlugSet :
            attribute = plug.name()
            self.attribute_connect_edits[attribute] = other_plug.name()

    def getRepNamespace(self):

        namespace_plug = OpenMaya.MPlug(self.thisMObject(), self.representation_namespace)
        namespace = namespace_plug.asString()
        if not namespace:
            namespace = self.get_default_namespace()
            namespace_plug.setString(namespace)

        afn = OpenMaya.MFnAssembly(self.thisMObject())
        parent = afn.getParentAssembly()
        if parent and not parent.isNull():
            parent_assembly = OpenMaya.MFnAssembly(parent)
            parent_namespace = parent_assembly.getRepNamespace()
            namespace = '%s:%s' % (parent_namespace, namespace)

        return namespace

    def postConstructor(self):
        self.loader_attributes_callback = OpenMaya.MNodeMessage.addAttributeChangedCallback(self.thisMObject(),
                                                                                            self.add_assembly_node_callbacks)
        super(GeometryAssembly, self).postConstructor()


    def postLoad(self, force=False):
        afn = OpenMaya.MFnAssembly(self.thisMObject())
        if not afn.isTopLevel() and force is False:
            return
        representation_plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)
        representation_type = representation_plug.asString()
        if not representation_type:
            representation_type = self._default_representation

        self.performActivate(representation_type)

    def add_assembly_node_callbacks(self, msg, plug, other_plug, data):

        if msg & OpenMaya.MNodeMessage.kAttributeSet:
            fnattr = OpenMaya.MFnAttribute(plug.attribute())
            categories = []
            fnattr.getCategories(categories)

            if 'representation' in categories:
                representation = plug.asString()
                active_rep = self.getActive()

                if representation in self.representations and representation != active_rep:
                    self.activateRep(representation)


            if 'refresh_geo' in categories:
                out = self.initialize_assembly()
                if out == -1:
                    return

                if helpers.in_io() and not self.is_loaded:
                    return

                self.force_refresh = True
                self.postLoad(force=True)
                self.force_refresh = False

            if 'refresh_shading' in categories:
                self.load_shading_data()


def initializePlugin(mobject):
    ''' Initialize the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.registerNode(GeometryAssembly.typename,
                             GeometryAssembly.id,
                             GeometryAssembly.creator,
                             GeometryAssembly.initialize,
                             OpenMayaMPx.MPxNode.kAssembly,
                             "drawdb/geometry/transform")  # , kPluginNodeClassify)

        cmds.assembly(edit=True, type=GeometryAssembly.typename, label=GeometryAssembly.typename)

        cmds.assembly(edit=True, type=GeometryAssembly.typename, repTypeLabelProc=GeometryAssembly.representation_label)

        cmds.assembly(edit=True, type=GeometryAssembly.typename, listRepTypesProc=GeometryAssembly.list_representation_types)

        # Register the assembly nodes to the filePathEditor

        #cmds.filePathEditor(registerType=GeometryAssembly.typename, typeLabel=GeometryAssembly.typename, temporary=True)

        #if not cmds.about(batch=True):
        #    import maya_assemblies.lib.templates.AEAssemblyTemplate as AEAssemblyTemplate

    except:
        sys.stderr.write("Failed to register node: " + GeometryAssembly.typename)
        raise

def uninitializePlugin(mobject):
    ''' Uninitializes the plug-in '''
    mplugin = OpenMayaMPx.MFnPlugin(mobject)
    try:
        mplugin.deregisterNode(GeometryAssembly.id)
        cmds.assembly(edit=True, deregister=GeometryAssembly.typename)


    except:
        sys.stderr.write("Failed to deregister node: " + GeometryAssembly.typename)
        raise
