import re
from functools import partial

import pymel.core as pm
import maya.cmds as cmds
import maya.OpenMaya as om

import maya_assemblies.lib.helpers as helpers


# Constants

kReloadDefinition = 'Reload the assembly definition'
kAEOpenNSEditor = 'Open Namespace Editor'
kAEassemblyNamespace = 'Namespace'
kInvalidNamespace = 'Namespace name \'{}\' contains invalid characters.'


def reload_assembly(nodename):
    current_rep = cmds.assembly(nodename, query=True, active=True)
    cmds.assembly(nodename, edit=True, active=current_rep)

def definition_new(attr_name):
    cmds.rowLayout(numberOfColumns=3, columnAttach=[2, 'left', 0],
                   adjustableColumn=2)
    cmds.text(label='path')
    cmds.textField('pathAttr')
    cmds.iconTextButton('filebrowser', i1='navButtonBrowse.png')
    cmds.setParent('..')

    cmds.rowLayout(numberOfColumns=1, columnAttach=[1, 'left', 148])
    cmds.iconTextButton('reload_button', annotation=kReloadDefinition,
                        i1='refresh.png')
    cmds.setParent('..')

    definition_replace(attr_name)


def definition_replace(attr_name):
    def tmp_show_filepath_browser(*args):
        filepaths = cmds.fileDialog2(
            caption='Choose alembic File',
            fileFilter='Alembic Files (*.abc)',
            fileMode=1)
        if filepaths:
            cmds.setAttr(attr_name, filepaths[0], type='string')

    nodename = attr_name.split('.', 1)[0]

    cmds.connectControl('pathAttr', attr_name, fileName=True)

    onode = helpers.get_mobject(nodename)
    afn = om.MFnAssembly(onode)

    is_top_level = afn.isTopLevel()
    if is_top_level:
        # If it is top-level, whether it is editable is controlled by the lock
        # status.
        cmds.textField('pathAttr', edit=True)
    else:
        cmds.textField('pathAttr', edit=True, editable=False)

    cmds.iconTextButton('reload_button', edit=True,
                        command=partial(reload_assembly, nodename))
    cmds.iconTextButton('filebrowser', edit=True, enable=is_top_level,
                        command=tmp_show_filepath_browser)


def namespace_new(attr_name):
    cmds.rowLayout(numberOfColumns=1, columnAttach=[1, 'left', 0],
                   adjustableColumn=1)
    cmds.textFieldGrp('assemblyNamespaceValue', label='Namespace')
    cmds.iconTextButton('assemblyNamespaceEditor', i1='namespaceEditor.png',
                        ann=kAEOpenNSEditor, parent='assemblyNamespaceValue')
    cmds.setParent('..')

    namespace_replace(attr_name)


def namespace_replace(attr_name):
    editable = not cmds.getAttr(attr_name, lock=True)

    current_namespace = cmds.getAttr(attr_name)
    cmds.textFieldGrp(
        'assemblyNamespaceValue', edit=True, text=current_namespace,
        changeCommand=partial(change_namespace_attr, attr_name),
        editable=editable)
    cmds.iconTextButton('assemblyNamespaceEditor', edit=True,
                        command=show_namespace_editor)


def change_namespace_attr(attr_name, new_namespace, *args, **kwargs):
    if re.match('^[a-zA-Z][a-zA-Z0-9_]*$', new_namespace):
        cmds.setAttr(attr_name, new_namespace, type="string")
    else:
        current_namespace = cmds.getAttr(attr_name)
        cmds.textFieldGrp('assemblyNamespaceValue', edit=True, text=current_namespace)
        cmds.error(kInvalidNamespace.format(new_namespace))


def show_namespace_editor():
    if not cmds.window('namespaceEditor', exists=True):
        pm.mel.namespaceEditor()

# ====================================================================


class AE__AssemblyTemplate(pm.ui.AETemplate):

    def __init__(self, nodename):
        self.beginScrollLayout()
        pm.mel.AEtransformMain(nodename)

        self.beginLayout('Assembly', collapse=False)

        #self.callCustom(definition_new,
        #                definition_replace,
        #                'path')

        self.addControl('path')

        self.addControl('assemblyType')
        self.addControl('repNamespace')
        self.addControl('activeRepresentation')
        self.addControl('initialRep')

        self.endLayout()
        self.beginLayout('Publish data', collapse=False)
        self.addControl('assetName')
        self.addControl('instanceNumber')

        self.addControl('pipelineStep')
        self.addControl('assetType')
        self.addControl('variant')
        self.addControl('version')
        self.addControl('fileTag')
        self.addControl('shotgrid_id')
        self.addControl('hash')
        self.endLayout()

        self.endLayout()
        self.beginLayout('Shader assignment', collapse=False)
        self.addControl('shaderFilePath')
        self.addControl('shaderVariant')
        self.addControl('shaderVersion')
        self.addControl('LevelOfDetail')


        self.endLayout()
        self.beginLayout('Asset edits', collapse=False)
        self.addControl('assemblyEdit')
        self.addControl('editPath')
        self.endLayout()

        pm.mel.AEtransformNoScroll(nodename)
        self.addExtraControls()

        # suppresses attributes

        self.suppress('assemblyEdits')
        self.suppress('blackBox')
        self.suppress('rmbCommand')
        self.suppress('templateName')
        self.suppress('templatePath')
        self.suppress('viewName')
        self.suppress('iconName')
        self.suppress('viewMode')
        self.suppress('templateVersion')
        self.suppress('uiTreatment')
        self.suppress('customTreatment')
        self.suppress('creator')
        self.suppress('creationDate')
        self.suppress('containerType')
        self.suppress('publishedNode')
        self.suppress('publishedNodeInfo')
        self.suppress('publishedNodeType')
        self.endScrollLayout()
