from functools import reduce
import operator
import json

import maya.OpenMaya as OpenMaya

def get_mobject(nodename):
    selList = OpenMaya.MSelectionList()
    selList.add(nodename)
    node = OpenMaya.MObject()
    selList.getDependNode(0, node)
    return node

def in_io():
    return OpenMaya.MFileIO.isOpeningFile() or OpenMaya.MFileIO.isReadingFile()

def get_from_dict(data_dict, map_list):

    return reduce(operator.getitem, map_list, data_dict)

def get_initial_representation(assemblynode):
    afn = OpenMaya.MFnAssembly(assemblynode)
    names = ['active']
    initial_rep = None

    while True:
        initialrep_plug = afn.findPlug('initialRepresentation')

        initial_data = json.loads(initialrep_plug.asString() or '{}')
        map_list = names
        try:
            initial_rep = get_from_dict(initial_data, map_list)
        except KeyError:
            pass

        names.insert(0, afn.name().rsplit(':', 1)[-1])
        parent = afn.getParentAssembly()
        if parent.isNull():
            break

        afn.setObject(parent)
        names.insert(0, afn.getActive())

    return initial_rep

def get_assembly_path(assemblynode):
    afn = OpenMaya.MFnAssembly(assemblynode)
    names = []

    while True:
        names.insert(0, afn.getActive())
        names.insert(0, afn.name().rsplit(':', 1)[-1])
        parent = afn.getParentAssembly()
        if parent.isNull():
            return names
        afn.setObject(parent)

def get_root_assembly(assemblynode):

    afn = OpenMaya.MFnAssembly(assemblynode)

    while True:
        parent = afn.getParentAssembly()
        if parent.isNull():
            return afn.object()
        afn.setObject(parent)


def children_iter(assemblynode, max_depth=99, _depth=0):

    yield assemblynode
    if _depth >= max_depth:
        return

    afn = OpenMaya.MFnAssembly(assemblynode)
    sub_assemblies = afn.getSubAssemblies()

    for idx in range(sub_assemblies.length()):
        for child_iter in children_iter(sub_assemblies[idx],
                                          max_depth=max_depth,
                                          _depth=_depth + 1):
            yield child_iter

def set_in_dict(data_dict, map_list, value):
    for key in map_list[:-1]:
        data_dict = data_dict.setdefault(key, {})
    data_dict[map_list[-1]] = value

def update_dict(d, u):
    import collections

    for k, v in u.items():
        if isinstance(d, collections.abc.Mapping):
            if isinstance(v, collections.abc.Mapping):
                r = update_dict(d.get(k, {}), v)
                d[k] = r
            else:
                d[k] = u[k]
        else:
            d = {k: u[k]}
    return d


def save_initial_representation(assemblynode):

    afn = OpenMaya.MFnAssembly(assemblynode)

    initialrep_plug = afn.findPlug('initialRepresentation')
    initial_data = json.loads(initialrep_plug.asString() or '{}')

    result = {}

    for child in children_iter(assemblynode):
        initial_rep = get_initial_representation(child)
        afn.setObject(child)
        active_rep = afn.getActive()
        if active_rep != initial_rep:
            path = get_assembly_path(child)[:-1]
            path.append('active')
            set_in_dict(result, path[1:], active_rep)

    update_dict(initial_data, result)
    initialrep_plug.setString(json.dumps(initial_data))


