import os
import weakref
import traceback

import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds




class Representation(OpenMayaMPx.MPxRepresentation):
    typename = 'Bake'

    def __init__(self, assembly, name, path):
        OpenMayaMPx.MPxRepresentation.__init__(self, assembly, name)
        self.path = path
        self.label = name
        self.name = name
        self._assembly = weakref.ref(assembly)

    def post_activate(self):
        pass

    def external_content(self):
        return [self.path]

    @property
    def assembly(self):
        o = self._assembly()
        if o is None:
            raise RuntimeError("Assemblynode no longer exists")
        else:
            return o

    def get_assembly_name(self):

        assembly_node = self.assembly.thisMObject()
        fnassembly = OpenMaya.MFnDagNode(assembly_node)
        return fnassembly.partialPathName()

    def get_assembly_namespace(self, absolute=True):

        assembly_node = self.assembly.thisMObject()
        fnassembly = OpenMaya.MFnAssembly(assembly_node)
        if absolute:
            return fnassembly.getAbsoluteRepNamespace()
        return fnassembly.getRepNamespace()

    def safe_activate(self):
        try:
            out = self.activate()
            return out
        except Exception:
            traceback.print_exc()
            OpenMaya.MGlobal.displayError('{}[{}] Failed to activate:'.format(
                self.get_assembly_name(), self.label))
            return False

    def supportsMemberChanges(self):
        return False

    def inactivate(self):
        if self.can_apply_edits():
            self.assembly.save_edits()

        assembly = self.assembly.thisMObject()
        fncontainer = OpenMaya.MFnContainerNode(assembly)
        members = OpenMaya.MObjectArray()
        fncontainer.getMembers(members)

        for i in range(members.length()):
            obj = members[i]
            fndep = OpenMaya.MFnDependencyNode(obj)
            fndep.setLocked(False)

        fncontainer.clear()
        #all_bellow = cmds.listRelatives(self.assembly.name(), ad=True)
        #cmds.delete(all_bellow)
        return True

    def can_apply_edits(self):
        return True

    def canApplyEdits(self):
        return True


class SetRepresentation(Representation):
    typename = "set"
    def __init__(self, assembly, name, path, renderpath=None, label=None):

        super(SetRepresentation, self).__init__(assembly, name, path)
        if label is None:
            label = os.path.basename(path)
        self.edit_path = None

    def create_asset(self, asset_data):
        node_name = asset_data['name']

        new_node = cmds.createNode('MyAssembly', n=node_name)
        model_path = asset_data['path']

        cmds.setAttr('%s.path' % new_node, model_path, type='string')
        shader_path = asset_data.get('shader_path', '')

        cmds.setAttr('%s.shadersPath' % new_node, shader_path, type='string')

        return new_node

    def set_transforms(self,new_node,  node_data):

    # cmds.setAttr('%s.path' % new_node, model_path, type='string')

        translate = node_data['translate']
        rotate = node_data['rotate']
        scale = node_data['scale']

        cmds.setAttr('%s.translateX' % new_node, translate[0])
        cmds.setAttr('%s.translateY' % new_node, translate[1])
        cmds.setAttr('%s.translateZ' % new_node, translate[2])

        cmds.setAttr('%s.rotateX' % new_node, rotate[0])
        cmds.setAttr('%s.rotateY' % new_node, rotate[1])
        cmds.setAttr('%s.rotateZ' % new_node,  rotate[2])

        cmds.setAttr('%s.scaleX' % new_node, scale[0])
        cmds.setAttr('%s.scaleY' % new_node, scale[1])
        cmds.setAttr('%s.scaleZ' % new_node, scale[2])

    def create_node(self, node_data, parent=None):
        node_name = node_data.get('name')
        asset_name = node_data.get('asset')

        if asset_name:
            new_node = self.create_asset(node_data)

        else:
            new_node = cmds.createNode('transform', name=node_name)

        if parent:
            cmds.parent(new_node, parent)

        for child_name, child_data in node_data.get('children', {}).items():
            self.create_node(child_data, new_node)

        #self.set_transforms(new_node, node_data)

        return new_node


    def apply_edit(self, set_data, edit_data):

        if isinstance(set_data, dict) and isinstance(edit_data, dict):
            new_dict = {}
            for key, value in set_data.items():
                if key in edit_data:
                    new_dict[key] = self.apply_edit(value, edit_data[key])
                else:
                    new_dict[key] = value
            for key, value in edit_data.items():
                if key not in set_data:
                    new_dict[key] = value

            return new_dict
        else:
            return edit_data


    def activate(self):
        import json
        with open(self.path) as json_file:
            data = json.load(json_file)

        if self.edit_path:
            with open(self.edit_path) as json_edit_file:
                update_data = json.load(json_edit_file)
                data = self.apply_edit(data, update_data)

        for key, value in data.items():

            node = self.create_node(value)

            cmds.parent(node, self.root_assembly.name(), s=True)



        return True

class GeoRepresentation(Representation):
    typename = "bake"

    def __init__(self, assembly, name, path, renderpath=None, label=None):
        if label is None:
            label = os.path.basename(path)

        super(GeoRepresentation, self).__init__(assembly, name, path)

    def activate(self):
        ## load everything
        node = cmds.AbcImport(self.path, mode='import', reparent=self.root_assembly.name())

        #if self.shaders_path and os.path.exists(self.shaders_path):
        #    self.apply_shaders()


        return True

    def can_apply_edits(self):
        """
        The SceneRepresentation supports edits
        """
        return True

class LocatorRepresentation(Representation):
    typename = "bake"

    def __init__(self, assembly, name, path, renderpath=None, label=None):
        if label is None:
            label = os.path.basename(path)

        super(LocatorRepresentation, self).__init__(assembly, name, path)
    def activate(self):
        ## load everything
        node = cmds.spaceLocator(name='Locator')
        cmds.parent(node, self.root_assembly.name(), s=True)
        return True



class VDBRepresentation(Representation):
    typename = "bake"

    def __init__(self, assembly, name, path, renderpath=None, label=None):
        if label is None:
            label = os.path.basename(path)

        super(VDBRepresentation, self).__init__(assembly, name, path)

    def activate(self):
        ## load everything
        node = cmds.createNode('RedshiftVolumeShape', name='volume', parent=self.root_assembly.name())
        cmds.setAttr('%s.fileName' % node, self.path, type='string')
        cmds.setAttr('%s.displayMode' % node, 2)
        cmds.setAttr('%s.maxPoints' % node, 200000)


        return True

