import logging
import importlib
import weakref


import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds

import maya_assemblies.lib.representations.locator_representation as locator_representation
import maya_assemblies.lib.helpers as helpers

importlib.reload(helpers)

logger = logging.getLogger(__name__)
logger.setLevel(logging.WARNING)

class MasterAssembly(OpenMayaMPx.MPxAssembly):
    typename = "MasterAssembly"
    id = OpenMaya.MTypeId()
    icon_name = "out_assemblyReference.png"

    # Maya Attributes
    agrid_uid = OpenMaya.MObject()
    active_representation = OpenMaya.MObject()
    arep_namespace = OpenMaya.MObject()
    ainitial_rep = OpenMaya.MObject()

    # Python Attributes
    representation_types = set()  # A set of supported representation types.
    representation_type_map = {locator_representation.LocatorRepresentation}
    tracking___dict = {}
    allow_transform = False

    def __del__(self):
        MasterAssembly.tracking_dict.pop(OpenMayaMPx.asHashable(self))

    def __init__(self):
        self.representations = {}
        self.active_rep = None
        self.is_updating_namespace = False
        self.is_activating = False

        self.is_loaded = False
        self.loader = None
        self.loader_attributes_callback = None
        self.set_attributes_callback = None
        self.connect_attributes_callback = None
        self.attribute_set_edits = {}
        self.attribute_connect_edits = {}
        self.applying_edit = False
        self.handle_edit = False
        OpenMayaMPx.MPxAssembly.__init__(self)
        self.tracking_dict[OpenMayaMPx.asHashable(self)] = weakref.ref(self)
        self._setDoNotWrite(True)

    @property
    def current_representation(self):
        plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)
        value = plug.asString()
        return value

    def initialize_assembly(self):
        path_plug = OpenMaya.MPlug(self.thisMObject(), self.adefinition_file)
        path = path_plug.asString()

        self.representations = {'Locator': locator_representation.LocatorRepresentation(self, 'Locator', path)}
        self._default_representation = 'Locator'

    @classmethod
    def initialize(cls):
        pass

    def supportsMemberChanges(self):
        return True
    def getRepNamespace(self):
        namespace_plug = OpenMaya.MPlug(self.thisMObject(), self.representation_namespace)
        namespace = namespace_plug.asString()
        if not namespace:
            namespace = self.get_default_namespace()
            namespace_plug.setString(namespace)
        return namespace

    @classmethod
    def add_boolean_attribute(cls, long_name, short_name, category=None, min=None, max=None, default=False):
        attr_fn = OpenMaya.MFnNumericAttribute()
        attribute = attr_fn.create(long_name, short_name, OpenMaya.MFnNumericData.kBoolean, default)
        if category is not None:
            attr_fn.addToCategory(category)
        cls.addAttribute(attribute)

        return attribute

    @classmethod
    def add_integer_attribute(cls, long_name, short_name, category=None, min=None, max=None, default=0):
        attr_fn = OpenMaya.MFnNumericAttribute()
        attribute = attr_fn.create(long_name, short_name, OpenMaya.MFnNumericData.kInt, 1)
        if min is not None:
            attr_fn.setMin(min)
        if max is not None:
            attr_fn.setMax(max)

        if category is not None:
            attr_fn.addToCategory(category)
        cls.addAttribute(attribute)

        return attribute

    @classmethod
    def add_string_attribute(cls, long_name, short_name, as_path=False, category=None):

        tAttr = OpenMaya.MFnTypedAttribute()

        attribute = tAttr.create(long_name, short_name, OpenMaya.MFnData.kString)
        tAttr.setStorable(True)
        tAttr.setWritable(True)
        if as_path:
            tAttr.setUsedAsFilename(True)
        if category is not None:
            tAttr.addToCategory(category)

        cls.addAttribute(attribute)

        return attribute


    @classmethod
    def creator(cls):
        return OpenMayaMPx.asMPxPtr(cls())

    @classmethod
    def representation_label(cls, rep_type):
        return rep_type

    @classmethod
    def list_representation_types(cls):

        return cls.representation_type_map.keys()

    def getActive(self):
        if self.active_rep:
            name = self.active_rep.name
            return name
        return ''

    def getRepresentations(self):
        return [x.name for x in self.representations.values()]

    def getRepType(self, rep_name):
        return self.representations[rep_name].typename

    def getRepLabel(self, rep_name):

        return self.representations[rep_name].label

    def repTypes(self):
        return []

    def deleteRepresentation(self, rep_name):
        raise RuntimeError

    def deleteAllRepresentations(self):
        for rep_name in self.representations.keys():
            self.deleteRepresentation(rep_name)

    def handlesAddEdits(self):
        return self.handle_edit

    def handlesApplyEdits(self):
        return False

    def get_edit_node(self, attribute):
        namespace = self.getRepNamespace()
        node = attribute.split('.', 1)[0]

        assembly_node = '%s:%s' % (namespace, node)
        if cmds.objExists(assembly_node):
            node = assembly_node

        return node

    def can_apply_to_representation(self):

        active_plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)
        active_representation = active_plug.asString()

        if not active_representation or not self.representations.get(active_representation):
            return

        apply = self.representations[active_representation].can_apply_edits()

        return apply

    def get_full_attribute(self, attribute):
        attribute = attribute.replace('"', '')

        node, attribute = attribute.rsplit('.', 1)
        namespace = self.getRepNamespace()
        assembly_node = '%s:%s' % (namespace, node)
        logger.debug('check: %s' % assembly_node)
        if cmds.objExists(assembly_node):
            logger.debug('%s exists' % assembly_node)
            node = assembly_node
        final_attr = '%s.%s' % (node, attribute)

        return final_attr

    def applyEdits(self):
        import maya.mel as mel
        if not self.can_apply_to_representation():
            return

        self.applying_edit = True
        root = helpers.get_root_assembly(self.thisMObject())
        iterator = OpenMaya.MItEdits(root, self.thisMObject(), OpenMaya.MItEdits.ALL_EDITS)

        while not iterator.isDone():
            edit = iterator.edit()
            command = edit.getString()
            logger.debug('edit command: %s' % command)
            cmd = command.split(' ')

            if cmd[0] == 'connectAttr':
                full_source_attribute = self.get_full_attribute(cmd[1])
                full_target_attribute = self.get_full_attribute(cmd[2])

                logger.debug('final source: %s' % full_source_attribute)
                logger.debug('final target: %s'% full_target_attribute)
                applied = False
                self.attribute_connect_edits[full_source_attribute] = full_target_attribute
                target_node = full_target_attribute.split('.')[0]
                #if not cmds.objExists(target_node):
                #    continue
                if cmds.objExists(target_node):
                    if cmds.nodeType(target_node) == 'shadingEngine':
                        if target_node != 'initialShadingGroup':
                            source_node = full_source_attribute.split('.')[0]
                            try:
                                cmds.sets(source_node, e=True, forceElement=target_node)
                                applied = True
                            except:
                                logger.warning('cant assign %s to %s' % (target_node, source_node))
                    else:
                        cmds.connectAttr(full_source_attribute, full_target_attribute)

                        applied = True

                    if applied == True:
                        iterator.removeCurrentEdit()

            elif cmd[0] == 'setAttr':
                full_source_attribute = self.get_full_attribute(cmd[1])

                self.attribute_set_edits[full_source_attribute] = cmd[-1]
                if command.find('[(') > -1:
                    command = command.split('[(')[0]
                if command.find(')]') > -1:
                    command = command.split(')]')[0]
                if command.endswith(' True'):
                    command = command[:-4] + ' true'
                command = command.replace('True', 'true')
                command = command.replace('False', 'false')

                bits = command.split(' ')
                if 'on' in bits:
                    index = bits.index('on')
                    bits.pop(index+1)
                    bits.pop(index)
                    command = ' '.join(bits)
                #print(command)
                mel.eval(command)
                iterator.removeCurrentEdit()

            iterator.next()

        self.applying_edit = False

    def supportsEdits(self):
        return False
    

    def inactivateRep(self):
        self.save_edits()
        OpenMayaMPx.MPxAssembly.inactivateRep(self)
        return True

    def activateRep(self, rep_name):
        self.is_activating = True
        active_plug = OpenMaya.MPlug(self.thisMObject(), self.active_representation)

        # Activation of an empty string is a no-op.
        if not rep_name:
            self.active_rep = None
            active_plug.setString('')
            return True

        selection = OpenMaya.MSelectionList()
        OpenMaya.MGlobal.getActiveSelectionList(selection)

        cmds.select(self.get_assembly_name(), r=True)
        to_activate = self.representations.get(rep_name)
        if not to_activate:
            return False

        if to_activate.safe_activate():
            self.active_rep = to_activate
            active_plug.setString(rep_name)
            cmds.flushUndo()
            OpenMaya.MGlobal.setActiveSelectionList(selection)
            return True
        return False

    def postActivateRep(self, rep_name):
        self.is_activating = False
        if not rep_name:
            return

        to_activate = self.representations.get(rep_name)

        if not to_activate:
            return

        to_activate.post_activate()
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        if afn.isTopLevel():
            self.applyEdits()
            self.add_callbacks()
        return True

    def get_mobject_from_name(self, name):
        sl = OpenMaya.MSelectionList()
        if not cmds.objExists(name):
            raise RuntimeError('Object does not exist: {}'.format(name))
        OpenMaya.MGlobal.getSelectionListByName(name, sl)
        node = OpenMaya.MObject()
        sl.getDependNode(0, node)
        return node

    def canRepApplyEdits(self, active_representation):
        if not active_representation:
            return False

        value = self.representations[active_representation].can_apply_edits()
        return value

    def beforeSave(self):
        print('before save', self.get_assembly_name())
        self.save_edits()


    def clean_attr(self, attribute):
        attribute = attribute.split('[')[0]

        bits = attribute.split('|')

        attribute = '|'.join(bit.split(':')[-1] for bit in bits)

        return attribute

    def create_connection_edit(self, node_name, source_connection):
        #print(node_name, source_connection)
        node_namespace = source_connection.split(':')[0]
        namespace = self.getRepNamespace()

        if node_namespace != namespace:
            source_connection = '%s:%s' % (namespace, source_connection)

        if not cmds.objExists(source_connection):
            source_connection = source_connection.split(':')[-1]
            if not cmds.objExists(source_connection):

                return

        input_connections = cmds.listConnections(source_connection, p=True, c=True, s=True, d=False)
        output_connections = cmds.listConnections(source_connection, p=True, c=True, d=True, s=False)

        #print('<<', input_connections)
        #print('>>', output_connections)

        reversed_order = False
        if input_connections:
            connections = input_connections
        elif output_connections:
            reversed_order = True
            connections = output_connections
        else:
            return

        for index in range(0, len(connections), 2):

            if reversed_order is False:
                source_attr = connections[index + 1]
                target_attr = connections[index]
            else:
                source_attr = connections[index]
                target_attr = connections[index + 1]

            #source_attr = self.clean_attr(source_attr)
            #target_attr = self.clean_attr(target_attr)
            self.addConnectAttrEdit('..:%s' % node_name, '..:%s' % source_attr, '..:%s' % target_attr)


    def save_edits(self):
        if not self.can_apply_to_representation():
            return
        node_name = self.get_assembly_name()

        self.handle_edit = True

        for source_connection in self.attribute_connect_edits:
            self.create_connection_edit(node_name, source_connection)

        for source_attr, value in self.attribute_set_edits.items():
            value = cmds.getAttr(source_attr)
            if source_attr in self.attribute_connect_edits:
                continue

            source_attr = source_attr.split(':', 1)[-1]
            if source_attr in self.attribute_connect_edits:
                continue

            source_attr = source_attr.split('[')[0]

            #value = value.split('[(')[0]


            self.addSetAttrEdit('..:%s' % node_name, '..:%s:%s' % (self.getRepNamespace(), source_attr), ' %s' % value)

        self.handle_edit = False

    def postLoad(self):
        self.reload_definition()
        self.is_loaded = True

        return True

    def reload_definition(self):
        afn = OpenMaya.MFnAssembly(self.thisMObject())

        active_rep = self.current_representation if active_rep is not None and active_rep in self.representations else self._default_representation

        afn.activate(active_rep)

    def preApplyEdits(self):
        if self.canRepApplyEdits(self.getActive()):
            self.autokeystate = cmds.autoKeyframe(q=True, state=True)
            cmds.autoKeyframe(state=False)

    def postApplyEdits(self):
        if self.canRepApplyEdits(self.getActive()):
            cmds.autoKeyframe(state=self.autokeystate)

    def preUnapplyEdits(self):
        self.autokeystate = cmds.autoKeyframe(q=True, state=True)
        cmds.autoKeyframe(state=False)

    def postUnapplyEdits(self):
        cmds.autoKeyframe(state=self.autokeystate)

    def get_default_namespace(self):
        name = self.name()
        if '____' in self.name():
            namespace, name = name.split('____')
            fndag = OpenMaya.MFnDagNode(self.thisMObject())
            fndag.setName(name)
            return namespace

        return OpenMayaMPx.MPxAssembly.getRepNamespace(self)

    def get_assembly_name(self):
        fnassembly = OpenMaya.MFnDagNode(self.thisMObject())
        return fnassembly.partialPathName()

