
import maya.cmds as cmds
from PySide import QtGui, QtWidgets

import shotgrid_lib.publish_entity as publish_entity
import shotgrid_lib.shotgrid_helpers as shotgrid_helpers


class SceneManager(QtWidgets.QFrame):

    def __init__(self, shot_name, parent=None, project=None):
        super(SceneManager, self).__init__(parent=parent)

        self.shot_name = shot_name
        self.project = project
        self.sg_connection = publish_entity.init_connection()
        self.node_info = self.get_elements_data()
        self.asset_versions = self.get_asset_versions()
        self.init_ui()
        self.set_value()

    def init_ui(self):

        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.main_layout = QtWidgets.QVBoxLayout()
        self.title_widget = QtWidgets.QLabel('Scene Manager:')

        self.value_widget = QtWidgets.QTableWidget()

        headers = ['Type', 'Asset', 'Variant', 'Current', 'Version', 'Status', 'node']
        self.value_widget.setColumnCount(7)
        self.value_widget.setColumnWidth(0, 50)
        self.value_widget.setColumnWidth(1, 100)
        self.value_widget.setColumnWidth(2, 50)
        self.value_widget.setColumnWidth(3, 50)
        self.value_widget.setColumnWidth(4, 80)
        self.value_widget.setColumnWidth(5, 50)

        self.value_widget.setHorizontalHeaderLabels(headers)
        self.main_layout.addWidget(self.value_widget)

        header = self.value_widget.horizontalHeader()
        header.setSectionResizeMode(6, QtWidgets.QHeaderView.ResizeMode.Stretch)
        #self.main_layout.addStretch(1)
        self.launch_button = QtWidgets.QPushButton('Update')
        self.launch_button.clicked.connect(self.update)
        self.main_layout.addWidget(self.launch_button)
        self.setLayout(self.main_layout)

    def set_value(self):
        self.value_widget.setRowCount(len(list(self.node_info.keys())))

        index = 0
        for item_name, shotgrid_data in self.node_info.items():
            if not shotgrid_data:
                continue

            self.value_widget.setItem(index, 6, QtWidgets.QTableWidgetItem(item_name))

            version_number = shotgrid_data['sg_version_number']
            variant_name = shotgrid_data['sg_variant']['name']
            publish_type_name = shotgrid_data['sg_publish_type']['name']
            if shotgrid_data['sg_asset']:
                asset_name = shotgrid_data['sg_asset']['name']
            else:
                asset_name = shotgrid_data['sg_context']['name']

            if asset_name not in self.asset_versions:
                continue

            version = self.asset_versions[asset_name][publish_type_name][variant_name][version_number]
            status = version['sg_status_list']

            if shotgrid_data['sg_asset']:
                self.value_widget.setItem(index, 1, QtWidgets.QTableWidgetItem(shotgrid_data['sg_asset']['name']))
            else:
                self.value_widget.setItem(index, 1, QtWidgets.QTableWidgetItem(shotgrid_data['sg_context']['name']))

            versions = []
            approved = None
            this_approved = False
            this_latest = False
            current_version_number = '%03d' % version_number

            self.value_widget.setItem(index, 0, QtWidgets.QTableWidgetItem(publish_type_name))
            self.value_widget.setItem(index, 2, QtWidgets.QTableWidgetItem(variant_name))
            self.value_widget.setItem(index, 3, QtWidgets.QTableWidgetItem('%03d' % version_number))
            self.value_widget.setItem(index, 5, QtWidgets.QTableWidgetItem(status))

            for version_n, version_data in reversed(sorted(self.asset_versions[asset_name][publish_type_name][variant_name].items())):
                if version_data['sg_status_list'] == 'cmpt' and approved is None:
                    approved = True

                    if '%03d' % version_n == current_version_number:
                        current_version_number = 'Approved'
                        this_approved = True
                versions.append('%03d' % version_n)

            if current_version_number == max(versions) and not this_approved:
                current_version_number = 'Latest'
                this_latest = True

            versions.insert(0, 'Latest')
            if approved:
                versions.insert(0, 'Approved')

            chooser = QtWidgets.QComboBox()
            chooser.addItems(versions)

            item_index = chooser.findText(current_version_number)
            chooser.setCurrentIndex(item_index)

            if this_latest:
                self.value_widget.item(index, 3).setBackground(QtGui.QColor(10, 10, 100))
            elif this_approved:
                self.value_widget.item(index, 3).setBackground(QtGui.QColor(10, 100, 10))
            else:
                self.value_widget.item(index, 3).setBackground(QtGui.QColor(100, 10, 10))
            self.value_widget.setCellWidget(index, 4, chooser)


            index += 1

    def get_elements_data(self):

        loader_nodes = {'Assembly': 'path',
                        'reference': ''
                        }
        all_paths = {}

        solved_paths = {}
        for node_type, attribute in loader_nodes.items():
            all_nodes = cmds.ls(type=node_type)
            for node in all_nodes:
                if node_type == 'reference':
                    path = cmds.referenceQuery(node, f=True)
                else:
                    path = cmds.getAttr('%s.%s' % (node, attribute))

                if path in solved_paths:
                    all_paths[node] = solved_paths[path]
                else:
                    data = publish_entity.get_data_from_path(self.sg_connection, path)
                    solved_paths[path] = data
                    all_paths[node] = data

        return all_paths

    def get_asset_versions(self):
        filters = [['project.Project.code', 'is', self.project],
                   ['sg_asset.Asset.sg_breakdowns.CustomEntity12.sg_link.Shot.code', 'is', self.shot_name]
                   #['sg_context.Shot.code', 'is', self.shot_name]
                   ]

        filters.append(['sg_complete', 'is', True])
        filters.append(['sg_delete', 'is', False])


        return_fields = ['id', 'code', 'sg_version_number', 'sg_asset', 'sg_task', 'project', 'sg_publish_type',
                         'sg_context', 'sg_status_list',
                         'sg_variant', 'project.Project.code', 'sg_task.Task.sg_variant', 'sg_breakdowns']

        published = self.sg_connection.find(shotgrid_helpers.PUBLISH_ENTITY, filters=filters, fields=return_fields)

        sorted_publish = {}
        for publish in published:
            asset_name = publish['sg_asset']['name']
            variant = publish['sg_variant']['name']
            publish_type = publish['sg_publish_type']['name']
            version_number = publish['sg_version_number']
            sorted_publish[asset_name] = sorted_publish.get(asset_name, {})
            sorted_publish[asset_name][publish_type] = sorted_publish[asset_name].get(publish_type, {})
            sorted_publish[asset_name][publish_type][variant] = sorted_publish[asset_name][publish_type].get(variant, {})
            sorted_publish[asset_name][publish_type][variant][version_number] = publish

        return sorted_publish