import getpass
import logging
import os

from PySide import QtWidgets, QtCore

import launcher.lib.set_environment as set_environment

import library.core.general_exceptions as general_exceptions
import library.ui.labeled as labeled
import shotgrid_lib.lib.progress_query_widget as progress_query_widget


logger = logging.getLogger(__name__)


class ContextWidget(QtWidgets.QFrame):
    precached_entities = []
    main_module = 'builder'

    context_changed = QtCore.Signal()
    task_changed = QtCore.Signal()

    def __init__(self, definition_type, parent=None, project=None):
        super(ContextWidget, self).__init__(parent=parent)
        self.definition_type = definition_type
        self.project = project

        self.project_environment = set_environment.ProjectEnvironment(project=self.project)

        self.project_config = self.project_environment.project_config

        if not self.project_config:
            raise general_exceptions.ProjectConfigNotFound(self.project)
        self.builder_config = self.project_environment.get_config_file(self.main_module, module=self.main_module)
        definition_path = 'definitions/%s' % definition_type
        self.definition = self.project_environment.get_config_file(definition_path,
                                                              module=self.main_module,
                                                              merge=False)

        if not self.definition:
            logger.error('Can\'t find definition of type: %s' % definition_type)
            raise general_exceptions.ConfigNotFound(definition_path,
                                                    self.main_module)
        self.definition_context = self.definition['definition_context']
        self.definition_app = self.definition.get('definition_app', 'maya')
        self.pipeline_step = definition_type

        self.user_login = getpass.getuser()
        self.context_name = self.get_context()

        self.progress_query = progress_query_widget.ProgressQuery(self.project, parent=self)

        base_precache_entities = [self.definition_context.title(), 'Task', 'CustomEntity11', 'Step']
        base_precache_entities += self.precached_entities
        self.progress_query.set_entity_prefilter('Task', [['step', 'name_is', self.pipeline_step]])
        self.progress_query.set_precache_entities(['Step', self.definition_context.title(), 'Task'])
        self.database = self.progress_query.database
        self.progress_query.finished.connect(self.init_context_widget)

        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.app_layout = QtWidgets.QVBoxLayout()
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(10, 10, 10, 5)

        self.current_task = None
        self.add_context_widget(self.main_layout)
        self.add_task_widget(self.main_layout)

        self.app_layout.addLayout(self.main_layout)

        self.setLayout(self.app_layout)

    def start(self):
        self.progress_query.start()

    def add_context_widget(self, layout):
        context_labels = self.builder_config['context_labels'][self.definition_context]
        self.context_widget = labeled.CascadeComboWidget(self.context_name,
                                                         self.definition_context,
                                                         combo_labels=context_labels)

        self.context_widget.set_width(100)
        layout.addWidget(self.context_widget)
        #self.context_widget.setEnabled(False)
        logging.info('set context to : %s' % self.context_name)
        self.context_widget.setValue(self.context_name)
        self.context_widget.modified.connect(self.update_task)

    def add_task_widget(self, layout):
        self.task_widget = labeled.LabeledCombo('', 'Task')
        self.task_widget.set_width(100)
        layout.addWidget(self.task_widget)

    def add_last_version_widget(self, layout):
        self.last_version_widget = labeled.LabeledBoolean(True, 'Apply last version')
        self.last_version_widget.set_width(100)
        layout.addWidget(self.last_version_widget)

    def init_context_widget(self):
        self.database.query_sg_database('Task', update=True)
        step_view = self.database['Step'].find_with_filters(code=self.pipeline_step,
                                                            entity_type=self.definition_context.title())
        self.tasks_view = self.database['Task'].find_with_filters(step=step_view)
        all_context = {}

        self.tasks_view.precache_dependencies(fields=['entity'])

        filters = [['login', 'is', self.user_login]]
        self.database.query_sg_database('HumanUser', filters=filters)
        user_view = self.database['HumanUser'].find_with_filters(login=self.user_login, single_item=True)
        self.tasks_view.precache_dependencies(fields=['sg_sequence'])
        for task in self.tasks_view:
            entity_view = task.entity

            if not user_view.empty and user_view not in task.task_assignees and user_view.department_name not in ['R&D']:
                continue
            entity_name = entity_view.code
            if entity_view.type != self.definition_context.title():
                continue

            if entity_view.type == 'Asset':
                asset_type = entity_view.sg_asset_type
                all_context[asset_type] = all_context.get(asset_type, [])
                if entity_name not in all_context[asset_type]:
                    all_context[asset_type].append(entity_name)

            elif entity_view.type == 'Shot':

                shot_name = entity_name
                sequence_name = entity_view.sg_sequence_name
                episode_name = entity_view.sg_sequence.episode_name

                all_context[episode_name] = all_context.get(episode_name, {})
                all_context[episode_name][sequence_name] = all_context[episode_name].get(sequence_name, [])
                all_context[episode_name][sequence_name].append(shot_name)

            elif entity_view.type == 'Episode':
                all_context[entity_name] = None


        if entity_view.type == 'Episode':
            all_context = list(all_context.keys())

        self.context_widget.setValues(all_context)
        #self.context_widget.setEnabled(True)
        self.context_widget.setValue(self.context_name)
        QtWidgets.QApplication.restoreOverrideCursor()

    def get_context(self):
        context_type = os.environ.get('PIPE_CONTEXT')
        if not context_type or context_type.lower() != self.definition_context.lower():
            return ''

        if context_type == 'Asset':
            asset_type = os.environ.get('PIPE_ASSET_TYPE')
            asset_name = os.environ.get('PIPE_ASSET_NAME')
            if asset_type and asset_name:
                return '%s.%s' % (asset_type, asset_name)

        elif context_type == 'Shot':
            episode_name = os.environ.get('PIPE_EPISODE')
            sequence_name = os.environ.get('PIPE_SEQUENCE')
            shot_name = os.environ.get('PIPE_SHOT')

            if episode_name and sequence_name and shot_name:
                return '%s.%s.%s' % (episode_name, sequence_name, shot_name)


        return ''


    def get_current_context(self):
        value = self.context_widget.getValue()
        if not value.get(self.definition_context) and not value.get('name'):
            self.task_widget.setValues([])
            return None, None

        logger.debug('context type: %s' % self.definition_context)
        if self.definition_context == 'shot':
            entity_name = value['shot']
            context = self.database.entity_view('Shot')[entity_name]

        elif self.definition_context == 'episode':
            entity_name = value['episode']
            context = self.database.entity_view('Episode')[entity_name]

        elif self.definition_context == 'sequence':
            entity_name = value['sequence']
            context = self.database.entity_view('Sequence')[entity_name]

        elif self.definition_context == 'asset':
            entity_name = value['name']
            context = self.database.entity_view('Asset')[entity_name]

        else:
            return None, None
        context.precache_dependencies(fields=['tasks'])

        return context, entity_name

    def update_task(self):

        self.context_changed.emit()

        context, entity_name = self.get_current_context()
        if not entity_name:
            self.task_widget.setValues([])
            return []

        task_list = context.tasks

        pipeline_step_view = self.database['Step'].find_with_filters(code=self.pipeline_step,
                                                                     entity_type=self.definition_context.title(),
                                                                     single_item=True
                                                                     )
        current_tasks = task_list.find_with_filters(entity=context, step=pipeline_step_view)
        if current_tasks.empty:
            return ''
        task_list = current_tasks.content
        if not isinstance(task_list, list):
            task_list = [task_list]

        self.task_widget.setValues(task_list)
        if os.environ.get('PIPE_TASK'):
            self.task_widget.setValue(os.environ['PIPE_TASK'])

        if self.definition_context == 'shot' and self.start_widget is not None:
            self.start_widget.setValue(context.sg_cut_in)
            self.end_widget.setValue(context.sg_cut_out)

        self.task_widget.currentIndexChanged.connect(self.emit_task_changed)
        self.emit_task_changed()
        return task_list

    def emit_task_changed(self):
        current_task_name = self.task_widget.getValue()
        context, entity_name = self.get_current_context()
        self.current_task = context.tasks[current_task_name]
        self.task_changed.emit()
