from PySide import QtWidgets, QtGui, QtCore


class TreeNode(object):
    def __init__(self, data, parent=None):
        self.parentItem = parent
        self.itemData = data
        self.children = []

    def child(self, row):
        return self.children[row]

    def childCount(self):
        return len(self.children)

    def childNumber(self):
        if self.parentItem is not None:
            return self.parentItem.children.index(self)

    def columnCount(self):
        return len(self.itemData)

    def data(self, column):
        return self.itemData[column]

    def insertChildren(self, position, count, columns):
        if position < 0 or position > len(self.children):
            return False

        for row in range(count):
            data = ['' for v in range(columns)]
            item = self.__class__(data, self)
            self.children.insert(position, item)

    def parent(self):
        return self.parentItem

    def setData(self, column, value):
        if column < 0 or column >= len(self.itemData):
            return False

        self.itemData[column] = value


class TreeModel(QtCore.QAbstractItemModel):
    def __init__(self, headers, data, parent=None, tree_node_class=TreeNode):
        super(TreeModel, self).__init__(parent)
        """ subclassing the standard interface item models must use and 
                implementing index(), parent(), rowCount(), columnCount(), and data()."""
        root_data = [header for header in headers]
        self.tree_node_class = tree_node_class
        self.rootItem = self.tree_node_class(root_data)
        indent = -1
        self.parents = [self.rootItem]
        self.indentations = [0]
        self.createData(data, indent)

    def flags(self, index):
        return QtCore.Qt.ItemIsSelectable|QtCore.Qt.ItemIsEnabled

    def createData(self, data, indent):
        raise NotImplementedError('need reimplement create data function')

    def index(self, row, column, index=QtCore.QModelIndex()):
        """ Returns the index of the item in the model specified by the given row, column and parent index """

        if not self.hasIndex(row, column, index):
            return QtCore.QModelIndex()
        if not index.isValid():
            item = self.rootItem
        else:
            item = index.internalPointer()

        child = item.child(row)
        if child:
            return self.createIndex(row, column, child)
        return QtCore.QModelIndex()

    def get_parent(self, position):
        if position > self.indentations[-1]:
            if self.parents[-1].childCount() > 0:
                self.parents.append(self.parents[-1].child(self.parents[-1].childCount() - 1))
                self.indentations.append(position)
        else:
            while position < self.indentations[-1] and len(self.parents) > 0:
                self.parents.pop()
                self.indentations.pop()

        parent = self.parents[-1]
        return parent


    def parent(self, index):
        """ Returns the parent of the model item with the given index
                If the item has no parent, an invalid QModelIndex is returned """

        if not index.isValid():
            return QtCore.QModelIndex()
        item = index.internalPointer()
        if not item:
            return QtCore.QModelIndex()

        parent = item.parentItem
        if parent == self.rootItem:
            return QtCore.QModelIndex()
        else:
            return self.createIndex(parent.childNumber(), 0, parent)

    def rowCount(self, index=QtCore.QModelIndex()):
        """ Returns the number of rows under the given parent
                When the parent is valid it means that rowCount is returning the number of children of parent """

        if index.isValid():
            parent = index.internalPointer()
        else:
            parent = self.rootItem
        return parent.childCount()

    def columnCount(self, index=QtCore.QModelIndex()):
        """ Returns the number of columns for the children of the given parent """

        return self.rootItem.columnCount()

    def data(self, index, role=QtCore.Qt.DisplayRole):
        """ Returns the data stored under the given role for the item referred to by the index """
        if not index.isValid():
            return
        if role == QtCore.Qt.CheckStateRole:
            return index.internalPointer().enabled

        if role == QtCore.Qt.DisplayRole:
            return index.internalPointer().data(index.column())

    def setData(self, index, value, role):
        if role == QtCore.Qt.CheckStateRole:
            checked = value == QtCore.Qt.Checked
            index.internalPointer().enabled = checked
            return True

    def headerData(self, section, orientation, role=QtCore.Qt.DisplayRole):
        """ Returns the data for the given role and section in the header with the specified orientation """

        if orientation == QtCore.Qt.Horizontal and role == QtCore.Qt.DisplayRole:
            return self.rootItem.data(section)


