import os
import re
from pprint import pprint



class Shot(object):
    def __init__(self, shot_name=None, project='TPT'):
        self.shot_name = shot_name
        self.project = project
        self.set_config()
        if not self.shot_name:
            self.get_shot_from_environment()

        if self.shot_name:
            self.get_shot_entities()


    def build_shot_name(self, separator='_'):
        current_bits = []

        for entity in self.entities_order:
            current_bits.append(self.shot_entities[entity])

        shot_name = separator.join(current_bits)
        return shot_name

    def get_shot_from_environment(self):
        self.shot_entities = {}
        if 'PIPE_EPISODE' in os.environ and os.environ['PIPE_EPISODE']:
            self.shot_entities['episode'] = os.environ['PIPE_EPISODE']

        if 'PIPE_SEQUENCE' in os.environ and os.environ['PIPE_SEQUENCE']:
            self.shot_entities['sequence'] = os.environ['PIPE_SEQUENCE']

        if 'PIPE_SHOT' in os.environ and os.environ['PIPE_SHOT']:
            self.shot_entities['shot'] = os.environ['PIPE_SHOT']

        if self.is_valid():
            self.shot_name = self.build_shot_name()

    def set_config(self):
        import library.core.config_manager as config_manager

        config_manager = config_manager.ConfigSolver(project=self.project)
        project_config = config_manager.project_config

        self.format_config = project_config.get('paths', {}).get('shots', {}).get('formats', {})

        self.entities_order = project_config.get('paths', {}).get('shots', {}).get('entity_order', [])

    def is_valid(self):
        return self.shot_entities.keys() == self.format_config.keys()

    def get_shot_entities(self):
        pattern_list = self.convert_to_match()
        found = {}
        for pattern_str in pattern_list:
            pattern = re.compile(pattern_str)
            match = pattern.search(self.shot_name)
            if match:
                found.update(match.groupdict())
        self.shot_entities = found
        return found

    @staticmethod
    def end_block(this_block):
        last_char = this_block[-1]
        length = len(this_block)
        if last_char == 'x':
            pattern = '[a-zA-Z]{%i}' % length
        elif last_char == '0':
            pattern = '[0-9]{%i}' % length
        else:
            pattern = this_block

        return pattern

    @staticmethod
    def get_block(string, entity):
        if string == 'X':
            return '[a-zA-Z0-9]*'
        blocks = []
        this_block = ''
        for char in string:
            if not this_block or char == this_block[-1]:
                this_block += char
            else:
                pattern = Shot.end_block(this_block)
                blocks.append(pattern)
                this_block = char
        pattern = Shot.end_block(this_block)
        blocks.append(pattern)
        merged_blocks = ''.join(blocks)
        merged_blocks = '(?P<%s>%s)' % (entity, merged_blocks)
        return merged_blocks

    def convert_to_match(self):
        block_list = []
        bits = self.format_config.values()
        entities = self.format_config.keys()

        for bit, entity in zip(bits, entities):
            block = self.get_block(bit, entity)
            block_list.append(block)
        return block_list


    def get_full_shot_entities(self):

        if not self.is_valid():
            print('not valid shot name')
            return False

        current_bits = []
        folders = []
        for entity in self.entities_order:
            current_bits.append(self.shot_entities[entity])
            current_folder = '_'.join(current_bits)
            folders.append(current_folder)

        return folders

def test():
    shot_name = 'ep000_sq010_sh010'
    shot_name = 's00_ep01_sq210_sh020'
    project = 'TPT'
    shot_solver = Shot(shot_name=shot_name, project=project)
    entities = shot_solver.get_full_shot_entities()
    print(entities)


def test_no_name():
    project = 'TPT'
    os.environ['PIPE_EPISODE'] = 's00_ep01'
    os.environ['PIPE_SEQUENCE'] = 'sq210'
    os.environ['PIPE_SHOT'] = 'sh020'
    shot_solver = Shot( project=project)
    print(shot_solver.shot_name)

if __name__ == '__main__':
    test_no_name()