import os
import yaml
import logging
from warnings import warn
from pathlib import Path
import importlib.util as ip_util


import library.core.references_solver as references_solver

logger = logging.getLogger(__name__)

preloaded_configs = {'project_config': 'project', 'project_tools': 'project_tools'}


class ConfigSolver(object):
    """
    Configuration solver
    """
    _instance = {}

    def __new__(cls, *args, **kwargs):
        project = kwargs['project']
        if project not in cls._instance :
            logger.debug('Creating the ConfigSolver class')
            cls._instance[project] = super(ConfigSolver, cls).__new__(cls)

        return cls._instance[project]

    def __init__(self, project=''):
        self.project = project
        logger.debug('init for project : %s' % self.project)

        self.company_config = self.get_global_config()
        self.dev_path = self.company_config['dev_search_path']
        self.search_paths = self.company_config['search_order']



    @property
    def project_config(self):
        project_config = self.get_config('project')
        print('WARNING!!!: ConfigSolver.project_config deprecated, use the set_environment.project_config instead')
        warn('WARNING!!!: ConfigSolver.project_config deprecated, use the set_environment.project_config instead',
             DeprecationWarning,
             stacklevel=2)
        return project_config

    @property
    def project_tools(self):
        project_tools = self.get_config('project_tools')
        print('WARNING!!!: ConfigSolver.project_tools deprecated, use the set_environment.project_tools instead')
        warn('WARNING!!!:  ConfigSolver.project_tools deprecated, use the set_environment.project_tools instead',
             DeprecationWarning,
             stacklevel=2)

        return project_tools

    def override_config(self, config, overrides):
        for key, value in overrides.items():
            value_config = config.get(key)
            if isinstance(value, dict) and isinstance(value_config, dict):
                self.override_config(value_config, value)
            else:
                config[key] = value

        return config

    def get_config_files_of_type(self, config_type, module=None, version=None, tool_type='default'):
        config_folders = self.get_all_config_files(config_type, module=module, tool_version=version, tool_type=tool_type, as_folder=True)
        all_config = []
        for folder in config_folders:
            for config_file in os.listdir(folder):
                if not config_file.endswith('.yaml'):
                    continue

                config_name = config_file.split('.')[0]
                if config_name not in all_config:
                    all_config.append(config_name)

        return(all_config)



    def get_project_tools(self, overrides=''):
        return self.get_project_tools_version(overrides=overrides)

    def get_project_tools_version(self, overrides=''):
        self.overrides = self.get_config('test_settings/%s' % overrides)

        if self.overrides:
            self.override_config(self.project_tools, self.overrides)

        return self.project_tools


    def get_search_paths(self, tool_name, tool_version, tool_type='default'):
        """
        get_search_paths

        """
        if tool_version == 'local':
            search_paths = self.dev_path.get(tool_type, {}).copy()
        else:
            search_paths = self.search_paths.get(tool_type, {}).copy()

        solver_data = self.company_config.copy()
        solver_data['tool_name'] = tool_name
        solver_data['tool_version'] = tool_version

        solver_data['launcher_version'] = os.environ['LAUNCHER_VERSION']

        paths_dict = {str(index):path for index, path in enumerate(search_paths)}

        solved_paths = references_solver.ReferencesSolver.parse_dict(paths_dict, aditional_values=solver_data)
        solved_paths = list(solved_paths.values())
        return solved_paths
    
    def get_global_config(self):
        company_root = os.environ.get('COMPANY_ROOT', r'\\company/company')
        global_config_path = os.path.join(company_root, 'config', 'global.yaml')

        self.global_config_root = os.path.dirname(global_config_path)
        logger.debug('Global_config_path: %s' % global_config_path)
        data = self.read_file(global_config_path)
        project_root = data.get('projects_roots', {}).get(self.project.lower())
        if project_root:
            data['project_root'] = project_root
            logger.debug('project_root: %s' % project_root)

        return data


    def read_file(self, path):
        with open(path, 'r') as config_file:
            data = yaml.safe_load(config_file)

        solved_data = references_solver.ReferencesSolver.parse_dict(data)

        return solved_data
   
    def overwrite_config(self, config_files):
        out_config = {}
        for config_file in config_files:
            data = self.read_file(config_file)
            out_config.update(data)
        return out_config
    

    def get_all_config_files(self, config_type, module=None, tool_version=None, tool_type=None, as_folder=False):
        if module:
            tool_name = module.replace('.', '/')
            if tool_version is None:
                tool_version = os.environ.get('%s_VERSION' % module.upper())
        else:
            tool_name = None
        logger.debug('Get all config files: %s %s %s %s ' % (config_type, tool_name, tool_version, module))

        search_folders = self.get_search_paths(tool_name, tool_version, tool_type=tool_type)
        if module:
            module_spec = ip_util.find_spec(module)
            if module_spec and module_spec.origin:
                module_path = os.path.dirname(os.path.abspath(module_spec.origin))

                if os.path.isfile(module_path):
                    module_path = os.path.dirname(module_path)
                module_path = os.path.join(module_path, 'config')

        for folder in search_folders:
            logger.debug('Search_folder: %s' % folder)

        if not config_type.endswith('yaml') and as_folder is False:
            config_type = '%s.yaml' % config_type

        all_paths = []
        for search_path in search_folders:
            full_path = os.path.join(search_path, config_type)
            full_path = Path(full_path)
            if full_path.exists():
                all_paths.append(full_path)

        return all_paths

    def get_config(self, config_type, merge=False, module=None, version=None, tool_type='default'):
        self.last_config_path = ''
        if os.environ['LAUNCHER_VERSION'] == 'local' and tool_type == 'tools':
            version = 'local'

        config_files = self.get_all_config_files(config_type, module=module, tool_version=version, tool_type=tool_type)
        if not config_files:
            return {}

        if merge:
            global_config_data = self.overwrite_config(config_files)
        else:
            self.last_config_path = config_files[-1]
            global_config_data = self.read_file(config_files[-1])

        solved_data = references_solver.ReferencesSolver.parse_dict(global_config_data)


        return solved_data


if __name__ == '__main__':
    from pprint import pprint

    overrides = 'local_test'
    solver = ConfigSolver(project='TPT')

    pprint(solver.company_config)
    pprint(solver.dev_path)
    pprint(solver.search_paths)

    project_tools = solver.get_project_tools(overrides=overrides)
    pprint(project_tools)
