import os
import sys
import logging
import importlib
from collections import OrderedDict

from pprint import pprint
from PySide import QtCore, QtGui, QtWidgets

import usd.lib.usd_manager as usd_manager
importlib.reload(usd_manager)

logger = logging.getLogger(__name__)
logger.setLevel(logging.WARNING)

class UsdLayerItem(QtWidgets.QTreeWidgetItem):
    def __init__(self, layer_name, pipeline_step, config, active=True, parent=None, update=False):
        super().__init__()
        self.setFlags(self.flags() & QtCore.Qt.ItemIsSelectable)

        self.tree_widget = parent
        self.layer_name = layer_name
        self.pipeline_step = pipeline_step
        self.allow_update = update
        self.all_variants = {}
        self.config = config
        self.published_elements = config.get('published_items', {})
        self.current_variant = config.get('variant', 'Master')
        self.current_version = config.get('version_number', 0)
        self.status = 5
        self.approved_version = None
        self.latest_version = None
        self.recommended_version = None
        self.init_variant = config.get('variant', 'Master')
        self.init_version = config.get('version_number', 0)

        self.entity_type = self.config.get('type')
        self.asset_name = config.get('code')
        self.setText(0, layer_name)

        self.variant_widget = QtWidgets.QComboBox()
        self.version_widget = QtWidgets.QComboBox()
        self.active = active
        if self.tree_widget.active_option:
            if self.active:
                self.setCheckState(0, QtCore.Qt.Checked)
            else:
                self.setCheckState(0, QtCore.Qt.Unchecked)


    def set_variants(self, variants_data):
        self.all_variants = variants_data
        variants = list(self.all_variants.keys())
        versions = []
        for int_version in self.all_variants[self.current_variant]['versions']:
            if isinstance(int_version, str):
                versions.append(int_version)
            else:
                versions.append('%03d' % int_version)
        versions.insert(0, 'latest')
        self.latest_version = '%03d' % self.all_variants[self.current_variant]['latest']

        versions.insert(0, 'recommended')
        self.recommended_version = '%03d' % self.all_variants[self.current_variant]['recommended']

        if self.all_variants[self.current_variant]['approved']:
            versions.insert(0, 'approved')
            self.approved_version = '%03d' % self.all_variants[self.current_variant]['approved']
        else:
            self.approved_version = None

        versions = sorted(versions)
        variants = sorted(variants)
        self.set_combo_value(self.variant_widget, self.current_variant, values=variants)
        self.set_combo_value(self.version_widget, self.current_version, values=versions)


    def set_combo_value(self, combo_widget, value, values=None):
        if values is not None:
            combo_widget.clear()
            combo_widget.addItems(values)
        print(value, values)
        selected_index = combo_widget.findText(value)
        if selected_index is None:
            return
        combo_widget.setCurrentIndex(selected_index)

    def set_combo(self):
        variant_name = '%s_variant' % self.layer_name

        variants = self.config.get('variants', [])
        if not variants:
            variants = self.config.get('published_items', {}).keys()
        if not variants:
            variants = [self.current_variant]

        print(self.layer_name, self.pipeline_step)
        print('Variant', self.current_variant, variants)

        self.set_combo_value(self.variant_widget, self.current_variant, values=variants)
        self.variant_widget.currentTextChanged.connect(self.update_version)
        self.version_widget.currentTextChanged.connect(self.set_version)

        self.tree_widget.setItemWidget(self, 1, self.variant_widget)
        self.tree_widget.setItemWidget(self, 2, self.version_widget)

        self.setText(3, self.config.get('loader_type', 'NaN'))
        self.setText(4, self.config.get('type', '----'))

        self.update_version(self.current_variant)

        if self.allow_update:
            self.update_button = QtWidgets.QPushButton('Update')
            self.update_button.clicked.connect(self.update_asset)
            self.tree_widget.setItemWidget(self, 5, self.update_button)


    def update_asset(self):

        variant = str(self.variant_widget.currentText())
        version = str(self.version_widget.currentText())
        if version.isdigit():
            version = int(version)
        else:
            version = self.all_variants[variant].get(version, 0)

        database = self.tree_widget.parent().database

        entity = database[self.config['type']][self.config['code']]
        published_elements = entity.sg_published_elements
        variant_view = database['CustomEntity11'][variant]
        step_view = database['Step'][self.layer_name]
        published_version = published_elements.find_with_filters(sg_variant=variant_view,
                                                                 sg_version_number=version,
                                                                 sg_step=step_view,
                                                                 single_item=True)

        if published_version.empty:
            return
        if self.config['loader_type'] == 'rig assembly':
            self.update_rig(self.config['node'], published_version)

        if self.init_version == version and self.init_variant == variant:
            return

    def update_version(self, variant_name):
        self.current_variant = variant_name
        if isinstance(self.current_version, int):
            versions = ['%03d' % self.current_version]
        else:
            versions = [self.current_version]

        if self.published_elements:
            variant_data = self.published_elements.get(self.current_variant, {})

            versions_view = variant_data.get('versions')
            if versions_view and not versions_view.empty:
                versions = ['%03d' % ver for ver in versions_view.sg_version_number]

        variant_name = '%s_version' % self.layer_name
        selected = self.config.get('version_number', 0)
        if isinstance(selected, int):
            selected = '%03d' % selected
        self.set_combo_value(self.version_widget, selected, values=versions)


    def set_variant(self, variant):
        self.current_variant = variant
        self.set_combo_value(self.variant_widget, variant)

    def set_version(self, version):

        if isinstance(version, int):
            version = '%03d' % version
        self.current_version = version
        self.set_combo_value(self.version_widget, version)

        if version == 'approved' or version == self.approved_version:
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5

        elif version == self.recommended_version or version == 'recommended':
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5
        elif self.approved_version is None and (version != 'latest' and version != self.latest_version):
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("red")))
            self.status = 1


        elif self.approved_version and (version != 'approved' or version != self.approved_version):
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("red")))
            self.status = 1

        elif self.approved_version is None and version == 'latest' or version == 'recommended':
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("green")))
            self.status = 5

        else:
            #self.setForeground(0, QtGui.QBrush(QtGui.QColor("orange")))
            self.status = 3

        self.tree_widget.check_status()

    def set_active(self, active):
        self.active = active
        if self.active:
            self.setCheckState(0, QtCore.Qt.Checked)
        else:
            self.setCheckState(0, QtCore.Qt.Unchecked)


class UsdLayersTree(QtWidgets.QTreeWidget):
    def __init__(self, parent=None, project=None, update=False, active_option=True):
        super().__init__(parent=parent)
        self.setColumnCount(6)
        self.active_option = active_option

        self.project = project
        self.allow_update = update
        header = self.headerItem()
        header.setText(0, 'Layer')
        header.setText(1, 'Variant')
        header.setText(2, 'Version')
        header.setText(3, 'Loader')
        header.setText(4, 'Type')
        header.setText(5, 'Action')

        self.manager = usd_manager.UsdManager(self.project)


    def check_item_status(self, item):
        child_count = item.childCount()
        if child_count == 0:
            return item.status

        child_status_list = []
        for index in range(child_count):
            child_item = item.child(index)
            child_status = self.check_item_status(child_item)
            child_status_list.append(child_status)

        child_status_list = list(set(child_status_list))
        if len(child_status_list) == 1:
            status = child_status_list[0]
        else:
            status = 3
        return status
        if status == 5:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('green')))
        elif status == 3:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('orange')))
        elif status == 1:
            item.setForeground(0, QtGui.QBrush(QtGui.QColor('red')))


        return status



    def check_status(self):
        item_count = self.topLevelItemCount()

        for index in range(item_count):
            item = self.topLevelItem(index)
            self.check_item_status(item)

    def clear_widget(self):
        iterator = QtWidgets .QTreeWidgetItemIterator(self, QtWidgets .QTreeWidgetItemIterator.All)
        while iterator.value():
            iterator.value().takeChildren()
            iterator += 1
        i = self.topLevelItemCount()
        while i > -1:
            self.takeTopLevelItem(i)
            i -= 1

    def update(self, config=None):
        if not config:
            config = {}
        self.config = config
        self.clear_widget()
        for asset_name_instance, asset_layers in self.config.items():
            self.add_layers(asset_name_instance, asset_layers)

    def get_layer_from_step(self, step):
        all_steps, layer_data = self.manager.check_layer_in_assembly(step)
        return all_steps, layer_data

    def add_layers(self, asset_name, config, root=None, entity_type='Asset'):
        root = UsdLayerItem(asset_name, None, config,  parent=self)
        self.addTopLevelItem(root)
        for instance_name, instance_data in config.items():
            if 'id' not in instance_data :
                instance_item = UsdLayerItem(instance_name, None, instance_data, parent=self)
                root.addChild(instance_item)

                for pipeline_step, step_data in instance_data.items():
                    item = UsdLayerItem(pipeline_step, pipeline_step, step_data, parent=self, update=self.allow_update)
                    instance_item.addChild(item)

                    item.set_combo()
            else:
                item = UsdLayerItem(instance_name, instance_name, instance_data, parent=self, update=self.allow_update)
                root.addChild(item)

                item.set_combo()

    def getValue(self):
        data = OrderedDict()
        item_count = self.topLevelItemCount()

        for index in range(item_count):
            item = self.topLevelItem(index)
            asset_name = item.layer_name
            data[asset_name] = {}
            if not item.childCount():
                continue
            for instance_index in range(item.childCount()):
                instance_item = item.child(instance_index)
                asset_alias = instance_item.layer_name
                if not instance_item.childCount():
                    continue
                data[asset_name][asset_alias] = {}

                for child_index in range(instance_item.childCount()):
                    child = instance_item.child(child_index)

                    pipeline_step = child.layer_name
                    data[asset_name][asset_alias][pipeline_step] = {'active': child.active,
                                                                    'code': asset_name,
                                                                    'step': pipeline_step,
                                                                    'type': child.entity_type}
                    if child.current_variant:
                        data[asset_name][asset_alias][pipeline_step]['variant'] = child.current_variant
                    if child.current_version:
                        current_version = child.current_version
                        if not child.current_version.isdigit():
                            print(asset_name)
                            print(child.latest_version, child.approved_version, child.recommended_version)
                            if current_version == 'latest':
                                current_version = child.latest_version
                            elif current_version == 'approved':
                                current_version = child.approved_version
                            elif current_version == 'recommended':
                                current_version = child.recommended_version
                        if current_version is None:
                            current_version = 0
                        data[asset_name][asset_alias][pipeline_step]['version'] = current_version


        return data


def test_builder():
    import builder.lib.builder_core as builder_core
    importlib.reload(builder_core)
    app = QtWidgets.QApplication(sys.argv)
    project = 'SGD'

    path = 'V:/SGD/publish/usd/assets/Main_Characters/LeoMesi/asset_assembly.usda'
    asset_name = 'LeoMesi'
    entity_name = 's00_ep00_sq010_sh010'
    entity_type = 'Shot'
    pipeline_step = 'Layout'
    builder_core = builder_core.AssetBuilder(pipeline_step, project=project)
    variant_name = 'Master'
    dependencies = builder_core.set_context(entity_name, variant=variant_name)

    window = UsdLayersTree()
    window.update(config=dependencies)
    window.show()
    app.exec_()

def test_publisher():
    import packages_io
    importlib.reload(packages_io)

    project = 'SGD'
    dependencies = packages_io.scan_file()
    pprint(dependencies)
    window = UsdLayersTree()
    window.update(config=dependencies)
    #pprint(window.getValue())
    window.show()


if __name__ == '__main__':
    test_builder()