import os
import importlib
from pprint import pprint

from PySide import QtCore, QtWidgets, QtGui

class ToolCard(QtWidgets.QFrame):
    tool_launched = QtCore.Signal(str)

    def __init__(self, label, tools_data, parent=None):
        super(ToolCard, self).__init__(parent=parent)
        self.tool_data = tools_data
        label = self.tool_data.get('label', label)
        self.name = label

        tool_tip = self.tool_data.get('tool tip')
        if tool_tip:
            self.setToolTip(tool_tip)

        self.layout = QtWidgets.QHBoxLayout()
        self.layout.setSpacing(5)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.thumbnail = ThumbnailFrame(label, tools_data)

        self.layout.addWidget(self.thumbnail)
        self.setLayout(self.layout)
        self.thumbnail.clicked.connect(self.clicked)

    def mousePressEvent(self, event):
        self.tool_launched.emit(self.tool_data['name'])
    def clicked(self):
        print('clicked thumbnail')
        self.tool_launched.emit(self.name)

    def getVersion(self):
        return self.thumbnail.version

class ToolsTable(QtWidgets.QFrame):

    def __init__(self, tools_config, project_tools, parent=None):
        super(ToolsTable, self).__init__(parent=parent)
        self.project = os.environ['PROJECT']
        self.tools_config = tools_config
        self.project_tools = project_tools
        self.task_card = parent.parent()
        self.task_view = self.task_card.task_view

        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.column_count = 3
        self.main_layout = QtWidgets.QVBoxLayout()

        self.setLayout(self.main_layout)
        self.current_department = 'td'
        self.tools_widgets = {}
        self.row_index = 0
        self.column_index = 0


    def clean_tools(self):
        for index_y in range(self.row_index + 1):
            for index_x in range(self.column_count + 1):
                to_delete = self.main_layout.itemAtPosition(index_y, index_x)
                if to_delete:
                    to_delete.widget().deleteLater()

        self.tools_widgets = {}
        self.row_index = 0
        self.column_index = 0

    def load_config(self, pipeline_step):
        tool_config_label = ['args', 'label']
        self.config_data = self.tools_config.get(pipeline_step, {})
        for main_tool_data in self.config_data:
            for tool_key, tool_data in main_tool_data.items():
                this_tool_data = tool_data.copy()
                tool_name = this_tool_data.get('name', tool_key.lower())
                if tool_name in self.project_tools:
                    for key, value in self.project_tools[tool_name].items():
                        if key in tool_config_label:
                            continue
                        this_tool_data[key] = value

                    self.add_tool(tool_name, this_tool_data)

        self.main_layout.addStretch(1)

    def add_tool(self, tool_name, tools_data):

        tool_widget = ToolCard(tool_name.title(), tools_data, parent=self)
        tool_widget.setToolTip(tools_data.get('tool tip'))
        tool_widget.tool_launched.connect(self.launch_tool)
        self.tools_widgets[tool_name.lower()] = tool_widget
        self.main_layout.addWidget(tool_widget)

    def get_tool_data(self, tool_name):
        tool_widget = self.tools_widgets.get(tool_name.lower())
        if tool_widget:
            tool_data = tool_widget.tool_data
        else:
            tool_data = self.project_tools[tool_name]

        tool_data['name'] = tool_name

        return tool_data


    def launch_tool(self, tools_name):
        print('launch tool', tools_name)
        tool_data = self.get_tool_data(tools_name)
        module_name = tool_data['module']
        function_name = tool_data['function']
        arguments = tool_data.get('args', {})

        solved_arguments = {}
        for arg_name, arg_value in arguments.items():
            if arg_name == '<project>':
                value = self.project
            elif arg_name == 'parent':
                value = self.window()
            elif arg_value[0] == '<' and arg_value[-1] == '>':
                var_name = arg_value[1:-1]
                if hasattr(self, var_name):
                    value = getattr(self, var_name)
                else:
                    value = arg_value
            else:
                value = arg_value
            solved_arguments[arg_name] = value


        module = importlib.import_module(module_name)
        importlib.reload(module)
        function = getattr(module, function_name)
        function(**solved_arguments)

class ThumbnailFrame(QtWidgets.QFrame):
    clicked = QtCore.Signal()

    def __init__(self, name, tool_data, parent=None):
        super(ThumbnailFrame, self).__init__(parent=parent)

        self.tool_data = tool_data

        self.icon_path = '%s/%s' % (tool_data['path'], tool_data['icon'])
        self.tool_name = name
        self.tool_tip = tool_data.get('tool tip', '')
        self.version = str(tool_data.get('version', 'NaN'))
        self.default_version = self.version
        self.card = parent
        self.icon_size = 40

        self.layout = QtWidgets.QHBoxLayout()
        self.layout.setSpacing(5)
        self.layout.setContentsMargins(10, 4, 4, 4)
        self.icon_Label = QtWidgets.QLabel()
        self.icon_Label.setObjectName('Icon')
        self.icon_Label.setAlignment(QtCore.Qt.AlignCenter | QtCore.Qt.AlignVCenter)

        pixmap = QtGui.QPixmap(self.icon_path)
        pixmap = pixmap.scaled(QtCore.QSize(self.icon_size - 6, self.icon_size - 6))
                               #aspectRatioMode=QtCore.Qt.KeepAspectRatio)

        self.icon_Label.setPixmap(pixmap)
        self.icon_Label.setFixedSize(self.icon_size, self.icon_size)

        self.data_layout = QtWidgets.QHBoxLayout()

        self.version_widget = QtWidgets.QLabel('v%s' % self.version)
        self.version_widget.setObjectName('Version')
        self.version_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignTop)

        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(12)

        self.label_widget = QtWidgets.QLabel(self.tool_name)
        self.label_widget.setFont(font)
        self.label_widget.setObjectName('Label')

        self.layout.addWidget(self.icon_Label)

        self.data_layout.addWidget(self.label_widget)
        self.data_layout.addWidget(self.version_widget)

        self.layout.addLayout(self.data_layout)

        self.setLayout(self.layout)
        self.installEventFilter(self)

    def set_version(self, new_version):
        if new_version:
            self.version = str(new_version)

        else:
            self.version = self.default_version

        self.tool_data['version'] = self.version
        if self.version == 'local':
            self.version_widget.setText(self.version)
            self.version_widget.setStyleSheet('QLabel{color: rgb(255, 166, 0);}')

        else:
            if self.version == self.default_version:
                self.version_widget.setStyleSheet('QLabel{color: rgb(210, 210, 250);}')
            else:
                self.version_widget.setStyleSheet('QLabel{color: rgb(21, 210, 25);}')
            self.version_widget.setText('v%s' % self.version)


    def launch_task(self):
        print('clicked')