import os

import requests
from PySide import QtCore, QtGui, QtWidgets

import shotgrid_lib.database as database

this_file = os.path.abspath(__file__)
this_folder = os.path.dirname(this_file)
icon_folder = os.path.join(this_folder, 'icons')
class SubClassThread(QtCore.QThread):

    def __init__(self, widget, parent=None):
        super().__init__(parent)
        self.widget = widget

    def run(self):
        self.widget.background()


class TitleWidget(QtWidgets.QFrame):
    arrange = 'horizontal'
    modified = QtCore.Signal()
    def __init__(self, value, label, parent=None):
        super().__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        font = QtGui.QFont()
        font.setBold(True)

        self.label = label
        self.value = value

        self.layout = QtWidgets.QVBoxLayout()
        self.layout.setSpacing(2)
        self.layout.setContentsMargins(0, 0, 0, 0)
        if self.label:
            title = '%s - %s' % (self.label.title(), self.value.title())
        else:
            title = self.value.title()

        self.label_widget = QtWidgets.QLabel(title)
        self.label_widget.setObjectName('Title')
        self.label_widget.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignVCenter)
        self.label_widget.setFont(font)
        self.setLayout(self.layout)

        self.layout.addWidget(self.label_widget)

    def getValue(self):
        return self.value

class ThumbnailWidget(QtWidgets.QFrame):
    clicked = QtCore.Signal()

    def __init__(self, value, label, parent=None, cache_key=None):
        super().__init__(parent=parent)
        self.card = parent
        self.value = value
        self.cache_key = cache_key

        self.visible = False

        self.setFixedHeight(92)
        self.setFixedWidth(142)
        self.layout = QtWidgets.QVBoxLayout()
        self.layout.setSpacing(0)
        self.layout.setContentsMargins(1, 1, 1, 1)

        self.button = QtWidgets.QPushButton(self)
        self.button.setFixedSize(140, 90)

        self.layout.addWidget(self.button)
        self.setLayout(self.layout)
        self.button.clicked.connect(self.launch_task)

        cache = self.parent().container.images_cache

        image_id = self.value.id
        if image_id not in cache:
            self.update_widgets = SubClassThread(self)
            self.update_widgets.start()
        else:
            self.update()

    def background(self):
        self.update()

    def update(self):
        cache = self.parent().container.images_cache
        image_id = self.value.id
        if image_id in cache:
            pixmap = cache[image_id]
        else:
            thumbnail_path = self.value.image

            if thumbnail_path is None or thumbnail_path == 'None' or isinstance(thumbnail_path, database.emptyView):
                return
            image = QtGui.QImage()
            image.loadFromData(requests.get(thumbnail_path).content)
            pixmap = QtGui.QPixmap.fromImage(image)
            pixmap = pixmap.scaled(self.button.size()) #, aspectRatioMode=QtCore.Qt.KeepAspectRatio)
            cache[image_id] = pixmap

        self.button.setIcon(pixmap)
        self.button.setIconSize(pixmap.size())

    def launch_task(self):
        self.clicked.emit()


class CascadeComboWidget(QtWidgets.QFrame):
    modified = QtCore.Signal()

    def __init__(self,
                 value,
                 label,
                 parent=None,
                 arrange='vertical',
                 combo_labels=[],
                 values=None,
                 single_output=False,
                 label_width=150):

        super().__init__(parent=parent)

        self.arrange = arrange
        self.values = values
        self.value = value
        self.label = label
        self.combo_labels = combo_labels
        self.single_output = single_output
        self.default = '-' * 10
        self.label_width = label_width
        if self.arrange == 'vertical':
            self.main_layout = QtWidgets.QVBoxLayout()

        else:
            self.main_layout = QtWidgets.QHBoxLayout()

        self.main_layout.setSpacing(2)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        self.widget_list = []

        self.label_widget = QtWidgets.QLabel('%s :' % self.label)
        self.label_widget.setFixedWidth(100)
        self.label_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)
        self.main_layout.addWidget(self.label_widget)

        for label in self.combo_labels:
            if self.arrange == 'vertical':
                item_label = label
            else:
                item_label = None
            item_widget = LabeledCombo(self.default, item_label, values=[self.default], label_width=self.label_width)
            self.main_layout.addWidget(item_widget)
            self.widget_list.append(item_widget)
            item_widget.currentIndexChanged.connect(self.update_items)
            item_widget.currentIndexChanged.connect(self.emit_changes)

        self.setLayout(self.main_layout)

        if self.values:
            self.setValues(self.values)

        self.setValue(self.value)

    def set_width(self, width):
        for widget in self.widget_list:
            widget.set_width(label=width)
    def setValues(self, values):
        if not values:
            return
        
        self.values = values
        root_values = list(self.values.keys())
        if len(root_values) > 1:
            root_values.insert(0, self.default)
        self.widget_list[0].setValues(root_values)

    def emit_changes(self):
        self.modified.emit()

    def update_items(self, index):
        sender = self.sender()
        current_index = self.widget_list.index(sender)
        if current_index >= len(self.widget_list) - 1:
            return

        current_dict = self.values
        for index in range(current_index + 1):
            current_widget = self.widget_list[index]
            value = current_widget.getValue()
            if not value or value == self.default:
                self.widget_list[current_index+1].setValues([self.default])
                self.widget_list[current_index+1].setValue(self.default)
                return

            current_dict = current_dict[value]

        if isinstance(current_dict, dict):
            elements = list(current_dict.keys())
        else:
            elements = current_dict

        if len(elements) > 1:
            elements.insert(0, self.default)
        self.widget_list[current_index+1].setValues(elements)

    def setValue(self, context_name):

        if isinstance(context_name, str) :
            context_bits = context_name.split('.')
        elif isinstance(context_name, list):
            context_bits = context_name

        for index, bit in enumerate(context_bits):
            widget = self.widget_list[index]
            widget.setValue(bit)

    def setEnabled(self, enabled):
        for widget in self.widget_list:
            widget.setEnabled(enabled)

    def getValue(self):
        context_items = {}
        for index, widget in enumerate(self.widget_list):
            value = widget.getValue()
            if value == self.default:
                continue
            label = self.combo_labels[index].lower()
            label = label.replace(' ', '_')
            context_items[label] = value

        if self.single_output:
            if not context_items:
                return ''

            return '_'.join(context_items.values())
        else:
            return context_items


class LabeledWidget(QtWidgets.QFrame):
    arrange = 'vertical'
    font_style_sheet = "color: rgb(210,210,210);"
    modified = QtCore.Signal()

    def __init__(self, value, label, parent=None, values=None, label_width=150):
        super().__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)

        self.label = label
        self.value = value
        if values is None:
            self.values = []
        else:
            self.values = values

        if self.arrange == 'vertical':
            self.layout = QtWidgets.QVBoxLayout()
        else:
            self.layout = QtWidgets.QHBoxLayout()

        self.layout.setSpacing(8)
        self.layout.setContentsMargins(0, 0, 0, 0)
        if self.label:
            self.label_widget = QtWidgets.QLabel('%s :' % self.label)
            self.label_widget.setObjectName('Label')
            if label_width:
                self.label_widget.setFixedWidth(label_width)

            if self.arrange == 'vertical':
                self.label_widget.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)
            else:
                self.label_widget.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)

            self.layout.addWidget(self.label_widget)
        self.setLayout(self.layout)


    def set_width(self, label=150, value=150, stretch=True):
        self.label_widget.setFixedWidth(label)
        if hasattr(self, 'value_widget'):
            #self.value_widget.setFixedWidth(value)
            size = self.size()
            self.value_widget.setMinimumWidth(value)

        #if stretch:
        #    self.layout.addStretch(1)

    def setEnabled(self, enabled):
        self.value_widget.setEnabled(enabled)

    def stretch(self):
        self.layout.addStretch(-1)


class LabeledMultiLineText(LabeledWidget):
    arrange = 'vertical'
    textChanged = QtCore.Signal(str)

    def __init__(self, value, label, parent=None):
        super().__init__(value, label, parent=parent)
        self.add_value_widget()

    def add_value_widget(self):
        font = QtGui.QFont()
        font.setBold(True)
        if self.value is None :
            self.value = ''

        self.value_widget = QtWidgets.QPlainTextEdit()
        self.value_widget.setPlainText(str(self.value))
        self.value_widget.setObjectName('Value')

        #self.value_widget.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)
        self.value_widget.setFont(font)

        self.value_widget.textChanged.connect(self.emit_changes)

        self.layout.addWidget(self.value_widget)

    def emit_changes(self):
        value = self.getValue()
        self.textChanged.emit(value)
        self.modified.emit()

    def setValue(self, value):
        self.value_widget.setPlainText(value)
        self.value = value

    def getValue(self):
        value = str(self.value_widget.toPlainText())
        self.value = value

        return self.value
    
    def append(self, msg):
        self.value_widget.moveCursor(QtGui.QTextCursor.End)
        self.value_widget.insertPlainText(msg)

class LabeledInt(LabeledWidget):
    arrange = 'horizontal'
    textChanged = QtCore.Signal(str)

    def __init__(self, value, label, values=100, parent=None):
        super().__init__(value, label, parent=parent)
        if values:
            self.value = values
        self.add_value_widget()

    def add_value_widget(self):
        font = QtGui.QFont()
        font.setBold(True)
        if self.value is None :
            self.value = 101

        self.value_widget = QtWidgets.QSpinBox()
        self.value_widget.setMaximum(10000)
        self.value_widget.setMinimum(0)

        self.value_widget.setValue(self.value)
        
        self.layout.addWidget(self.value_widget)
        self.layout.addStretch(1)

    def emit_changes(self):
        value = self.getValue()
        self.valueChanged.emit(value)
        self.modified.emit()

    def setValue(self, value):
        if isinstance(value, str):
            value = int(value)
        self.value_widget.setValue(value)
        self.value = value

    def getValue(self):
        value = int(self.value_widget.value())
        self.value = value

        return self.value


class LabeledText(LabeledWidget):
    arrange = 'horizontal'
    textChanged = QtCore.Signal(str)

    def __init__(self, value, label, parent=None, editable=False, as_label=False):
        super().__init__(value, label, parent=parent)
        self.editable = editable
        self.as_label = as_label
        self.add_value_widget()

    def add_value_widget(self):
        if self.value is None :
            self.value = ''

        if isinstance(self.value, list):
            self.value = ', '.join(self.value)

        if self.as_label is False:

            self.value_widget = QtWidgets.QLineEdit()
            self.value_widget.setText(str(self.value))

            if not self.editable:
                self.value_widget.setEnabled(False)
        else:
            self.value_widget = QtWidgets.QLabel(self.value)

        self.value_widget.setObjectName('Value')

        if not self.editable:
            self.value_widget.setEnabled(False)

        self.value_widget.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)
        if self.editable:
            self.value_widget.textChanged.connect(self.emit_changes)

        self.layout.addWidget(self.value_widget)

    def emit_changes(self):
        value = self.getValue()
        self.textChanged.emit(value)
        self.modified.emit()

    def setValue(self, value):
        self.value_widget.setText(value)
        self.value = value

    def getValue(self):
        value = str(self.value_widget.text())
        self.value = value

        return self.value


class LabeledTextChooser(LabeledText):
    def __init__(self, value, label, parent=None, action='', action_module='', action_label=''):
        self.action = action
        self.action_module = action_module
        self.action_label = action_label

        super().__init__(value, label, parent=parent, editable=True)

    def add_value_widget(self):
        super().add_value_widget()
        self.browse_button = QtWidgets.QPushButton(self.action_label)

        self.browse_button.setFixedWidth(100)
        self.layout.addWidget(self.browse_button)
        self.layout.addStretch(1)
        self.browse_button.clicked.connect(self.launch_action)
        self.value_widget.textChanged.connect(self.emit_changes())


    def emit_changes(self):
        self.modified.emit()

    def setEnabled(self, enabled):
        super().setEnabled(enabled)
        self.browse_button.setEnabled(enabled)

    def launch_action(self):
        import importlib
        my_module = importlib.import_module(self.action_module)
        if my_module and hasattr(my_module, self.action):
            run_function = getattr(my_module, self.action)
            output = run_function()
            self.setValue(output)

class LabeledPath(LabeledText):

    def __init__(self, value, label, parent=None, filter=''):

        super().__init__(value, label, parent=parent, editable=True)
        self.filter = filter

    def add_value_widget(self):
        super().add_value_widget()
        self.browse_button = QtWidgets.QPushButton()

        current_icon = QtGui.QPixmap('%s/open-folder.png' % icon_folder)
        self.browse_button.setIcon(current_icon)

        self.browse_button.setFixedWidth(30)
        self.layout.addWidget(self.browse_button)
        self.browse_button.clicked.connect(self.get_filename)
        self.value_widget.textChanged.connect(self.emit_changes)


    def emit_changes(self):
        self.modified.emit()

    def get_filename(self):
        path = str(self.value_widget.text())
        folder = os.path.dirname(path)
        filename, _ = QtWidgets.QFileDialog.getOpenFileName(
            None,
            "QFileDialog.getOpenFileName()",
            folder,
            self.filter,
        )
        if filename:
            self.value_widget.setText(filename)
            #self.emit_changes()

        return filename

    def setEnabled(self, enabled):
        LabeledText.setEnabled(self, enabled)
        self.browse_button.setEnabled(enabled)


class LabeledNamedPath(LabeledWidget):
    arrange = 'horizontal'
    textChanged = QtCore.Signal(str)

    def __init__(self, value, label, parent=None, tag=None, filter='', tags=[]):
        super().__init__(value, label, parent=parent)
        self.filter = filter
        self.tag = tag
        self.tags = tags
        self.add_value_widget()

    def add_value_widget(self):
        if self.value is None:
            self.value = ''

        if self.tag is None:
            self.tag = ''

        self.tag_widget = QtWidgets.QComboBox()
        if self.tags:
            self.tag_widget.addItems(self.tags)

        self.tag_widget.setEnabled(True)
        self.tag_widget.setFixedWidth(100)
        self.tag_widget.setEditable(True)
        self.value_widget = QtWidgets.QLineEdit()
        self.value_widget.setText(str(self.value))

        self.value_widget.setEnabled(True)

        self.value_widget.setObjectName('Value')
        self.tag_widget.setObjectName('Value')

        self.value_widget.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)

        self.browse_button = QtWidgets.QPushButton()

        current_icon = QtGui.QPixmap('%s/open-folder.png' % icon_folder)
        self.browse_button.setIcon(current_icon)

        self.browse_button.setFixedWidth(30)
        self.browse_button.clicked.connect(self.get_filename)

        self.tag_widget.currentIndexChanged.connect(self.emit_changes)
        self.value_widget.textChanged.connect(self.emit_changes)

        self.layout.addWidget(self.tag_widget)
        self.layout.addWidget(self.value_widget)
        self.layout.addWidget(self.browse_button)

    def emit_changes(self):
        self.modified.emit()

    def get_filename(self):
        path = str(self.value_widget.text())
        folder = os.path.dirname(path)
        filename, _ = QtWidgets.QFileDialog.getOpenFileName(
            None,
            "QFileDialog.getOpenFileName()",
            folder,
            self.filter,
        )
        if filename:
            self.value_widget.setText(filename)
            #self.emit_changes()

        return filename


    def emit_changes(self):
        value = self.getValue()
        self.textChanged.emit(value)
        self.modified.emit()

    def setValue(self, value):
        for key, value in value:
            self.value_widget.setText(value)
            self.tag_widget.setCurrentText(key)

        self.value = value

    def getValue(self):
        value = str(self.value_widget.text())
        tag = str(self.tag_widget.currentText())
        self.value = {tag: value}

        return self.value

class LabeledMultiPaths(QtWidgets.QFrame):
    arrange = 'vertical'
    font_style_sheet = "color: rgb(210,210,210);"
    modified = QtCore.Signal()

    def __init__(self, value, label, parent=None, values=None, label_width=150, tags=[]):
        super().__init__(parent=parent)
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.label = label
        self.value = value
        self.values = values
        self.tags = values

        self.path_widgets = []
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(2)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        print('LABEL', self.label)
        self.setMinimumHeight(20)

        self.path_widgets_layout = QtWidgets.QVBoxLayout()
        self.add_path_layout(self.main_layout)

        self.main_layout.addLayout(self.path_widgets_layout)
        self.setLayout(self.main_layout)

    def add_path_layout(self, layout):
        self.add_actions_layout = QtWidgets.QHBoxLayout()

        self.label_widget = QtWidgets.QLabel(self.label)
        self.label_widget.setObjectName('Label')

        self.add_button = QtWidgets.QPushButton('+')
        self.add_button.setFixedWidth(32)
        self.add_actions_layout.addWidget(self.label_widget)
        self.add_actions_layout.addStretch()
        self.add_actions_layout.addWidget(self.add_button)
        self.add_button.clicked.connect(self.add_new_path_widget)
        layout.addLayout(self.add_actions_layout)


    def add_new_path_widget(self):
        new_path = LabeledNamedPath('', '', tag='', tags=self.tags)
        self.path_widgets.append(new_path)
        self.path_widgets_layout.addWidget(new_path)


    def getValue(self):
        all_value = {}
        for path_w in self.path_widgets:
            value = path_w.getValue()
            all_value.update(value)


        self.value = all_value
        print(self.value)
        return self.value
class LabeledBoolean(LabeledWidget):
    arrange = 'horizontal'
    clicked = QtCore.Signal(bool)

    def __init__(self, value, label, parent=None):
        super().__init__(value, label, parent=parent)
        self.add_value_widget()

    def add_value_widget(self):
        font = QtGui.QFont()
        font.setBold(True)
        if self.value is None :
            self.value = False

        self.value_widget = QtWidgets.QCheckBox()

        self.value_widget.clicked.connect(self.emit_click)
        self.layout.addWidget(self.value_widget)

        self.setValue(self.value)
    def setValue(self, value):
        self.value_widget.setChecked(value)
        self.value = value

    def getValue(self):
        value = self.value_widget.isChecked()
        return value

    def emit_click(self, value):
        self.clicked.emit(value)
        self.modified.emit()


class LabeledCombo(LabeledWidget):
    currentIndexChanged = QtCore.Signal(int)

    arrange = 'horizontal'

    def __init__(self, value, label, parent=None, values=None, sort=True, label_width=150):
        super().__init__(value, label, parent=parent, values=values, label_width=label_width)
        self.sort = sort
        self.add_value_widget()

        if values is None:
            values = [value]

        self.setValues(values)
        self.setValue(value)


    def add_value_widget(self):
        font = QtGui.QFont()
        font.setBold(True)

        self.value_widget = QtWidgets.QComboBox()
        self.value_widget.setObjectName('label_combo')
        self.value_widget.setMinimumWidth(100)
        self.value_widget.setFont(font)
        self.layout.addWidget(self.value_widget)
        self.layout.addStretch(1)

        self.value_widget.currentIndexChanged.connect(self.index_changed)

    def index_changed(self, index):
        self.currentIndexChanged.emit(index)
        self.modified.emit()

    def setValue(self, value):
        index = self.value_widget.findText(value, QtCore.Qt.MatchFixedString)
        if index >= 0:
            self.value_widget.setCurrentIndex(index)
        self.value = value

    def setValues(self, values):
        if self.sort:
            values = list(sorted(values))

        self.values = values
        self.value_widget.clear()
        self.value_widget.addItems(values)

    def getValue(self):
        value = (self.value_widget.currentText())
        return value