import importlib
import sys

from PySide import QtWidgets, QtCore

import library.core.config_manager as config_manager
import library.core.dynamic_run as dynamic_run
import packages_io

DEFAULT_WIDGET_MODULE= 'library.ui.labeled'


class InterfaceWidget(QtWidgets.QFrame):
    def __init__(self, definition, parent=None, label_width=150):
        super(InterfaceWidget, self).__init__(parent=parent)
        self.definition = definition
        self.label_width = label_width
        self.widget_list = {}
        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.main_layout = QtWidgets.QVBoxLayout()
        self.setLayout(self.main_layout)
        self.main_layout.setContentsMargins(0, 0, 0, 0)

        self.build_interface()

    def build_interface(self):
        widget_list = {}
        if self.definition is None:
            return
        for data in self.definition:
            for name, widget_data in data.items():
                print('=== %s ===' % name)
                module_name = widget_data.get('module')
                if not module_name:
                    module_name = DEFAULT_WIDGET_MODULE

                my_module = importlib.import_module(module_name)
                importlib.reload(my_module)
                widget_class_name = widget_data['type']
                widget_class = getattr(my_module, widget_class_name)
                args = widget_data.get('args', {})
                value = widget_data.get('default', '')
                values = widget_data.get('values', [])
                if widget_data.get('editable', False):
                    args['editable'] = True

                if values:
                    args['values'] = values
                if widget_data.get('scripts'):
                    values = dynamic_run.run_scripts(widget_data['scripts'].get('fill_script'))
                    if values:
                        args['values'] = values

                widget = widget_class(value, widget_data['label'], **args)

                if hasattr(widget, 'set_width'):
                    widget.set_width(label=self.label_width)
                widget.modified.connect(self.launch_update_script)
                widget.var_name = name

                self.main_layout.addWidget(widget)
                widget_list[name] = widget

        self.main_layout.addStretch(1)
        self.widget_list = widget_list

        for data in self.definition:
            for name, widget_data in data.items():
                print('widget', name)
                if widget_data.get('scripts'):
                    selected_script_data = widget_data['scripts'].get('select_script')
                    print(selected_script_data)
                    if selected_script_data:
                        check_value = dynamic_run.run_scripts(selected_script_data)
                        print(check_value)
                        if check_value is not None:
                            widget_list[name].setValue(check_value)
                            if selected_script_data.get('lock_on_selected', False):
                                widget_list[name].setEnabled(False)


    def launch_update_script(self):
        data = None
        for item in self.definition:
            name = list(item.keys())[0]
            if name == self.sender().var_name:
                data = list(item.values())[0]

        if not data:
            return

        scripts_data = data.get('scripts')
        if not scripts_data:
            return

        update_script_data = data['scripts'].get('update_script')
        if not update_script_data:
            return

        update_script_data['args'] = update_script_data.get('args', {})
        update_script_data['args']['value']= self.sender().getValue()
        output = dynamic_run.run_scripts(update_script_data)
        if update_script_data.get('destination'):
            if isinstance(output, list):
                self.widget_list[update_script_data.get('destination')].setValues(output)
            else:
                self.widget_list[update_script_data.get('destination')].setValue(output)

    def getValue(self):
        values = {}
        for name, widget in self.widget_list.items():
            values[name] = widget.getValue()

        return values



class InterfaceBuilderDemoWindow(packages_io.MainWindow):
    """
    Main publisher for all the pipeline steps

    publish_type: The type of the publish to be created
    project: The code of the Shotgrid project to work on it

    """
    def __init__( self, publish_type='Model', project=None):

        self.publish_type = publish_type

        self.title = 'Publisher: %s' % publish_type
        super( InterfaceBuilderDemoWindow, self ).__init__(self.title)

        self.setContentsMargins(6, 6, 6, 6)
        self.project = project

        self.config_manager = config_manager.ConfigSolver(project=self.project)
        self.definition = self.config_manager.get_config('demo_interface_building',
                                                         module='library',
                                                         merge=False)

        self.main_widget = QtWidgets.QWidget()
        self.main_layout =QtWidgets.QVBoxLayout()
        self.main_widget.setLayout(self.main_layout)
        self.interface_widget = InterfaceWidget(self.definition['interface'])
        self.apply_button = QtWidgets.QPushButton('Run')

        self.main_layout.addWidget(self.interface_widget)
        self.main_layout.addWidget(self.apply_button)

        self.apply_style(self)
        self.setMinimumSize(500, 800)
        self.setCentralWidget(self.main_widget)

        self.apply_button.clicked.connect(self.get_data)

    def get_data(self):
        data = self.interface_widget.getValue()
        print(data)

def set_text(value='NONE'):
    text = 'the chosen value in the combo box is %s' % value
    return text


def get_list():
    return ['a', 'b', 'c', 'd']

def set_list():
    return 'c'


if __name__ == '__main__':
    QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_ShareOpenGLContexts)

    app = QtWidgets.QApplication(sys.argv)
    window_demo = InterfaceBuilderDemoWindow(project='TPT')
    window_demo.show()
    app.exec_()