import getpass
import logging
import os

from PySide import QtWidgets, QtCore

import library.core.config_manager as config_manager
import library.core.general_exceptions as general_exceptions
import library.ui.labeled as labeled
import shotgrid_lib.lib.progress_query_widget as progress_query_widget


logger = logging.getLogger(__name__)


class ContextWidget(QtWidgets.QFrame):
    precached_entities = []
    main_module = 'builder'

    context_changed = QtCore.Signal()
    task_changed = QtCore.Signal()

    def __init__(self, definition_type, parent=None, project=None):
        super(ContextWidget, self).__init__(parent=parent)
        self.definition_type = definition_type
        self.project = project

        self.config_manager = config_manager.ConfigSolver(project=self.project)
        self.project_config = self.config_manager.project_config
        if not self.project_config:
            raise general_exceptions.ProjectConfigNotFound(self.project)
        self.builder_config = self.config_manager.get_config(self.main_module, module=self.main_module)
        definition_path = 'definitions/%s' % definition_type
        self.definition = self.config_manager.get_config(definition_path,
                                                         module=self.main_module,
                                                         merge=False)

        if not self.definition:
            logger.error('Can\'t find definition of type: %s' % definition_type)
            raise general_exceptions.ConfigNotFound(definition_path,
                                                    self.main_module)
        self.definition_context = self.definition['definition_context']
        self.definition_app = self.definition.get('definition_app', 'maya')
        self.pipeline_step = definition_type

        self.user_login = getpass.getuser()
        self.context_name = self.get_context()

        self.progress_query = progress_query_widget.ProgressQuery(self.project, parent=self)

        base_precache_entities = [self.definition_context.title(), 'Task', 'CustomEntity11', 'Step']
        base_precache_entities += self.precached_entities
        self.progress_query.set_entity_prefilter('Task', [['step', 'name_is', self.pipeline_step]])
        self.progress_query.set_precache_entities(['Step', self.definition_context.title(), 'Task'])
        self.database = self.progress_query.database
        self.progress_query.finished.connect(self.init_context_widget)

        self.setSizePolicy(QtWidgets.QSizePolicy.MinimumExpanding, QtWidgets.QSizePolicy.MinimumExpanding)
        self.app_layout = QtWidgets.QVBoxLayout()
        self.main_layout = QtWidgets.QVBoxLayout()
        self.main_layout.setSpacing(5)
        self.main_layout.setContentsMargins(10, 10, 10, 5)

        self.add_context_widget(self.main_layout)
        self.add_task_widget(self.main_layout)

        self.app_layout.addLayout(self.main_layout)

        self.setLayout(self.app_layout)

    def start(self):
        self.progress_query.start()

    def add_context_widget(self, layout):
        context_labels = self.builder_config['context_labels'][self.definition_context]
        self.context_widget = labeled.CascadeComboWidget(self.context_name,
                                                         self.definition_context,
                                                         combo_labels=context_labels)

        self.context_widget.set_width(100)
        layout.addWidget(self.context_widget)
        #self.context_widget.setEnabled(False)
        logging.info('set context to : %s' % self.context_name)
        self.context_widget.setValue(self.context_name)
        self.context_widget.modified.connect(self.update_task)

    def add_task_widget(self, layout):
        self.task_widget = labeled.LabeledCombo('', 'Task')
        self.task_widget.set_width(100)
        layout.addWidget(self.task_widget)

    def add_last_version_widget(self, layout):
        self.last_version_widget = labeled.LabeledBoolean(True, 'Apply last version')
        self.last_version_widget.set_width(100)
        layout.addWidget(self.last_version_widget)

    def init_context_widget(self):
        self.database.query_sg_database('Task', update=True)
        step_view = self.database['Step'].find_with_filters(code=self.pipeline_step,
                                                            entity_type=self.definition_context.title())
        self.tasks_view = self.database['Task'].find_with_filters(step=step_view)
        all_context = {}
        self.tasks_view.precache_dependencies(fields=['entity'])
        for task in self.tasks_view:
            entity_view = task.entity
            entity_name = entity_view.code
            if entity_view.type != self.definition_context.title():
                continue

            if entity_view.type == 'Asset':
                asset_type = entity_view.sg_asset_type
                all_context[asset_type] = all_context.get(asset_type, [])
                if entity_name not in all_context[asset_type]:
                    all_context[asset_type].append(entity_name)

            elif entity_view.type == 'Shot':
                bits = entity_name.split('_')
                if len(bits) != 4:
                    continue
                all_context[bits[0]] = all_context.get(bits[0], {})
                all_context[bits[0]][bits[1]] = all_context[bits[0]].get(bits[1], {})
                all_context[bits[0]][bits[1]][bits[2]] = all_context[bits[0]][bits[1]].get(bits[2], [])
                if bits[3] not in all_context[bits[0]][bits[1]][bits[2]]:
                    all_context[bits[0]][bits[1]][bits[2]].append(bits[3])

            elif entity_view.type == 'Episode':
                bits = entity_name.split('_')
                if len(bits) != 2:
                    continue
                all_context[bits[0]] = all_context.get(bits[0], [])
                if bits[1] not in all_context[bits[0]]:
                    all_context[bits[0]].append(bits[1])


        self.context_widget.setValues(all_context)
        #self.context_widget.setEnabled(True)
        self.context_widget.setValue(self.context_name)
        QtWidgets.QApplication.restoreOverrideCursor()

    def get_context(self):
        context_type = os.environ.get('PIPE_CONTEXT')
        if not context_type or context_type.lower() != self.definition_context.lower():
            return ''

        if context_type == 'Asset':
            asset_type = os.environ.get('PIPE_ASSET_TYPE')
            asset_name = os.environ.get('PIPE_ASSET_NAME')
            if asset_type and asset_name:
                return '%s.%s' % (asset_type, asset_name)

        elif context_type == 'Shot':
            season_name = os.environ.get('PIPE_SEASON')
            episode_name = os.environ.get('PIPE_EPISODE')
            sequence_name = os.environ.get('PIPE_SEQUENCE')
            shot_name = os.environ.get('PIPE_SHOT')

            if season_name and episode_name and sequence_name and shot_name:
                return '%s.%s.%s.%s' % (season_name, episode_name, sequence_name, shot_name)

        return ''

    def update_task(self):
        print('update task --------')
        value = self.context_widget.getValue()
        if not value.get(self.definition_context) and not value.get('name'):
            self.task_widget.setValues([])
            return

        logger.debug('---- update task ----')
        logger.debug(value)
        self.context_changed.emit()

        logger.debug('context type: %s' % self.definition_context)
        if self.definition_context == 'shot':
            entity_name = '{season}_{episode}_{sequence}_{shot}'.format(**value)
            context = self.database.entity_view('Shot')[entity_name]

        elif self.definition_context == 'episode':
            entity_name = '{season}_{episode}'.format(**value)
            context = self.database.entity_view('Episode')[entity_name]


        elif self.definition_context == 'asset':
            entity_name = value['name']
            context = self.database.entity_view('Asset')[entity_name]

        else:
            return
        context.precache_dependencies(fields=['tasks'])
        if not entity_name:
            self.task_widget.setValues([])
            return []

        task_list = context.tasks

        pipeline_step_view = self.database['Step'].find_with_filters(code=self.pipeline_step,
                                                                     entity_type=self.definition_context.title(),
                                                                     single_item=True
                                                                     )

        current_task = task_list.find_with_filters(entity=context, step=pipeline_step_view)
        if current_task.empty:
            return ''
        task_list = current_task.content
        if not isinstance(task_list, list):
            task_list = [task_list]

        self.task_widget.setValues(task_list)
        if os.environ.get('PIPE_TASK'):
            self.task_widget.setValue(os.environ['PIPE_TASK'])

        if self.definition_context == 'shot' and self.start_widget is not None:
            self.start_widget.setValue(context.sg_cut_in)
            self.end_widget.setValue(context.sg_cut_out)

        self.task_widget.currentIndexChanged.connect(self.emit_task_changed)
        self.emit_task_changed()
        return current_task.content

    def emit_task_changed(self):
        self.task_changed.emit()
