import os
import logging
import importlib
import getpass

import library.core.references_solver as references_solver
import launcher.lib.set_environment as set_environment

import packages_io

importlib.reload(references_solver)

logger = logging.getLogger(__name__)

def get_version_path( path, extension='mb'):
    import glob
    import difflib
    path = path.replace('\\', '/')

    pattern = path.replace('<version_number>', '*')
    pattern = pattern.replace('<extension>', '*')

    files = glob.glob(pattern)
    max_version = 0
    for scene_file in files:
        scene_file = scene_file.replace('\\', '/')
        changes = []
        this_string = ''
        for i, s in enumerate(difflib.ndiff(scene_file, pattern)):
            if s[0] == ' ' or s[0] == '+':
                if this_string:
                    changes.append(this_string)
                    this_string = ''
                continue

            else:
                this_string += s[2]

        if this_string:
            changes.append(this_string)
        for change in changes:
            if change.isdigit():
                version_number = int(change)
                if version_number > max_version:
                    max_version = version_number

    next_version = max_version + 1

    path = path.replace('<version_number>', '%03d' % next_version)
    path = path.replace('<extension>', extension)
    path = path.replace(' ', '_')

    return path

class Sandbox(references_solver.ReferencesSolver):
    def __init__(self, project=None):
        super().__init__(self)

        self.project = project

        self.user_name = getpass.getuser()
        self.project_environment = set_environment.ProjectEnvironment(project=self.project)

        self.project_config = self.project_environment.project_config
        self.sandbox_server = self.project_config['paths']['sandbox_server']
        self.publish_server = self.project_config['paths']['publish_server']
        self.images_server = self.project_config['paths']['images_server']

        self._vars_to_solve = {}

        self._vars_to_solve['asset_paths'] = self.project_config['paths']['assets']
        self.asset_paths = self.project_config['paths']['assets']

        self._vars_to_solve['shot_paths'] = self.project_config['paths']['shots']
        self.shot_paths = self.project_config['paths']['shots']
        self._vars_to_solve['publish_paths'] = self.project_config['paths']['publish']
        self.publish_paths = self.project_config['paths']['publish']


    def set_context(self,
                    asset_name='',
                    asset_type='',
                    shot_name='',
                    sequence_name='',
                    episode_name='',
                    season_name=''):

        self.season_name = season_name
        self.episode_name = episode_name
        self.sequence_name = sequence_name
        self.shot_name = shot_name
        logger.debug('Season name: %s' % self.season_name)
        logger.debug('Episode name: %s' % self.episode_name)
        logger.debug('Sequence name: %s' % self.sequence_name)
        logger.debug('Shot name: %s' % self.shot_name)

        self.asset_name = asset_name
        logger.debug('Asset name: %s' % self.asset_name)

        self.asset_type = asset_type

        if self.asset_name:
            self.context_type = 'assets'
        else:
            self.context_type = 'shots'
            self.check_shot_data()



    def get_version_path(self, path):
        import glob
        import difflib
        path = path.replace('\\', '/')
        pattern = path.replace('<version_number>', '*')
        pattern = pattern.replace('<extension>', '*')
        extension = path.split('.')[-1]
        files = glob.glob(pattern)
        max_version = 0
        for scene_file in files:
            scene_file = scene_file.replace('\\', '/')
            changes = []
            this_string = ''
            for i, s in enumerate(difflib.ndiff(scene_file, pattern)):
                if s[0] == ' ' or s[0] == '+':
                    if this_string:
                        changes.append(this_string)
                        this_string = ''
                    continue

                else:
                    this_string += s[2]

            if this_string:
                changes.append(this_string)
            for change in changes:
                if change.isdigit():
                    version_number = int(change)
                    if version_number > max_version:
                        max_version = version_number

        next_version = max_version + 1

        path = path.replace('<version_number>', '%03d' % next_version)
        path = path.replace('<extension>', extension)
        path = path.replace(' ', '_')

        return path


    def check_shot_data(self):
        bits = self.shot_name.split('_')
        if len(bits) != 4:
            return

        if not self.season_name:
            self.season_name = bits[0]

        if not self.episode_name:
            self.episode_name = '_'.join(bits[:2])

        if not self.sequence_name:
            self.sequence_name = '_'.join(bits[:3])

    def generate_path(self):
        sandbox_path = ''
        if self.context_type == 'assets':
            sandbox_path = self.asset_paths['sandbox']
        elif self.context_type == 'shots':
            sandbox_path = self.shot_paths['sandbox']
        sandbox_path = sandbox_path.replace(' ', '_')
        logger.debug('Sandbox pattern: %s' % sandbox_path)
        return sandbox_path

    def generate_filename(self):
        sandbox_filename = ''
        if self.context_type == 'assets':
            sandbox_filename = self.asset_paths['filename']
        elif self.context_type == 'shots':
            sandbox_filename = self.shot_paths['filename']
        sandbox_filename = sandbox_filename.replace(' ', '_')

        return sandbox_filename


    def build(self):
        path = self.generate_path()
        if not os.path.exists(path):
            logger.info('Creating Sandbox path: %s' % path)
            os.makedirs(path)

        for folder in self.project_config['paths']['assets']['template']:
            full_path = os.path.join(path, folder)
            if folder == 'maya' and packages_io.get_application() in ['maya', 'mayapy']:
                if not os.path.exists(full_path):
                    logger.debug('create maya project: %s' % path)
                    self.build_maya_sandbox(path)
                else:
                    self.set_maya_sandbox(path)

            else:
                if not os.path.exists(full_path):
                    logger.debug('create folder: %s' % full_path)
                    os.makedirs(full_path)

        return path

    def set_maya_sandbox(self, path):
        import maya.cmds as cmds
        ws_name = 'maya'
        full_path = os.path.join(path, ws_name)
        full_path = full_path.replace('\\', '/')
        logger.info('set workspace to : %s' % full_path)
        cmds.workspace(full_path, o=True)
        cmds.workspace(dir=full_path)

    def build_maya_sandbox(self, path):
        import maya.cmds as cmds

        ws_name = 'maya'
        cmds.workspace(dir=path)
        cmds.workspace(cr=ws_name)

        full_path = os.path.join(path, ws_name)
        cmds.workspace(dir=full_path)
        try:
            cmds.workspace(full_path, n=True)
        except:
            pass
        cmds.workspace(full_path, o=True)
        cmds.workspace(full_path, ua=True)
        cmds.workspace(full_path, s=True)
        rule_list = cmds.workspace(full_path, frl=True)
        all_paths = []
        for rule in rule_list:
            folder_path = cmds.workspace(full_path, fre=rule)
            if folder_path != '.':
                all_paths.append(folder_path)

        for folder_path in set(all_paths):
            rule_path = os.path.join(full_path, folder_path)
            if not os.path.exists(rule_path):
                os.makedirs(rule_path)

def create_sandbox(asset_name='',
                   asset_type='',
                   shot_name='',
                   project=None):

    sandbox_builder = Sandbox(asset_name=asset_name,
                              asset_type=asset_type,
                              shot_name=shot_name,
                              project=project)
    path = sandbox_builder.build()
    return {'sandbox_path': path}


if __name__ == '__main__':
    path = r'V:\SGD\assets\Main_Characters\LeoMesi2'

    sandbox_builder = Sandbox(asset_name='LeoMesi',asset_type='Main Characters', project='SGD')
    sandbox_builder.build()
    logger.debug('*' * 100)
    sandbox_shot_builder = Sandbox(shot_name='s00_ep00_sq010_sh010' , project='SGD')
    sandbox_shot_builder.build()
