
import logging

from PySide import QtWidgets, QtCore
import library.ui.labeled as labeled


class LoggingDialog(QtWidgets.QDialog):
    closing = QtCore.Signal()
    def __init__(self,  parent, title):
        super(LoggingDialog, self).__init__(parent)
        self.setWindowTitle(title)
        self.setMinimumWidth(500)
        self.setMinimumHeight(400)
        self.parent_pos = parent.mapToGlobal(parent.pos())
        self.parent_size = parent.size()

        self.move(self.parent_size.width()+ self.parent_pos.x(), self.parent_pos.y()-31)
        self.setFixedHeight(self.parent_size.height())
        self.log_lines = []
        QBtn = QtWidgets.QDialogButtonBox.Ok # | QtWidgets.QDialogButtonBox.Cancel
        self.level_widget = labeled.LabeledCombo('Info', 'Debug levels', values=['Debug', 'Info', 'Warning', 'Error', 'Critical'], sort=False)
        self.level_widget.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)

        self.layout = QtWidgets.QVBoxLayout()

        self.buttonBox = QtWidgets.QDialogButtonBox(QBtn)
        self.buttonBox.setFixedWidth(30)
        self.log_widget = QtWidgets.QTextBrowser()

        self.log_widget.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

        self.layout.addWidget(self.level_widget)
        self.layout.addWidget(self.log_widget)
        self.layout.addWidget(self.buttonBox)
        self.setLayout(self.layout)
        self.buttonBox.accepted.connect(self.accept)
        self.level_widget.currentIndexChanged.connect(self.set_level)
        self.init_handler()

    def init_handler(self):
        logger = logging.getLogger()
        logger.setLevel(logging.DEBUG)

        handles = logger.handlers
        for handle in handles:
            logger.removeHandler(handle)

        handles = logger.handlers
        for handle in handles:
            logger.removeHandler(handle)
        self.signal_handler = SignalHandler(self)

        logger.addHandler(self.signal_handler)


    def closeEvent(self, event):
        logger = logging.getLogger()
        logger.removeHandler(self.signal_handler)
        self.deleteLater()
        event.accept()

    def set_level(self, level):
        self.log_widget.setText('')
        for event_log in self.log_lines:
            self.write_log(event_log['message'], level = event_log['level'], module=event_log['module'])

    def write_log(self, message, level=None, module=''):
        cursor = self.log_widget.textCursor()

        if level == 'ERROR':
            color = '#BF8080'
        elif level == 'INFO':
            color = 'lightgreen'
        elif level == 'DEBUG':
            color = 'lightblue'
        elif level == 'WARNING':
            color = 'orange'
        level_str = self.level_widget.getValue().upper()
        min_level = getattr(logging, level_str)
        event_level = getattr(logging, level.upper())
        if event_level >= min_level:
            cursor.insertHtml('''<p><span style="color: {};"><b>{}: {} :</b></span>  {} <br>'''.format(color, level, module, message))

    def append(self, message, level=None, module='root'):
        self.log_lines.append({'level': level, 'message': message, 'module':module})
        self.write_log(message, level=level, module=module)
        QtWidgets.QApplication.processEvents()


class SignalHandler(logging.Handler):
    def __init__(self, log_widget):
        super(SignalHandler, self).__init__()
        self.setLevel(logging.DEBUG)
        self.log_widget = log_widget

    def emit(self, logRecord):
        msg = "{0}".format(logRecord.getMessage())
        module = logRecord.module
        self.log_widget.append(msg, level=logRecord.levelname, module=module)

